\name{od_REX}
\alias{od_REX}

\title{Optimal approximate size-constrained design}

\description{
  Computes an optimal approximate design under the standard (size) constraint using one of three methods.
}

\usage{
  od_REX(Fx, crit="D", h=NULL, w1=NULL, alg.AA="REX",
         eff=0.999999, it.max=Inf, t.max=60, echo=TRUE, track=TRUE)
}

\arguments{
\item{Fx}{
the \code{n} times \code{m} (where \code{m>=2}, \code{m<=n}) matrix containing all candidate regressors (as rows), i.e., \code{n} is the number of candidate design points, and \code{m} (where \code{m>=2}) is the number of parameters
}
\item{crit}{
the optimality criterion. Possible values are \code{"D"}, \code{"A"}, \code{"I"}, \code{"C"} and \code{"c"}.
}
\item{h}{
a non-zero vector of length \code{m} corresponding to the coefficients of the linear parameter combination of interest. If \code{crit} is not \code{"C"} nor \code{"c"} then \code{h} is ignored. If \code{crit} is \code{"C"} or \code{"c"} and \code{h=NULL} then \code{h} is assumed to be \code{c(0,...,0,1)}.
}
\item{w1}{
a real vector of length \code{n} with non-negative components, representing the initial design. The information matrix of \code{w1} must be nonsingular. It should have a small support (e.g., \code{m}) provided that \code{alg.AA="REX"} and it should have the full support of length \code{n} provided that \code{alg.AA="MUL"}. The argument \code{w1} can also be \code{NULL}; in that case a non-singular initial design is generated by \code{od_PIN}.
}
\item{alg.AA}{
the computational method to be applied; possible choices are \code{"REX"}, \code{"MUL"}, and \code{"VDM"}. For \code{crit="c"}, argument \code{alg.AA} is ignored and the function uses the rapid linear programming approach (see the references).
}
\item{eff}{
the efficiency for the stopping rule in the interval \code{(0,1)}. The algorithm will be stopped if a lower bound on the efficiency of the actual design is equal or greater than \code{eff}.
}
\item{it.max}{
the maximum allowed number of iterations of the method.
}
\item{t.max}{
the time limit for the computation.
}
\item{echo}{
Print the call of the function?
}
\item{track}{
Trace the computation?
}
}

\details{
The function implements three algorithms for the computation of optimal approximate designs with respect to the criteria of D-, A-, I-, and C-optimality: the standard vertex-direction method (\code{"VDM"}), the standard multiplicative method (\code{"MUL"}), and the randomized exchange method (\code{"REX"}). The first two methods are classical and the method REX is proposed in Harman et al (2019).

For the specific criterion of c-optimality, the function runs the LP-based method from Harman and Jurik (2008).

The information matrix of \code{w1} should have the reciprocal condition number of at least \code{1e-5}. Even if no initial design is provided, the model should be non-singular in the sense that there \emph{exists} an approximate design \code{w} with an information matrix that is not severely ill-conditioned. If this requirement is not satisfied, the computation may fail, or it may produce a deficient design. If \code{w1=NULL}, the initial design is computed with \code{\link{od_PIN}}.

Since the result is a normalized approximate design, it only gives recommended \emph{proportions} of trials in individual design points. To convert it to an optimal approximate design of size N (under the standard, i.e., size, constraints), just multiply \code{w.best} by N. To obtain an efficient exact design with N trials, \code{w.best} must be multiplied by N and the result should be properly rounded to the neighboring integers by, for example, \code{od_PUK}. However, it is often more efficient to directly use \code{od_KL} to obtain an efficient exact design of size N.
}

\value{
  A list with the following components:
   \item{call}{The call of the function.}
   \item{w.best}{The resulting exact design.}
   \item{supp}{The indices of the support of \code{w.best}.}
   \item{w.supp}{The weights of \code{w.best} on the support.}
   \item{M.best}{The information matrix of \code{w.best}.}
   \item{Phi.best}{The criterion value of \code{w.best}.}
   \item{eff.best}{A lower bound on the efficiency of \code{w.best} with respect to the optimal approximate design.}
   \item{n.iter}{The number of iterations performed.}
   \item{t.act}{The actual time of the computation.}
}

\author{
Radoslav Harman, Lenka Filova
}

\note{REX is a randomized algorithm, therefore the resulting designs may differ from run to run. In case that the optimal design is unique, the fluctuation of the results are minor and can be made negligible by setting \code{eff} to a value very close to 1.

If the optimal design is not unique, REX provides a selection of significantly different optimal designs by running it multiple times, which can help choosing the best optimal design based on a secondary criterion.

A unique and often "symmetric" optimal design (within the possibly infinite set of optimal designs) can be computed by \code{od_SYM}.

Note also that the optimal \emph{information matrix} is always unique for criteria of D-, A-, I- and C-optimality, even if the optimal design is not unique.

While the default choice is \code{alg.AA="REX"}, our numerical experience suggests that \code{alg.AA="MUL"} may be a better choice in problems with a relatively small \code{n} and a relatively large \code{m}.

The method VDM is included mostly for teaching purposes; it is only rarely competitive with REX or MUL. Its advantage is that it tends to be easy to generalize to more complex optimum design problems.
}

\seealso{
\code{\link{od_KL}, \link{od_RC}, \link{od_MISOCP}, \link{od_AQUA}}
}

\references{
Harman R, Jurik T (2008). Computing c-optimal experimental designs using the simplex method of linear programming. Computational Statistics and Data Analysis 53 (2008) 247-254

Harman R, Filova L, Richtarik P (2019). A randomized exchange algorithm for computing optimal approximate designs of experiments. Journal of the American Statistical Association, 1-30.
}

\examples{
\dontrun{
# Note: Many small examples of od_REX are in other help files.

# Compute an essentially perfect D-optimal design
# on 10 million design points in a few seconds
n <- 10000000; m <- 5
Fx <- matrix(rnorm(n*m), ncol = m)
w <- od_REX(Fx, t.max = 10)$w.best
Fx.small <- od_DEL(Fx, w)$Fx.keep
w <- od_REX(Fx.small, eff = 0.999999999)$w.best
od_plot(Fx.small, w, Fx.small[, 1:2], dd.pch = 16, dd.size = 0.35)
}
}
