	!Written by John Michael O. Ranola - Statistics University of Washington
	!For questions, concerns, or suggestions, please email:ranolaj@uw.edu

	!This program takes allelic data along with their location data.
	!1. It uses a penalized loglikelihood method to estimate frequencies at each location including unsampled ones by ensuring that
	!        close neighbors have more similar values than further ones.
	!2. Using this info it could then generate a heat map indicating the probability that a new sample
	!        came from that portion of the map and thus yield the most likely origin of the sample.

MODULE CONSTANTS

	IMPLICIT NONE

	INTEGER, PARAMETER :: DBLE_PREC = KIND(0.0D0)
	INTEGER, PARAMETER :: REAL_PREC = KIND(0.0E0)
	REAL(KIND=DBLE_PREC), PARAMETER :: PI = 3.14159
	REAL(KIND=DBLE_PREC), PARAMETER :: ZERO = 0.00000000
	REAL(KIND=DBLE_PREC), PARAMETER :: ONE = 1.00000000
	REAL(KIND=DBLE_PREC), PARAMETER :: TWO = 2.00000000
	REAL(KIND=DBLE_PREC), PARAMETER :: TEN = 10.0000000
	REAL(KIND=DBLE_PREC), PARAMETER :: ROOT2=SQRT(2.0)
	REAL(KIND=DBLE_PREC), PARAMETER :: EPS=1.E-10
	REAL(KIND=DBLE_PREC), PARAMETER :: DELTA=ONE/16.000!1.E-2
	REAL(KIND=DBLE_PREC), PARAMETER :: PSEUDO_COUNT=1.0000/10.0000

END MODULE CONSTANTS




!THE BEGINNING OF THE MODULE ---------------------------------------------------------------------
MODULE QUASINEWTON

	USE CONSTANTS

	IMPLICIT NONE

	CONTAINS

	SUBROUTINE RANK_REALS(VECTOR,VECTOR_LENGTH,RANKINGS)
		!CHECKED
	!
	!     This subroutine performs a heap sort on a list of REALS.  See:
	!     Nijenhuis A and Wilf HS (1978) "Combinatorial Algorithms for
	!     Computers and Calculators, 2nd ed", Chapter 15, Academic Press.
	!

	IMPLICIT NONE

	INTEGER :: I,J,K,L,N,TEMP_INT,VECTOR_LENGTH
	REAL(KIND=DBLE_PREC), DIMENSION(:), INTENT(INOUT) :: VECTOR
	REAL(KIND=DBLE_PREC) :: TEMP
	INTEGER, DIMENSION(VECTOR_LENGTH) :: RANKINGS
	
	!
	N = SIZE(VECTOR)
	RANKINGS=(/ (I, I = 1, N) /)
	IF (N<=1) RETURN
	L = 1+N/2
	K = N
	DO
		IF (L>1) THEN
			L = L-1
			TEMP = VECTOR(L)
			TEMP_INT=L
		ELSE
			TEMP = VECTOR(K)
			TEMP_INT=RANKINGS(K)
			VECTOR(K) = VECTOR(1)
			RANKINGS(K)=RANKINGS(1)
			K = K-1
			IF (K<=1) THEN
			   VECTOR(1) = TEMP
			   RANKINGS(1)=TEMP_INT
			   RETURN
			END IF
		END IF
		I = L
		J = L+L
		DO WHILE (J<=K)
			IF (J<K) THEN
			   IF (VECTOR(J)<VECTOR(J+1)) J = J+1
			END IF
			IF (TEMP<VECTOR(J)) THEN
			   VECTOR(I) = VECTOR(J)
			   RANKINGS(I) = RANKINGS(J)
			   I = J
			   J = J+J
			ELSE
			   J = K+1
			END IF
		END DO
		VECTOR(I) = TEMP
		RANKINGS(I)=TEMP_INT
!		PRINT*, RANKINGS
	END DO

	END SUBROUTINE RANK_REALS



	

	SUBROUTINE SORT_STRINGS(LIST)

	!

	!     This subroutine performs a heap sort on a list of strings.  See:

	!     Nijenhuis A and Wilf HS (1978) "Combinatorial Algorithms for

	!     Computers and Calculators, 2nd ed", Chapter 15, Academic Press.

	!



	IMPLICIT NONE



	CHARACTER(LEN=*), DIMENSION(:), INTENT(INOUT) :: LIST

	INTEGER :: I,J,K,L,N

	CHARACTER(LEN=LEN(LIST(1))) :: TEMP

	!

	N = SIZE(LIST)

	IF (N<=1) RETURN

	L = 1+N/2

	K = N

	DO

		IF (L>1) THEN

			L = L-1

			TEMP = LIST(L)

		ELSE

			TEMP = LIST(K)

			LIST(K) = LIST(1)

			K = K-1

			IF (K<=1) THEN

			   LIST(1) = TEMP

			   RETURN

			END IF

		END IF

		I = L

		J = L+L

		DO WHILE (J<=K)

			IF (J<K) THEN

			   IF (LIST(J)<LIST(J+1)) J = J+1

			END IF

			IF (TEMP<LIST(J)) THEN

			   LIST(I) = LIST(J)

			   I = J

			   J = J+J

			ELSE

			   J = K+1

			END IF

		END DO

		LIST(I) = TEMP

	END DO



	END SUBROUTINE SORT_STRINGS









	SUBROUTINE PURGE_STRINGS(LIST,UNIQUE)

	!

	!     This subroutine purges the ordered string array LIST of duplicate

	!     entries.  The number of UNIQUE entries is calculated in the process.

	!     Note that UNIQUE returns with the value 1 if the LIST is empty.

	!



	IMPLICIT NONE



	CHARACTER(LEN=*), DIMENSION(:) :: LIST

	INTEGER :: I,J,UNIQUE

	!

	J = 1

	DO I = 2,SIZE(LIST)

		IF (LIST(I)/=LIST(J)) THEN

			J = J+1

			LIST(J) = LIST(I)

		END IF

	END DO

	UNIQUE = J



	END SUBROUTINE PURGE_STRINGS









	FUNCTION BISECT_STRING_LIST(LIST,ITEM)

	!

	!     This function returns the position of the particular ITEM in the

	!     sorted string list.  The search is conducted by bisection.  The

	!     user should check that the proposed position actually contains

	!     the item.

	!



	IMPLICIT NONE



	CHARACTER(LEN=*) :: ITEM

	INTEGER :: BISECT_STRING_LIST,FIRST,LAST,MID

	CHARACTER(LEN=*), DIMENSION(:) :: LIST

	!

	FIRST = 1

	LAST = SIZE(LIST)

	DO

		IF (FIRST==LAST) THEN

			IF (ITEM==LIST(FIRST)) THEN

			   BISECT_STRING_LIST = FIRST

			ELSE

			   BISECT_STRING_LIST = 0

			END IF

			RETURN

		END IF

		MID = (FIRST+LAST)/2

		IF (ITEM<=LIST(MID)) THEN

			LAST = MID

		ELSE

			FIRST = MID+1

		END IF

	END DO





	END FUNCTION BISECT_STRING_LIST








	SUBROUTINE FINDINV(MATRIX, INVERSE, N, ERRORFLAG)
	!SUBROUTINE TO FIND THE INVERSE OF A SQUARE MATRIX
	!AUTHOR : LOUISDA16TH A.K.A ASHWITH J. REGO
	!REFERENCE : ALGORITHM HAS BEEN WELL EXPLAINED IN:
	!HTTP://MATH.UWW.EDU/~MCFARLAT/INVERSE.HTM
	!HTTP://WWW.TUTOR.MS.UNIMELB.EDU.AU/MATRIX/MATRIX_INVERSE.HTML
	IMPLICIT NONE
	!DECLARATIONS
	INTEGER, INTENT(IN) :: N
	INTEGER, INTENT(OUT) :: ERRORFLAG  !RETURN ERROR STATUS. -1 FOR ERROR, 0 FOR NORMAL
	REAL(KIND=DBLE_PREC), INTENT(IN), DIMENSION(N,N) :: MATRIX  !INPUT MATRIX
	REAL(KIND=DBLE_PREC), INTENT(OUT), DIMENSION(N,N) :: INVERSE !INVERTED MATRIX

	LOGICAL :: FLAG = .TRUE.
	INTEGER :: I, J, K
	REAL(KIND=DBLE_PREC) :: M
	REAL(KIND=DBLE_PREC), DIMENSION(N,2*N) :: AUGMATRIX !AUGMENTED MATRIX

	!AUGMENT INPUT MATRIX WITH AN IDENTITY MATRIX
	DO I = 1, N
		DO J = 1, 2*N
			IF (J <= N ) THEN
				AUGMATRIX(I,J) = MATRIX(I,J)
			ELSE IF ((I+N) == J) THEN
				AUGMATRIX(I,J) = 1
			ELSE
				AUGMATRIX(I,J) = 0
			ENDIF
		END DO
	END DO

	!REDUCE AUGMENTED MATRIX TO UPPER TRAINGULAR FORM
	DO K =1, N-1
		IF (AUGMATRIX(K,K) == 0) THEN
			FLAG = .FALSE.
			DO I = K+1, N
				IF (AUGMATRIX(I,K) /= 0) THEN
					DO J = 1,2*N
						AUGMATRIX(K,J) = AUGMATRIX(K,J)+AUGMATRIX(I,J)
					END DO
					FLAG = .TRUE.
					EXIT
				ENDIF
				IF (FLAG .EQV. .FALSE.) THEN
					!!CALL INTPR("NON-INVERTIBLE MATRIX",-1,1,0)
					!PRINT*, "MATRIX IS NON - INVERTIBLE"
					INVERSE = 0
					ERRORFLAG = -1
					RETURN
				ENDIF
			END DO
		ENDIF
		DO J = K+1, N
			M = AUGMATRIX(J,K)/AUGMATRIX(K,K)
			DO I = K, 2*N
				AUGMATRIX(J,I) = AUGMATRIX(J,I) - M*AUGMATRIX(K,I)
			END DO
		END DO
	END DO

	!TEST FOR INVERTIBILITY
	DO I = 1, N
		IF (AUGMATRIX(I,I) == 0) THEN
			!PRINT*, "MATRIX IS NON - INVERTIBLE"
			!!CALL INTPR("NON-INVERTIBLE MATRIX",-1,1,0)
			INVERSE = 0
			ERRORFLAG = -1
			RETURN
		ENDIF
	END DO

	!MAKE DIAGONAL ELEMENTS AS 1
	DO I = 1 , N
		M = AUGMATRIX(I,I)
		DO J = I , (2 * N)
			   AUGMATRIX(I,J) = (AUGMATRIX(I,J) / M)
		END DO
	END DO

	!REDUCED RIGHT SIDE HALF OF AUGMENTED MATRIX TO IDENTITY MATRIX
	DO K = N-1, 1, -1
		DO I =1, K
		M = AUGMATRIX(I,K+1)
			DO J = K, (2*N)
				AUGMATRIX(I,J) = AUGMATRIX(I,J) -AUGMATRIX(K+1,J) * M
			END DO
		END DO
	END DO

	!STORE ANSWER
	DO I =1, N
		DO J = 1, N
			INVERSE(I,J) = AUGMATRIX(I,J+N)
		END DO
	END DO
	ERRORFLAG = 0

	END SUBROUTINE FINDINV








	SUBROUTINE TRIANGULAR_MAT_TO_VEC(MATRIX1,MATRIX1_LENGTH,VEC1,VEC1_LENGTH)

	IMPLICIT NONE
	!CHECKED

	INTEGER :: MATRIX1_LENGTH,J,K,COUNTER,VEC1_LENGTH
	REAL(KIND=DBLE_PREC), DIMENSION(MATRIX1_LENGTH,MATRIX1_LENGTH) :: MATRIX1
	REAL(KIND=DBLE_PREC), DIMENSION(VEC1_LENGTH) :: VEC1

	COUNTER=0
	DO J=1,MATRIX1_LENGTH-1
		DO K=J+1,MATRIX1_LENGTH
			COUNTER=COUNTER+1
			VEC1(COUNTER)=MATRIX1(K,J)
		END DO
	END DO

	END SUBROUTINE TRIANGULAR_MAT_TO_VEC




	SUBROUTINE VEC_TO_TRIANGULAR_MAT(MATRIX1,MATRIX1_LENGTH,VEC1,VEC1_LENGTH)
	IMPLICIT NONE
	!CHECKED

	INTEGER :: VEC1_LENGTH,J,K,MATRIX1_LENGTH,COUNTER
	REAL(KIND=DBLE_PREC), DIMENSION(VEC1_LENGTH) :: VEC1
	REAL(KIND=DBLE_PREC), DIMENSION(MATRIX1_LENGTH,MATRIX1_LENGTH) :: MATRIX1

	COUNTER=0
	DO J=1,MATRIX1_LENGTH-1
		DO K=J+1,MATRIX1_LENGTH
			COUNTER=COUNTER+1
			MATRIX1(J,K)=VEC1(COUNTER)
			MATRIX1(K,J)=VEC1(COUNTER)
		END DO
	END DO
	DO J=1,MATRIX1_LENGTH
		MATRIX1(J,J)=1.
	END DO

	END SUBROUTINE VEC_TO_TRIANGULAR_MAT



	SUBROUTINE COMPOSE_MAP_I(PMATRIX,ALLELES,SAMPLE_SITES,MAP_VEC,MAP_LENGTH)
	!THIS FUNCTION APPENDS THE VALUES OF AHAT AT THE END OF PHAT
	!CHECKED
	IMPLICIT NONE

	INTEGER :: I,J,ALLELES,SAMPLE_SITES,MAP_LENGTH
	REAL(KIND=DBLE_PREC), DIMENSION(MAP_LENGTH) ::  MAP_VEC
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,SAMPLE_SITES) ::  PMATRIX

	DO I=1,SAMPLE_SITES
		DO J=1,ALLELES
			MAP_VEC(I+(J-1)*SAMPLE_SITES)=PMATRIX(J,I)
		END DO
	END DO

	END SUBROUTINE COMPOSE_MAP_I



	SUBROUTINE DECOMPOSE_MAP_I(PMATRIX,ALLELES,SAMPLE_SITES,MAP_VEC,MAP_LENGTH)
	!THIS FUNCTION APPENDS THE VALUES OF AHAT AT THE END OF PHAT
	!CHECKED
	IMPLICIT NONE

	INTEGER :: I,J,ALLELES,SAMPLE_SITES,MAP_LENGTH
	REAL(KIND=DBLE_PREC), DIMENSION(MAP_LENGTH) ::  MAP_VEC
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,SAMPLE_SITES) ::  PMATRIX

	DO I=1,SAMPLE_SITES
		DO J=1,ALLELES
			PMATRIX(J,I)=MAP_VEC(I+(J-1)*SAMPLE_SITES)
		END DO
	END DO

	END SUBROUTINE DECOMPOSE_MAP_I



	SUBROUTINE SHIFT_MAT_LEFT(MATRIX,DIM,DIM_SHIFT)
	!SHIFTS THE CONTENTS OF A MATRIX ALONG DIM_SHIFT TO ADD A NEW ENTRY AT THE END
	!SHIFT_MAT_LEFT(MAP_U,TEMP2,QSEC)
	!CHECKED
	IMPLICIT NONE

	INTEGER :: DIM,DIM_SHIFT,I
	REAL(KIND=DBLE_PREC), DIMENSION(DIM,DIM_SHIFT) :: MATRIX

	DO I=1,DIM_SHIFT-1
		MATRIX(:,I)=MATRIX(:,I+1)
	END DO

	END SUBROUTINE SHIFT_MAT_LEFT



	FUNCTION FIRST_MATMULT(MATU,MATV,DIM1,DIM2)
	!
	!     This routine returns: matmul(t(matu),matu)-matmul(t(matu),matv)
	!
	!
	IMPLICIT NONE
	INTEGER :: I,J,K,DIM1,DIM2
	REAL(KIND=DBLE_PREC), DIMENSION(DIM1,DIM2) :: MATU,MATV
	REAL(KIND=DBLE_PREC), DIMENSION(DIM2,DIM2) :: MATA,MATB,FIRST_MATMULT

	!
	MATA=0.
	MATB=0.
	FIRST_MATMULT=0.
	DO I=1,DIM2
		DO J=1,DIM2
			DO K=1,DIM1
				MATA(J,I)=MATA(J,I)+MATU(K,J)*MATU(K,I)
				MATB(J,I)=MATB(J,I)+MATU(K,J)*MATV(K,I)
			END DO
		END DO
	END DO

	FIRST_MATMULT=MATA-MATB

	END FUNCTION FIRST_MATMULT



	ELEMENTAL LOGICAL FUNCTION IEEE_IS_NAN(X)



	IMPLICIT NONE



	REAL(KIND=DBLE_PREC),INTENT(IN):: X



	!IEEE_IS_NAN = (ISNAN(X) .OR. ABS(X) > HUGE(X))
	IEEE_IS_NAN = ((X.NE.X) .OR. ABS(X) > HUGE(X))


	END FUNCTION






END MODULE QUASINEWTON
!END OF THE MODULE --------------------------------------------------------------------



!THE BEGINNING OF THE MODULE ---------------------------------------------------------------------
MODULE ALLELEFREQUENCIES

	USE CONSTANTS

	USE QUASINEWTON

	IMPLICIT NONE

	CONTAINS



	SUBROUTINE INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
	!
	!     This subroutine opens the text file and determines the number of
	!     lines in it.
	!

	IMPLICIT NONE

	CHARACTER(LEN=*) :: TEXT_FILE,LINE
	INTEGER :: I,INPUT_UNIT,IOERROR,LINES
	!
	!     Open the text file.
	!
	OPEN(UNIT=INPUT_UNIT,FILE=TEXT_FILE)
	!
	!     Continue reading lines until the end of file is encountered.
	!
	LINES = 0
	DO
		READ(INPUT_UNIT,'(1000A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
		IF (IOERROR<0) THEN
			RETURN
		ELSE
			LINES = LINES+1
		END IF
	END DO
	END SUBROUTINE INPUT_DATA

	
!
!	
!	SUBROUTINE WEED_DATA_OURS_ONLY_FULL2(NUMBER_SNPS,DATA_FILE)
!	!CHECKED BY HAND!
!
!	!THIS SUBROUTINE PASSES THROUGH THE DATA AND FILTERS ALL THE DATA EXCEPT THOSE SNPS
!	!WHICH ARE LISTED IN TEXT_FILE
!	!IT OUTPUTS A FILE WITH THE OBSERVATIONS FOR THOSE SNPS ONLY.
!
!	IMPLICIT NONE
!
!	CHARACTER(LEN=800) :: TEXT_FILE,LINE,TEXT_FILE2,TEXT_END,DATA_FILE
!	INTEGER :: OUTPUT_UNIT=2,I,J,K,INPUT_UNIT=4,LINES,TEMP_INT,ALLELES,TENTH,TEMP_COUNT
!	INTEGER :: INDIVIDUALS,IOERROR,LOCI,NUMBER_SNPS,SAMPLE_SITES
!	INTEGER :: TEMP_MEM,NUMBER_UNKNOWN
!	REAL(KIND=DBLE_PREC) :: TEMP_REAL1,TEMP_REAL2
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: MEMBERSHIP, TOP_SNPS, TRUE_MEMBERSHIP !MEMBERSHIP(INDIVIDUALS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: COORD !COORD(LONG=1;LAT=2,POPULATION)
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: MOD_OBS, UNKNOWN_OBS
!	INTEGER, DIMENSION(6) :: TEMP_INT_VEC
!	CHARACTER(LEN=50), DIMENSION(2000) :: MEMBERS_LIST
!	CHARACTER(LEN=1), ALLOCATABLE, DIMENSION(:,:) :: MAJOR_ALLELES
!	CHARACTER(LEN=1), ALLOCATABLE, DIMENSION(:,:) :: TEMP_CHAR !LOCI,ALLELES
!	CHARACTER(LEN=50) :: MEMBER_NAME, TEMP_TEXT
!
!	PRINT*,"WEED_DATA_OURS_ONLY_FULL BEGINNING... SNPS: ",NUMBER_SNPS
!	!COUNT THE NUMBER OF SAMPLE_SITES...1387 INDIVIDUALS
!	!ALSO POPULATES THE LIST MEMBERS_LIST WITH THE POPULATION NAMES
!	TEXT_FILE="POPRES/PC1PC2ED.txt"
!	!FINDS THE NUMBER OF LINES IN THE TEXT
!	CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!	!skip first line
!	LINE = " "
!	READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
!	SAMPLE_SITES=0
!	DO J = 1,LINES-1
!		READ(INPUT_UNIT,*) TEMP_INT,MEMBER_NAME
!		SAMPLE_SITES=SAMPLE_SITES+1
!		MEMBERS_LIST(SAMPLE_SITES)=TRIM(MEMBER_NAME)
!		!
!		!     Every 100 lines sort and purge the list of ITEMS.
!		!
!		IF (MOD(J,100)==0.OR.J==LINES-1) THEN
!			CALL SORT_STRINGS(MEMBERS_LIST(1:SAMPLE_SITES))
!			CALL PURGE_STRINGS(MEMBERS_LIST(1:SAMPLE_SITES),SAMPLE_SITES)
!		END IF
!	END DO
!	PRINT*, "POPS: ", SAMPLE_SITES
!	!PRINT*, (TRIM(MEMBERS_LIST(J)),J=1,SAMPLE_SITES)
!
!
!	!READ IN POPULATION DATA (WHAT NUMBER BELONGS TO WHICH POPULATION)
!	!INDIVIDUALS IN PC1_PC2_summary.txt are in the same order as .bed file... So will update
!	!membership using this file.
!	!ALSO UPDATES THE COORDINATE DATA
!	INDIVIDUALS=LINES-1
!	!PRINT*, "INDIVIDUALS(1387)", INDIVIDUALS
!	ALLOCATE(MEMBERSHIP(INDIVIDUALS),COORD(2,SAMPLE_SITES))
!	COORD=1.
!	REWIND(INPUT_UNIT)
!	!skip first line
!	LINE = " "
!	READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
!	DO J=1,LINES-1
!		READ(INPUT_UNIT,*) TEMP_INT,MEMBER_NAME,TEMP_TEXT,TEMP_REAL1,TEMP_REAL2
!		TEMP_INT=BISECT_STRING_LIST(MEMBERS_LIST(1:SAMPLE_SITES),TRIM(MEMBER_NAME))
!		!PRINT*, "TEMP_INT: ", TEMP_INT
!		MEMBERSHIP(J)=TEMP_INT
!		!PRINT*, "MEMBER NAME, MEMBERS_LIST: ", TRIM(MEMBER_NAME), ", ", &
!		!		TRIM(MEMBERS_LIST(TEMP_INT))
!		COORD(1,TEMP_INT)=TEMP_REAL1
!		COORD(2,TEMP_INT)=TEMP_REAL2
!	END DO
!	CLOSE(INPUT_UNIT)
!
!
!	!COUNT THE NUMBER OF LOCI...SHOULD BE 197146 SNPS
!	TEXT_FILE="POPRES/Imputed.bim"
!	!FINDS THE NUMBER OF LINES/LOCI IN THE TEXT
!	CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LOCI)
!
!	!ALLELES=2 SINCE SNP DATA
!	PRINT*, "LOCI(197146): ", LOCI
!	ALLELES=2
!	CLOSE(INPUT_UNIT)
!
!
!	!READS IN THE TOP N SNPS
!!	TEXT_FILE2="TopSNPs.txt"
!	TEXT_FILE2="TOPSNPSNewInf.csv"
!!	TEXT_FILE2="InformationVsLikelihoodInc.csv"
!!	TEXT_FILE2="InformationVsLikelihoodDec.csv"
!	CALL INPUT_DATA(TEXT_FILE2,LINE,INPUT_UNIT,LINES)
!	!NUMBER_SNPS=LINES-1
!	ALLOCATE(TOP_SNPS(NUMBER_SNPS))
!	REWIND(INPUT_UNIT)
!	!SKIPS FIRST LINE
!	READ(INPUT_UNIT,*)
!	DO I=1,NUMBER_SNPS
!		READ(INPUT_UNIT,*) TOP_SNPS(I), TEMP_REAL1, TEMP_REAL1
!	END DO
!	CLOSE(INPUT_UNIT)
!	PRINT*, "TOP_SNPS",TOP_SNPS(1:2)
!
!	!UPDATES THE MAJOR ALLELES VECTOR
!	ALLOCATE(MAJOR_ALLELES(ALLELES,LOCI),TEMP_CHAR(LOCI,ALLELES))
!	TEXT_FILE="POPRES/Imputed.bim"
!	!FINDS THE NUMBER OF LINES IN THE TEXT
!	CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!	PRINT*, "LINES, LOCI(=):", LINES, LOCI
!	DO I=1,LINES
!		READ(INPUT_UNIT,*) TEMP_INT,LINE,TEMP_INT,TEMP_INT, &
!					MAJOR_ALLELES(1,I), MAJOR_ALLELES(2,I)
!	END DO
!	CLOSE(INPUT_UNIT)
!
!	!SETS THE NUMBER OF SNPS DESIRED (SHOULD BE 120 ACCORDING TO PASCHOU2007)
!	!--------------------------Number SNPs ----------------------------------------
!	!NUMBER_SNPS=3 !this is now an input
!	!1390   !1-7 would have 139 unknown !8-0 would have 138
!	!NUMBER_UNKNOWN=1387
!	TENTH=1
!	NUMBER_UNKNOWN=0
!	DO I=1,1387
!		IF(MOD(I,10).EQ.TENTH) THEN
!			NUMBER_UNKNOWN=NUMBER_UNKNOWN+1
!		END IF
!	END DO
!	NUMBER_UNKNOWN=1387
!	PRINT*,"NUMBER_UNKNOWN: ", NUMBER_UNKNOWN
!	ALLOCATE(MOD_OBS(SAMPLE_SITES,NUMBER_SNPS*ALLELES))
!	MOD_OBS=0
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS*ALLELES),TRUE_MEMBERSHIP(NUMBER_UNKNOWN))
!	TEXT_END=" "
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	TEXT_FILE="POPRES/ImputedPed.ped"
!
!	CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!	PRINT*, "LINES,INDIV(=) ", LINES,INDIVIDUALS
!
!	TEMP_COUNT=0
!	DO I=1,LINES
!		READ(INPUT_UNIT,*) (TEMP_INT_VEC(J),J=1,6), &
!					(TEMP_CHAR(J,1),TEMP_CHAR(J,2),J=1,LOCI)
!		TEMP_MEM=MEMBERSHIP(I)
!
!		!SEPARATING DATA INTO KNOWN/UNKNOWN
!		DO K=1,NUMBER_SNPS
!
!			!!MOD_OBS(SAMPLE_SITES,NUMBER_SNPS*ALLELES)
!			!SAVING OUR DATA
!			IF(TEMP_CHAR(TOP_SNPS(K),1)==MAJOR_ALLELES(1,TOP_SNPS(K))) THEN
!				MOD_OBS(TEMP_MEM,K*2-1)=MOD_OBS(TEMP_MEM,K*2-1)+1
!			ELSE IF(TEMP_CHAR(TOP_SNPS(K),1)==MAJOR_ALLELES(2,TOP_SNPS(K))) THEN
!				MOD_OBS(TEMP_MEM,K*2)=MOD_OBS(TEMP_MEM,K*2)+1
!			END IF
!			IF(TEMP_CHAR(TOP_SNPS(K),2)==MAJOR_ALLELES(1,TOP_SNPS(K))) THEN
!				MOD_OBS(TEMP_MEM,K*2-1)=MOD_OBS(TEMP_MEM,K*2-1)+1
!			ELSE IF(TEMP_CHAR(TOP_SNPS(K),2)==MAJOR_ALLELES(2,TOP_SNPS(K))) THEN
!				MOD_OBS(TEMP_MEM,K*2)=MOD_OBS(TEMP_MEM,K*2)+1
!			END IF
!		END DO
!		
!		!SAVING UNKNOWN DATA FOR OUR ANALYSIS
!!		IF(MOD(I,10).EQ.TENTH) THEN
!			TEMP_COUNT=TEMP_COUNT+1
!			TRUE_MEMBERSHIP(TEMP_COUNT)=TEMP_MEM
!			DO K=1,NUMBER_SNPS
!				TEMP_INT=0
!				IF(TEMP_CHAR(TOP_SNPS(K),1)==MAJOR_ALLELES(1,TOP_SNPS(K))) THEN
!					TEMP_INT=TEMP_INT+1
!				END IF
!				IF(TEMP_CHAR(TOP_SNPS(K),2)==MAJOR_ALLELES(1,TOP_SNPS(K))) THEN
!					TEMP_INT=TEMP_INT+1
!				END IF
!				IF((TEMP_INT<1).AND.(TEMP_CHAR(TOP_SNPS(K),1)/=MAJOR_ALLELES(2,TOP_SNPS(K)))) THEN
!					TEMP_INT=-1
!					UNKNOWN_OBS(TEMP_COUNT,2*K-1)=TEMP_INT
!					UNKNOWN_OBS(TEMP_COUNT,2*K)=TEMP_INT
!				ELSE
!					UNKNOWN_OBS(TEMP_COUNT,2*K-1)=TEMP_INT
!					UNKNOWN_OBS(TEMP_COUNT,2*K)=2-TEMP_INT
!				END IF
!			END DO
!!		END IF
!		IF(MOD(I,100).EQ.1) THEN
!			PRINT*, "INDIVIDUAL: ",I
!		END IF
!	END DO
!	
!	CLOSE(INPUT_UNIT)
!	PRINT*, "FINISHED READING DATA... BEGIN WRITE DATA"
!
!	!WRITES OUR INPUT DATA
!	!DATA SHOULD BE IN THE FOLLOWING FORM
!	!ALLELES A LOCI L
!	!POP1 LONG1 LAT1 L1A1 L1A2 L2A1 L2A2
!	!POP2 LONG2 LAT2 L1A1 L1A2 L2A1 L2A2
!	TEXT_FILE="Input"//TRIM(DATA_FILE)//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,*) "ALLELES", ALLELES, "NUMBER_SNPS", NUMBER_SNPS
!	DO I=1,SAMPLE_SITES
!		WRITE(OUTPUT_UNIT,*) I, COORD(1,I), COORD(2,I), &
!						(MOD_OBS(I,J),J=1,ALLELES*NUMBER_SNPS)
!	END DO
!	CLOSE(OUTPUT_UNIT)
!
!	PRINT*, "FINISHED WRITING INPUTDATA...WRITING UNKNOWNDATA..."
!	!WRITES OUR UNKNOWN DATA
!	TEXT_FILE="Unknown"//TRIM(DATA_FILE)//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,*) "ALLELES", ALLELES, "NUMBER_UNKNOWN", NUMBER_UNKNOWN, &
!				"NUMBER_SNPS ", NUMBER_SNPS
!	DO I=1,NUMBER_UNKNOWN
!		WRITE(OUTPUT_UNIT,*) TRUE_MEMBERSHIP(I), TRUE_MEMBERSHIP(I), &
!					(UNKNOWN_OBS(I,J),J=1,ALLELES*NUMBER_SNPS)
!	END DO
!	CLOSE(OUTPUT_UNIT)
!
!	PRINT*, "FINISHED WEED_DATA_OURS_ONLY_FULL"	
!
!	END SUBROUTINE WEED_DATA_OURS_ONLY_FULL2


!
!	
!	SUBROUTINE POOL_PLINK_DATA(NUMBER_SNPS,DATA_FILE)
!	!CHECKED BY HAND!
!
!	!THIS SUBROUTINE PASSES THROUGH THE DATA AND FILTERS ALL THE DATA EXCEPT THOSE SNPS
!	!WHICH ARE LISTED IN TEXT_FILE
!	!IT OUTPUTS A FILE WITH THE OBSERVATIONS FOR THOSE SNPS ONLY.
!
!	IMPLICIT NONE
!
!	CHARACTER(LEN=800) :: TEXT_FILE,LINE,TEXT_FILE2,TEXT_END,DATA_FILE
!	INTEGER :: OUTPUT_UNIT=2,I,J,K,INPUT_UNIT=4,LINES,TEMP_INT,ALLELES,TENTH,TEMP_COUNT
!	INTEGER :: INDIVIDUALS,IOERROR,LOCI,NUMBER_SNPS,SAMPLE_SITES
!	INTEGER :: TEMP_MEM,NUMBER_UNKNOWN
!	REAL(KIND=DBLE_PREC) :: TEMP_REAL1,TEMP_REAL2
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: MEMBERSHIP, TOP_SNPS, TRUE_MEMBERSHIP !MEMBERSHIP(INDIVIDUALS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: COORD !COORD(LONG=1;LAT=2,POPULATION)
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: MOD_OBS, UNKNOWN_OBS
!	INTEGER, DIMENSION(6) :: TEMP_INT_VEC
!	CHARACTER(LEN=50), DIMENSION(2000) :: MEMBERS_LIST
!	CHARACTER(LEN=1), ALLOCATABLE, DIMENSION(:,:) :: MAJOR_ALLELES
!	CHARACTER(LEN=1), ALLOCATABLE, DIMENSION(:,:) :: TEMP_CHAR !LOCI,ALLELES
!	CHARACTER(LEN=50) :: MEMBER_NAME, TEMP_TEXT
!
!	PRINT*,"POOL_PLINK_DATA BEGINNING... SNPS: ",NUMBER_SNPS
!	!COUNT THE NUMBER OF SAMPLE_SITES...1387 INDIVIDUALS
!	!ALSO POPULATES THE LIST MEMBERS_LIST WITH THE POPULATION NAMES
!	TEXT_FILE="POPRES/PC1PC2ED.txt"
!	!FINDS THE NUMBER OF LINES IN THE TEXT
!	CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!	!skip first line
!	LINE = " "
!	READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
!	SAMPLE_SITES=0
!	DO J = 1,LINES-1
!		READ(INPUT_UNIT,*) TEMP_INT,MEMBER_NAME
!		SAMPLE_SITES=SAMPLE_SITES+1
!		MEMBERS_LIST(SAMPLE_SITES)=TRIM(MEMBER_NAME)
!		!
!		!     Every 100 lines sort and purge the list of ITEMS.
!		!
!		IF (MOD(J,100)==0.OR.J==LINES-1) THEN
!			CALL SORT_STRINGS(MEMBERS_LIST(1:SAMPLE_SITES))
!			CALL PURGE_STRINGS(MEMBERS_LIST(1:SAMPLE_SITES),SAMPLE_SITES)
!		END IF
!	END DO
!	PRINT*, "POPS: ", SAMPLE_SITES
!	!PRINT*, (TRIM(MEMBERS_LIST(J)),J=1,SAMPLE_SITES)
!
!
!	!READ IN POPULATION DATA (WHAT NUMBER BELONGS TO WHICH POPULATION)
!	!INDIVIDUALS IN PC1_PC2_summary.txt are in the same order as .bed file... So will update
!	!membership using this file.
!	!ALSO UPDATES THE COORDINATE DATA
!	INDIVIDUALS=LINES-1
!	!PRINT*, "INDIVIDUALS(1387)", INDIVIDUALS
!	ALLOCATE(MEMBERSHIP(INDIVIDUALS),COORD(2,SAMPLE_SITES))
!	COORD=1.
!	REWIND(INPUT_UNIT)
!	!skip first line
!	LINE = " "
!	READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
!	DO J=1,LINES-1
!		READ(INPUT_UNIT,*) TEMP_INT,MEMBER_NAME,TEMP_TEXT,TEMP_REAL1,TEMP_REAL2
!		TEMP_INT=BISECT_STRING_LIST(MEMBERS_LIST(1:SAMPLE_SITES),TRIM(MEMBER_NAME))
!		!PRINT*, "TEMP_INT: ", TEMP_INT
!		MEMBERSHIP(J)=TEMP_INT
!		!PRINT*, "MEMBER NAME, MEMBERS_LIST: ", TRIM(MEMBER_NAME), ", ", &
!		!		TRIM(MEMBERS_LIST(TEMP_INT))
!		COORD(1,TEMP_INT)=TEMP_REAL1
!		COORD(2,TEMP_INT)=TEMP_REAL2
!	END DO
!	CLOSE(INPUT_UNIT)
!
!
!
!
!
!	!READS IN THE TOP N SNPS
!!	TEXT_FILE2="TopSNPs.txt"
!	TEXT_FILE2="TOPSNPSNewInf.csv"
!!	TEXT_FILE2="InformationVsLikelihoodInc.csv"
!!	TEXT_FILE2="InformationVsLikelihoodDec.csv"
!	CALL INPUT_DATA(TEXT_FILE2,LINE,INPUT_UNIT,LINES)
!	!NUMBER_SNPS=LINES-1
!	ALLOCATE(TOP_SNPS(NUMBER_SNPS))
!	REWIND(INPUT_UNIT)
!	!SKIPS FIRST LINE
!	READ(INPUT_UNIT,*)
!	DO I=1,NUMBER_SNPS
!		READ(INPUT_UNIT,*) TOP_SNPS(I), TEMP_REAL1, TEMP_REAL1
!	END DO
!	CLOSE(INPUT_UNIT)
!	PRINT*, "TOP_SNPS",TOP_SNPS(1:2)
!
!
!	!COUNT THE NUMBER OF LOCI...SHOULD BE 197146 SNPS
!	!TEXT_FILE="POPRES/Imputed.bim"
!	TEXT_FILE="POPRES/ImputedPed.map"
!	!FINDS THE NUMBER OF LINES/LOCI IN THE TEXT
!	CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LOCI)
!
!	!ALLELES=2 SINCE SNP DATA
!	PRINT*, "LOCI(197146): ", LOCI
!	ALLELES=2
!
!
!	!UPDATES THE MAJOR ALLELES VECTOR
!	ALLOCATE(MAJOR_ALLELES(ALLELES,LOCI),TEMP_CHAR(LOCI,ALLELES))
!!	TEXT_FILE="POPRES/Imputed.bim"
!	!FINDS THE NUMBER OF LINES IN THE TEXT
!!	CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!!	PRINT*, "LINES, LOCI(=):", LINES, LOCI
!!	DO I=1,LINES
!!		READ(INPUT_UNIT,*) TEMP_INT,LINE,TEMP_INT,TEMP_INT, &
!!					MAJOR_ALLELES(1,I), MAJOR_ALLELES(2,I)
!!	END DO
!!	CLOSE(INPUT_UNIT)
!	READ(INPUT_UNIT,*) (TEMP_INT_VEC(J),J=1,6), &
!					(TEMP_CHAR(J,1),TEMP_CHAR(J,2),J=1,LOCI)
!					
!	MAJOR_ALLELES(1,:)=TEMP_CHAR(:,1)
!	DO J=1,LOCI
!		IF(MAJOR_ALLELES(1,I)==TEMP_CHAR(J,1)) THEN
!			PRINT*, "FIX THIS"
!		END IF
!	END DO
!	
!
!
!	!SETS THE NUMBER OF SNPS DESIRED (SHOULD BE 120 ACCORDING TO PASCHOU2007)
!	!--------------------------Number SNPs ----------------------------------------
!	!NUMBER_SNPS=3 !this is now an input
!	!1390   !1-7 would have 139 unknown !8-0 would have 138
!	!NUMBER_UNKNOWN=1387
!	TENTH=1
!	NUMBER_UNKNOWN=0
!	DO I=1,1387
!		IF(MOD(I,10).EQ.TENTH) THEN
!			NUMBER_UNKNOWN=NUMBER_UNKNOWN+1
!		END IF
!	END DO
!	NUMBER_UNKNOWN=1387
!	PRINT*,"NUMBER_UNKNOWN: ", NUMBER_UNKNOWN
!	ALLOCATE(MOD_OBS(SAMPLE_SITES,NUMBER_SNPS*ALLELES))
!	MOD_OBS=0
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS*ALLELES),TRUE_MEMBERSHIP(NUMBER_UNKNOWN))
!	TEXT_END=" "
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	TEXT_FILE="POPRES/ImputedPed.ped"
!
!	CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!	PRINT*, "LINES,INDIV(=) ", LINES,INDIVIDUALS
!
!	TEMP_COUNT=0
!	DO I=1,LINES
!		READ(INPUT_UNIT,*) (TEMP_INT_VEC(J),J=1,6), &
!					(TEMP_CHAR(J,1),TEMP_CHAR(J,2),J=1,LOCI)
!		TEMP_MEM=MEMBERSHIP(I)
!
!		!SEPARATING DATA INTO KNOWN/UNKNOWN
!		DO K=1,NUMBER_SNPS
!
!			!!MOD_OBS(SAMPLE_SITES,NUMBER_SNPS*ALLELES)
!			!SAVING OUR DATA
!			IF(TEMP_CHAR(TOP_SNPS(K),1)==MAJOR_ALLELES(1,TOP_SNPS(K))) THEN
!				MOD_OBS(TEMP_MEM,K*2-1)=MOD_OBS(TEMP_MEM,K*2-1)+1
!			ELSE IF(TEMP_CHAR(TOP_SNPS(K),1)==MAJOR_ALLELES(2,TOP_SNPS(K))) THEN
!				MOD_OBS(TEMP_MEM,K*2)=MOD_OBS(TEMP_MEM,K*2)+1
!			END IF
!			IF(TEMP_CHAR(TOP_SNPS(K),2)==MAJOR_ALLELES(1,TOP_SNPS(K))) THEN
!				MOD_OBS(TEMP_MEM,K*2-1)=MOD_OBS(TEMP_MEM,K*2-1)+1
!			ELSE IF(TEMP_CHAR(TOP_SNPS(K),2)==MAJOR_ALLELES(2,TOP_SNPS(K))) THEN
!				MOD_OBS(TEMP_MEM,K*2)=MOD_OBS(TEMP_MEM,K*2)+1
!			END IF
!		END DO
!		
!		!SAVING UNKNOWN DATA FOR OUR ANALYSIS
!!		IF(MOD(I,10).EQ.TENTH) THEN
!			TEMP_COUNT=TEMP_COUNT+1
!			TRUE_MEMBERSHIP(TEMP_COUNT)=TEMP_MEM
!			DO K=1,NUMBER_SNPS
!				TEMP_INT=0
!				IF(TEMP_CHAR(TOP_SNPS(K),1)==MAJOR_ALLELES(1,TOP_SNPS(K))) THEN
!					TEMP_INT=TEMP_INT+1
!				END IF
!				IF(TEMP_CHAR(TOP_SNPS(K),2)==MAJOR_ALLELES(1,TOP_SNPS(K))) THEN
!					TEMP_INT=TEMP_INT+1
!				END IF
!				IF((TEMP_INT<1).AND.(TEMP_CHAR(TOP_SNPS(K),1)/=MAJOR_ALLELES(2,TOP_SNPS(K)))) THEN
!					TEMP_INT=-1
!					UNKNOWN_OBS(TEMP_COUNT,2*K-1)=TEMP_INT
!					UNKNOWN_OBS(TEMP_COUNT,2*K)=TEMP_INT
!				ELSE
!					UNKNOWN_OBS(TEMP_COUNT,2*K-1)=TEMP_INT
!					UNKNOWN_OBS(TEMP_COUNT,2*K)=2-TEMP_INT
!				END IF
!			END DO
!!		END IF
!		IF(MOD(I,100).EQ.1) THEN
!			PRINT*, "INDIVIDUAL: ",I
!		END IF
!	END DO
!	
!	CLOSE(INPUT_UNIT)
!	PRINT*, "FINISHED READING DATA... BEGIN WRITE DATA"
!
!	!WRITES OUR INPUT DATA
!	!DATA SHOULD BE IN THE FOLLOWING FORM
!	!ALLELES A LOCI L
!	!POP1 LONG1 LAT1 L1A1 L1A2 L2A1 L2A2
!	!POP2 LONG2 LAT2 L1A1 L1A2 L2A1 L2A2
!	TEXT_FILE="Input"//TRIM(DATA_FILE)//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,*) "ALLELES", ALLELES, "NUMBER_SNPS", NUMBER_SNPS
!	DO I=1,SAMPLE_SITES
!		WRITE(OUTPUT_UNIT,*) I, COORD(1,I), COORD(2,I), &
!						(MOD_OBS(I,J),J=1,ALLELES*NUMBER_SNPS)
!	END DO
!	CLOSE(OUTPUT_UNIT)
!
!	PRINT*, "FINISHED WRITING INPUTDATA...WRITING UNKNOWNDATA..."
!	!WRITES OUR UNKNOWN DATA
!	TEXT_FILE="Unknown"//TRIM(DATA_FILE)//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,*) "ALLELES", ALLELES, "NUMBER_UNKNOWN", NUMBER_UNKNOWN, &
!				"NUMBER_SNPS ", NUMBER_SNPS
!	DO I=1,NUMBER_UNKNOWN
!		WRITE(OUTPUT_UNIT,*) TRUE_MEMBERSHIP(I), TRUE_MEMBERSHIP(I), &
!					(UNKNOWN_OBS(I,J),J=1,ALLELES*NUMBER_SNPS)
!	END DO
!	CLOSE(OUTPUT_UNIT)
!
!	PRINT*, "FINISHED WEED_DATA_OURS_ONLY_FULL"	
!
!	END SUBROUTINE POOL_PLINK_DATA






	SUBROUTINE CHECK_CONVERGENCE(OLD_VAL,NEW_VAL,ITERATIONS,NOTCONVERGED_BOOL)
	!CHECKS THE CONVERGENCE CRITERIA

	IMPLICIT NONE

	REAL(KIND=DBLE_PREC) :: OLD_VAL,NEW_VAL
	INTEGER :: ITERATIONS
	LOGICAL :: NOTCONVERGED_BOOL

	IF((ABS((OLD_VAL-NEW_VAL)/(ABS(OLD_VAL)+1)).LE.1E-9).OR.(ITERATIONS.GE.2000)) THEN
		NOTCONVERGED_BOOL=.FALSE.
	END IF

	END SUBROUTINE CHECK_CONVERGENCE




	
	SUBROUTINE CHECK_CONVERGENCE_MATRIX(OLD_PMATRIX,NEW_PMATRIX,ITERATIONS,NOTCONVERGED_BOOL)
	!CHECKS THE CONVERGENCE CRITERIA BASED OFF OF THE MAX VALUE OF THE
	!DIFFERENCE IN PMATRIX

	IMPLICIT NONE

	REAL(KIND=DBLE_PREC), DIMENSION(:,:) :: OLD_PMATRIX,NEW_PMATRIX
	INTEGER :: ITERATIONS
	LOGICAL :: NOTCONVERGED_BOOL
	REAL(KIND=DBLE_PREC) :: TEMP_REAL
	
	TEMP_REAL=MAXVAL(ABS(OLD_PMATRIX-NEW_PMATRIX))

	IF((TEMP_REAL.LE.1E-7).OR.(ITERATIONS.GE.20000)) THEN
		NOTCONVERGED_BOOL=.FALSE.
	END IF

	END SUBROUTINE CHECK_CONVERGENCE_MATRIX
	
	
	

	FUNCTION INFORMATIVENESS_FOR_ASSIGNMENT(OBS,ALLELES,SAMPLE_SITES)
	!THIS FUNCTION CALCULATES THE "INFORMATIVENESS FOR ASSIGNMENT" FOR A PARTICULAR LOCUS
	!AS DEPICTED IN THE PAPER
	!"INFORMATIVENESS OF GENETIC MARKERS FOR INFERENCE OF ANCESTRY" BY ROSENBERG ET AL. 2003
	!IT GIVES A MEASURE OF THE AMOUNT OF INFORMATION A MARKER HAS ON ANCESTRY AND CAN BE USED
	!TO REDUCE THE NUMBER OF MARKERS OR SNPS FOR ANALYSIS

	INTEGER :: I,J,K,ALLELES,SAMPLE_SITES
	INTEGER, DIMENSION(ALLELES,SAMPLE_SITES) :: OBS
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,SAMPLE_SITES) :: REL_FREQ
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES) :: AVG_FREQ
	REAL(KIND=DBLE_PREC) :: INFORMATIVENESS_FOR_ASSIGNMENT,TEMP_REAL,TEMP_REAL2

	REL_FREQ=0.
	AVG_FREQ=0.

	DO I=1,ALLELES
		DO J=1,SAMPLE_SITES
			TEMP_REAL2=(SUM(OBS(:,J))+0.)
			IF(TEMP_REAL2>0) THEN
				REL_FREQ(I,J)=(OBS(I,J)+0.)/TEMP_REAL2
			ELSE
				REL_FREQ(I,J)=0.
			END IF
		END DO
	END DO

	DO I=1,ALLELES
		AVG_FREQ(I)=SUM(REL_FREQ(I,:))/(SAMPLE_SITES+0.)
!		PRINT*, "AVG_FREQ(I): ", AVG_FREQ(I)
	END DO

	TEMP_REAL=0.
	DO I=1,ALLELES
		TEMP_REAL2=0.
		DO J=1,SAMPLE_SITES
			IF(REL_FREQ(I,J)>0) THEN
				TEMP_REAL2=TEMP_REAL2+REL_FREQ(I,J)*LOG(REL_FREQ(I,J))/SAMPLE_SITES
			END IF
		END DO
		IF(AVG_FREQ(I)>0.) THEN
			TEMP_REAL=TEMP_REAL-AVG_FREQ(I)*LOG(AVG_FREQ(I))+TEMP_REAL2
		ELSE
			TEMP_REAL=TEMP_REAL+TEMP_REAL2
		END IF
	END DO

!	PRINT*, "INFORMATIVENESS: ", TEMP_REAL
	INFORMATIVENESS_FOR_ASSIGNMENT=TEMP_REAL

	END FUNCTION INFORMATIVENESS_FOR_ASSIGNMENT


	
	FUNCTION GAMLOG(X)
	!
	!     THIS ROUTINE COMPUTES LOG(GAMMA(X)) VIA THE APPROXIATION OF SPOUGE (1994)
	!     COMPUTATION OF THE GAMMA, DIGAMMA, AND TRIGAMMA FUNCTIONS.
	!     SIAM J NUMERICAL ANAL 31:931-944.
	!
	IMPLICIT NONE
	INTEGER, PARAMETER :: A = 10
	INTEGER :: K,L
	LOGICAL, SAVE :: FIRST_VISIT = .TRUE.
	REAL(KIND=DBLE_PREC) :: GAMLOG,X,Y
	REAL(KIND=DBLE_PREC), SAVE, DIMENSION(0:A-1) :: C
	!
	IF (FIRST_VISIT) THEN
		C(0) = SQRT(2.0*PI)
		L = 1
		DO K = 1,A-1
			Y = -K+A
			C(K) = (-1)**(K-1)*Y**(K-0.5)*EXP(Y)/L
			L = K*L
		END DO
		FIRST_VISIT = .FALSE.
	END IF
	GAMLOG = C(0)
	DO K = 1,A-1
		GAMLOG = GAMLOG+C(K)/(X+K)
	END DO
	Y = X+A
	GAMLOG = (X+0.5)*LOG(Y)-Y+LOG(GAMLOG/X)
	
	END FUNCTION GAMLOG



	FUNCTION LOG_COMBINATIONS(N,X)
	!
	!     THIS ROUTINE COMPUTES THE LOG OF THE NUMBER OF WAYS OF CHOOSING X
	!     ELEMENTS FROM A SET WITH N ELEMENTS.
	!
	IMPLICIT NONE
	REAL(KIND=DBLE_PREC) :: LOG_COMBINATIONS,N,X
	
	LOG_COMBINATIONS = GAMLOG(N+1.0)-GAMLOG(X+1.0)-GAMLOG(N-X+1.0)
	
	END FUNCTION LOG_COMBINATIONS


	
	FUNCTION LOG_BINOMIAL(N,X,P)
	!
	!	THIS SUBROUTINE COMPUTES THE LOGLIKELIHOOD FOR A BINOMIAL DISTRIBUTION
	!
	IMPLICIT NONE
	REAL(KIND=DBLE_PREC) :: LOG_BINOMIAL,N,X,P
	INTEGER :: I,J,K
	
	IF((P>0.).AND.(P<1.)) THEN
		LOG_BINOMIAL=LOG_COMBINATIONS(N,X)+X*LOG(P)+(N-X)*LOG(1.0-P)
	ELSE
		LOG_BINOMIAL=LOG_COMBINATIONS(N,X)
	END IF
	
	END FUNCTION LOG_BINOMIAL
	
	
	
	
	
	FUNCTION LIKELIHOOD_RATIO(OBS,ALLELES,SAMPLE_SITES)
	!THIS FUNCTION CALCULATES THE MODIFIED VERSION OF THE 
	!"INFORMATIVENESS FOR ASSIGNMENT" FOR A PARTICULAR LOCUS
	!AS DEPICTED IN THE PAPER
	!"INFORMATIVENESS OF GENETIC MARKERS FOR INFERENCE OF ANCESTRY" BY ROSENBERG ET AL. 2003
	!IT GIVES A MEASURE OF THE AMOUNT OF INFORMATION A MARKER HAS ON ANCESTRY AND CAN BE USED
	!TO REDUCE THE NUMBER OF MARKERS OR SNPS FOR ANALYSIS.  NOTE THAT THIS
	!MODIFIED VERSION DOES NOT ASSUME EQUAL SAMPLE SIZES.

	INTEGER :: I,J,K,ALLELES,SAMPLE_SITES,INDIVIDUALS
	INTEGER, DIMENSION(ALLELES,SAMPLE_SITES) :: OBS
	INTEGER, DIMENSION(SAMPLE_SITES) :: POP_SIZES
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,SAMPLE_SITES) :: REL_FREQ
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES) :: AVG_FREQ
	REAL(KIND=DBLE_PREC) :: LIKELIHOOD_RATIO,TEMP_REAL1,TEMP_REAL2

	REL_FREQ=0.
	AVG_FREQ=0.
	POP_SIZES=0
	
	DO I=1,SAMPLE_SITES
		POP_SIZES(I)=SUM(OBS(:,I))/2
	END DO
	INDIVIDUALS=SUM(POP_SIZES)
	
	DO I=1,ALLELES
		DO J=1,SAMPLE_SITES
			TEMP_REAL2=(SUM(OBS(:,J))+0.)
			IF(TEMP_REAL2>0) THEN
				REL_FREQ(I,J)=(OBS(I,J)+0.)/TEMP_REAL2
			ELSE
				REL_FREQ(I,J)=0.
			END IF
		END DO
	END DO

	DO I=1,ALLELES
		AVG_FREQ(I)=SUM(OBS(I,:))/(SUM(OBS)+0.)
!		PRINT*, "AVG_FREQ(I): ", AVG_FREQ(I)
	END DO

	TEMP_REAL1=0.
	DO I=1,SAMPLE_SITES
!		TEMP_REAL1=TEMP_REAL1+LOG_BINOMIAL(TWO*POP_SIZES(I),OBS(1,I)+ZERO,REL_FREQ(1,I))
		IF((REL_FREQ(1,I)>0.).AND.(REL_FREQ(1,I)<1.)) THEN
			TEMP_REAL1=OBS(1,I)*LOG(REL_FREQ(1,I))+(TWO*POP_SIZES(I)-OBS(1,I))*LOG(1.0-REL_FREQ(1,I))
!			LOG_BINOMIAL=LOG_COMBINATIONS(N,X)+X*LOG(P)+(N-X)*LOG(1.0-P)
		ELSE
			TEMP_REAL1=0.
		END IF
	END DO
	
!	TEMP_REAL2=LOG_BINOMIAL(TWO*INDIVIDUALS,SUM(OBS(1,:))+ZERO,AVG_FREQ(1))
	IF((AVG_FREQ(1)>0.).AND.(AVG_FREQ(1)<1.)) THEN
		TEMP_REAL2=SUM(OBS(1,:))*LOG(AVG_FREQ(1))+(TWO*INDIVIDUALS-SUM(OBS(1,:)))*LOG(1.0-AVG_FREQ(1))
	ELSE
		TEMP_REAL2=0.
	END IF

!	PRINT*, "INFORMATIVENESS: ", TEMP_REAL
	LIKELIHOOD_RATIO=TEMP_REAL1-TEMP_REAL2

	END FUNCTION LIKELIHOOD_RATIO
	
	
	
	




	SUBROUTINE CONVERT_TO_COORD(BEST_LOCATION,COORD,LONG,LAT,GRID_LENGTH,SAMPLE_SITES)

	IMPLICIT NONE

	INTEGER :: GRID_LENGTH,SAMPLE_SITES
	INTEGER, DIMENSION(2) :: BEST_LOCATION
	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: COORD !COORD(LONG=1;LAT=2,POPULATION)
	REAL(KIND=DBLE_PREC) :: LONG,LAT,LATSTEP,LONGSTEP,MINLONG,MAXLONG,MINLAT,MAXLAT


	!ESTABLISHING THE CORNERS OF THE BOX
	MINLONG=MINVAL(COORD(1,:))
	MAXLONG=MAXVAL(COORD(1,:))
	MINLAT=MINVAL(COORD(2,:))
	MAXLAT=MAXVAL(COORD(2,:))

	!ESTABLISHING STEP LENGTHS
	LONGSTEP=(MAXLONG-MINLONG)/(GRID_LENGTH-1.0)
	LATSTEP=(MAXLAT-MINLAT)/(GRID_LENGTH-1.0)

	LAT=MINLAT+(BEST_LOCATION(2)-1)*LATSTEP
	LONG=MINLONG+(BEST_LOCATION(1)-1)*LONGSTEP

	END SUBROUTINE CONVERT_TO_COORD





	SUBROUTINE UPDATE_GRID_COORD_SQUARE(GRID_COORD,COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
	!THIS SUBROUTINE UPDATES THE GRID COORDINATES MAKING EACH SITE A SQUARE
	!IT ALSO UPDATES THE GRID_LENGTH(2) VECTOR FOR THE DIFFERENT SIZES

	IMPLICIT NONE

	INTEGER :: I,J,K,SAMPLE_SITES,MAX_GRID_LENGTH
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: COORD !COORD(LONG=1;LAT=2,POPULATION)
	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
	REAL(KIND=DBLE_PREC) :: STEP,MINLONG,MAXLONG,MINLAT,MAXLAT
	REAL(KIND=DBLE_PREC) :: MAXSIZE,LONGDIST,LATDIST

	!ESTABLISHING THE CORNERS OF THE BOX
	MINLONG=MINVAL(COORD(1,:))
	!overriding value to inclde all of one country
!	MINLONG=-10.0
!	PRINT*, "Min Longitude set at -10"
	MAXLONG=MAXVAL(COORD(1,:))
	MINLAT=MINVAL(COORD(2,:))
	MAXLAT=MAXVAL(COORD(2,:))
!	MAXLAT=63.0
!	PRINT*, "Max Latitude set at 63"
	MAXSIZE=MAX(MAXLAT-MINLAT,MAXLONG-MINLONG)

	!ESTABLISHING STEP LENGTHS
	LONGDIST=(MAXLONG-MINLONG)
	LATDIST=(MAXLAT-MINLAT)
	
	IF(LONGDIST.GE.LATDIST) THEN
		GRID_LENGTH(1)=MAX_GRID_LENGTH
		STEP=LONGDIST/(GRID_LENGTH(1)-1.)
		GRID_LENGTH(2)=1
		DO I=1,MAX_GRID_LENGTH
			IF(MINLAT+I*STEP<MAXLAT) THEN
				GRID_LENGTH(2)=GRID_LENGTH(2)+1
			ELSE
				EXIT
			END IF
		END DO
	ELSE
		GRID_LENGTH(2)=MAX_GRID_LENGTH
		STEP=LATDIST/(GRID_LENGTH(2)-1.)
		GRID_LENGTH(1)=1
		DO I=1,MAX_GRID_LENGTH
			IF(MINLONG+I*STEP<MAXLONG) THEN
				GRID_LENGTH(1)=GRID_LENGTH(1)+1
			ELSE
				EXIT
			END IF
		END DO
	END IF
	
	GRID_COORD=0.
	DO I=1,GRID_LENGTH(1)
		GRID_COORD(1,I)=MINLONG+(I-1)*STEP
	END DO
	
	DO I=1,GRID_LENGTH(2)
		GRID_COORD(2,I)=MINLAT+(I-1)*STEP
	END DO

	END SUBROUTINE UPDATE_GRID_COORD_SQUARE





	SUBROUTINE FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID_INDIV,LONG,LAT,&
				MAX_GRID_LENGTH,GRID_COORD,IS_LAND)

	IMPLICIT NONE

	INTEGER :: I,J,MAX_GRID_LENGTH
	REAL(KIND=DBLE_PREC), DIMENSION(:,:) :: GRID_COORD !LONG=1;LAT=2,GRID_LENGTH
	REAL(KIND=DBLE_PREC), DIMENSION(:,:) :: UNKNOWN_GRID_INDIV
	LOGICAL, DIMENSION(:,:) :: IS_LAND
	INTEGER, DIMENSION(2) :: BEST_LOCATION
	REAL(KIND=DBLE_PREC) :: LONG,LAT,TEMP_REAL

!	TEMP_GRID=UNKNOWN_GRID_INDIV
!	TEMP_REAL=MINVAL(TEMP_GRID)
!!	WHERE(.NOT.IS_LAND) TEMP_GRID=TEMP_REAL-100
!	
!	DO I=1,GRID_LENGTH
!		DO J=1,GRID_LENGTH
!			IF(.NOT.IS_LAND(I,J)) THEN
!!				PRINT*, "NOT IS_LAND", I,J,IS_LAND(I,J)
!				TEMP_GRID(I,J)=TEMP_REAL-100
!			END IF
!		END DO
!	END DO
!	
!	BEST_LOCATION=MAXLOC(TEMP_GRID)
	
	BEST_LOCATION=MAXLOC(UNKNOWN_GRID_INDIV,IS_LAND)

	LONG=GRID_COORD(1,BEST_LOCATION(1))
	LAT=GRID_COORD(2,BEST_LOCATION(2))

	END SUBROUTINE FIND_OPTIMAL_LOCATION_LOCI



	SUBROUTINE UPDATE_UNKNOWN_GRID_SQUARE_LOG(ALLELE_FREQUENCY,UNKNOWN_OBS_LOCI,UNKNOWN_GRID,&
				GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN)
	
	IMPLICIT NONE

	INTEGER :: I,J,K,L,SAMPLE_SITES,LOCI,MAX_GRID_LENGTH,NUMBER_UNKNOWN,ERROR_INT
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY !LOCI,X,Y OR LOCI,LONG,LAT
	INTEGER, DIMENSION(NUMBER_UNKNOWN) :: UNKNOWN_OBS_LOCI
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN) :: UNKNOWN_GRID !LONG,LAT,NUMBER_UNKNOWN
	REAL(KIND=DBLE_PREC) :: TEMP_REAL,TEMP_REAL0,TEMP_REAL1,TEMP_REAL2,TEMP_REAL3,TEMP_REAL4
	
	ERROR_INT=0
	TEMP_REAL=0.
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			TEMP_REAL=ABS(ALLELE_FREQUENCY(I,J))
			TEMP_REAL3=ABS(ONE-ALLELE_FREQUENCY(I,J))
			TEMP_REAL0=TEMP_REAL3**2
			TEMP_REAL1=2*TEMP_REAL*(TEMP_REAL3)
			TEMP_REAL2=TEMP_REAL**2
			DO K=1,NUMBER_UNKNOWN
				IF(UNKNOWN_OBS_LOCI(K).EQ.0) THEN
					TEMP_REAL=TEMP_REAL0
				ELSE IF(UNKNOWN_OBS_LOCI(K).EQ.1) THEN
					TEMP_REAL=TEMP_REAL1
				ELSE IF(UNKNOWN_OBS_LOCI(K).EQ.2) THEN
					TEMP_REAL=TEMP_REAL2
				ELSE
					!PRINT*, "MISSING ALLELE"
					TEMP_REAL=-1.
				END IF
				IF(TEMP_REAL>0) THEN
					UNKNOWN_GRID(I,J,K)=UNKNOWN_GRID(I,J,K)+LOG10(TEMP_REAL)
				ELSE IF((TEMP_REAL.EQ.0.))THEN
					ERROR_INT=ERROR_INT-1
					TEMP_REAL4=ALLELE_FREQUENCY(I,J)
					UNKNOWN_GRID(I,J,K)=UNKNOWN_GRID(I,J,K)-20
				ELSE
!					PRINT*, "ERROR IN UPDATE_UNKNOWN_GRID. PROBABILITY LESS THAN 0.", TEMP_REAL
					CALL INTPR("ERROR IN UPDATE_UNKNOWN_GRID.  PROBABILITY LESS THAN 0.",-1,1,0)
					CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
				END IF
			END DO
		END DO
	END DO
	
!	IF(ERROR_INT<0) THEN
!		PRINT*, "TEMP_REAL = 0 IN UPDATE_UNKNOWN_GRID...AF: ", TEMP_REAL4
!		PRINT*, "NUMBER OF ERRORS: ", ERROR_INT
!	END IF

	END SUBROUTINE UPDATE_UNKNOWN_GRID_SQUARE_LOG



	SUBROUTINE UPDATE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_OBS_LOCI,UNKNOWN_GRID,&
				GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN,IS_LAND)
	
	IMPLICIT NONE

	INTEGER :: I,J,K,L,SAMPLE_SITES,LOCI,MAX_GRID_LENGTH,NUMBER_UNKNOWN,ERROR_INT
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY !LOCI,X,Y OR LOCI,LONG,LAT
	INTEGER, DIMENSION(NUMBER_UNKNOWN) :: UNKNOWN_OBS_LOCI
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN) :: UNKNOWN_GRID !LONG,LAT,NUMBER_UNKNOWN
	REAL(KIND=DBLE_PREC) :: TEMP_REAL,TEMP_REAL0,TEMP_REAL1,TEMP_REAL2,TEMP_REAL3,TEMP_REAL4
	LOGICAL, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: IS_LAND
	
	ERROR_INT=0
!	TEMP_REAL=SUM(UNKNOWN_GRID(:,:,INDIV))
!	PRINT*, "TEMP_REAL ORIGINAL: ", TEMP_REAL
	TEMP_REAL=0.
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			IF(IS_LAND(I,J)) THEN
				TEMP_REAL=ABS(ALLELE_FREQUENCY(I,J))
				TEMP_REAL3=ABS(ONE-ALLELE_FREQUENCY(I,J))
				TEMP_REAL0=TEMP_REAL3**2
				TEMP_REAL1=2*TEMP_REAL*(TEMP_REAL3)
				TEMP_REAL2=TEMP_REAL**2
				DO K=1,NUMBER_UNKNOWN
					IF(UNKNOWN_OBS_LOCI(K).EQ.0) THEN
						TEMP_REAL=TEMP_REAL0
					ELSE IF(UNKNOWN_OBS_LOCI(K).EQ.1) THEN
						TEMP_REAL=TEMP_REAL1
					ELSE IF(UNKNOWN_OBS_LOCI(K).EQ.2) THEN
						TEMP_REAL=TEMP_REAL2
					ELSE
						!PRINT*, "MISSING ALLELE"
						TEMP_REAL=-1.
					END IF
					IF(TEMP_REAL.GE.0) THEN
						UNKNOWN_GRID(I,J,K)=UNKNOWN_GRID(I,J,K)*TEMP_REAL
					ELSE
!						PRINT*, "ERROR IN UPDATE_UNKNOWN_GRID. PROBABILITY LESS THAN 0.", TEMP_REAL
						CALL INTPR("SOMETHINGS WRONG...ERROR IN UPDATE_UNKNOWN_GRID. PROBABILITY LESS THAN 0.",-1,1,0)
						CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
					END IF
				END DO
			END IF
		END DO
	END DO
	
	DO K=1,NUMBER_UNKNOWN
		TEMP_REAL=SUM(UNKNOWN_GRID(:,:,K))
	!	PRINT*, "TEMP_REAL: ", TEMP_REAL
		DO J=1,GRID_LENGTH(2)
			DO I=1,GRID_LENGTH(1)
				UNKNOWN_GRID(I,J,K)=UNKNOWN_GRID(I,J,K)/TEMP_REAL
			END DO
		END DO
	END DO
		
!	IF(ERROR_INT<0) THEN
!		PRINT*, "TEMP_REAL = 0 IN UPDATE_UNKNOWN_GRID...AF: ", TEMP_REAL4
!		PRINT*, "NUMBER OF ERRORS: ", ERROR_INT
!	END IF

	END SUBROUTINE UPDATE_UNKNOWN_GRID_SQUARE



	SUBROUTINE UPDATE_SINGLE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_OBS_LOCI,UNKNOWN_GRID,&
				GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN,INDIV,IS_LAND)

	IMPLICIT NONE

	INTEGER :: I,J,K,L,SAMPLE_SITES,LOCI,MAX_GRID_LENGTH,NUMBER_UNKNOWN,ERROR_INT,INDIV
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY !LOCI,X,Y OR LOCI,LONG,LAT
	INTEGER, DIMENSION(NUMBER_UNKNOWN) :: UNKNOWN_OBS_LOCI
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN) :: UNKNOWN_GRID !LONG,LAT,NUMBER_UNKNOWN
	REAL(KIND=DBLE_PREC) :: TEMP_REAL,TEMP_REAL0,TEMP_REAL1,TEMP_REAL2,TEMP_REAL3,TEMP_REAL4
	LOGICAL, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: IS_LAND
	
	ERROR_INT=0
!	TEMP_REAL=SUM(UNKNOWN_GRID(:,:,INDIV))
!	PRINT*, "TEMP_REAL ORIGINAL: ", TEMP_REAL
	TEMP_REAL=0.
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			IF(IS_LAND(I,J)) THEN
				TEMP_REAL=ABS(ALLELE_FREQUENCY(I,J))
				TEMP_REAL3=ABS(ONE-ALLELE_FREQUENCY(I,J))
				TEMP_REAL0=TEMP_REAL3**2
				TEMP_REAL1=2*TEMP_REAL*(TEMP_REAL3)
				TEMP_REAL2=TEMP_REAL**2
			
				IF(UNKNOWN_OBS_LOCI(INDIV).EQ.0) THEN
					TEMP_REAL=TEMP_REAL0
				ELSE IF(UNKNOWN_OBS_LOCI(INDIV).EQ.1) THEN
					TEMP_REAL=TEMP_REAL1
				ELSE IF(UNKNOWN_OBS_LOCI(INDIV).EQ.2) THEN
					TEMP_REAL=TEMP_REAL2
				ELSE
					!PRINT*, "MISSING ALLELE"
					TEMP_REAL=-1.
				END IF
				IF(TEMP_REAL.GE.0) THEN
					UNKNOWN_GRID(I,J,INDIV)=UNKNOWN_GRID(I,J,INDIV)*TEMP_REAL
				ELSE
!					PRINT*, "ERROR IN UPDATE_UNKNOWN_GRID. PROBABILITY LESS THAN 0.", TEMP_REAL
					CALL INTPR("SOMETHINGS WRONG...ERROR IN UPDATE_UNKNOWN_GRID. PROBABILITY LESS THAN 0.",-1,1,0)
					CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
				END IF
			END IF
		END DO
	END DO
	
	TEMP_REAL=SUM(UNKNOWN_GRID(:,:,INDIV))
!	PRINT*, "TEMP_REAL: ", TEMP_REAL
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			UNKNOWN_GRID(I,J,INDIV)=UNKNOWN_GRID(I,J,INDIV)/TEMP_REAL
		END DO
	END DO
		
!	IF(ERROR_INT<0) THEN
!		PRINT*, "TEMP_REAL = 0 IN UPDATE_UNKNOWN_GRID...AF: ", TEMP_REAL4
!		PRINT*, "NUMBER OF ERRORS: ", ERROR_INT
!	END IF

	END SUBROUTINE UPDATE_SINGLE_UNKNOWN_GRID_SQUARE






	SUBROUTINE UPDATE_UNKNOWN_VECTOR(ALLELE_FREQUENCY_VECTOR,UNKNOWN_OBS_LOCI,&
						UNKNOWN_VECTOR,VECTOR_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN)

	IMPLICIT NONE

	INTEGER :: I,J,K,L,SAMPLE_SITES,LOCI,VECTOR_LENGTH,NUMBER_UNKNOWN,ERROR_INT
	REAL(KIND=DBLE_PREC), DIMENSION(VECTOR_LENGTH) :: ALLELE_FREQUENCY_VECTOR !LOCI,X,Y OR LOCI,LONG,LAT
	INTEGER, DIMENSION(NUMBER_UNKNOWN) :: UNKNOWN_OBS_LOCI
	REAL(KIND=DBLE_PREC), DIMENSION(VECTOR_LENGTH,NUMBER_UNKNOWN) :: UNKNOWN_VECTOR !LONG,LAT,NUMBER_UNKNOWN
	REAL(KIND=DBLE_PREC) :: TEMP_REAL,TEMP_REAL0,TEMP_REAL1,TEMP_REAL2,TEMP_REAL3,TEMP_REAL4
	
	ERROR_INT=0
	TEMP_REAL=0.
	DO I=1,VECTOR_LENGTH
		TEMP_REAL=ABS(ALLELE_FREQUENCY_VECTOR(I))
		TEMP_REAL3=ABS(ONE-ALLELE_FREQUENCY_VECTOR(I))
		TEMP_REAL0=TEMP_REAL3**2
		TEMP_REAL1=2*TEMP_REAL*(TEMP_REAL3)
		TEMP_REAL2=TEMP_REAL**2
		DO K=1,NUMBER_UNKNOWN
			IF(UNKNOWN_OBS_LOCI(K).EQ.0) THEN
				TEMP_REAL=TEMP_REAL0
			ELSE IF(UNKNOWN_OBS_LOCI(K).EQ.1) THEN
				TEMP_REAL=TEMP_REAL1
			ELSE IF(UNKNOWN_OBS_LOCI(K).EQ.2) THEN
				TEMP_REAL=TEMP_REAL2
			ELSE
				!PRINT*, "MISSING ALLELE"
				TEMP_REAL=-1.
			END IF
			IF(TEMP_REAL.GE.0) THEN
				UNKNOWN_VECTOR(I,K)=UNKNOWN_VECTOR(I,K)*TEMP_REAL
			ELSE
!				PRINT*, "ERROR IN UPDATE_UNKNOWN_GRID. PROBABILITY LESS THAN 0.", TEMP_REAL
				CALL INTPR("SOMETHINGS WRONG...ERROR IN UPDATE UNKNOWN GRID.  PROBABILITY LESS THAN 0.",-1,1,0)
				CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
			END IF
		END DO
	END DO
	DO K=1,NUMBER_UNKNOWN
		TEMP_REAL=SUM(UNKNOWN_VECTOR(:,K))
		DO I=1,VECTOR_LENGTH
			UNKNOWN_VECTOR(I,K)=UNKNOWN_VECTOR(I,K)/TEMP_REAL
		END DO
	END DO
!	IF(ERROR_INT<0) THEN
!		PRINT*, "TEMP_REAL = 0 IN UPDATE_UNKNOWN_GRID...AF: ", TEMP_REAL4
!		PRINT*, "NUMBER OF ERRORS: ", ERROR_INT
!	END IF

	END SUBROUTINE UPDATE_UNKNOWN_VECTOR







	

	SUBROUTINE UPDATE_TRUE_MEMBERSHIPS(TRUE_MEMBERSHIP,UNKNOWN_COORD,COORD,SAMPLE_SITES,NUMBER_UNKNOWN)

	IMPLICIT NONE

	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_UNKNOWN,NUM_FOUND
	INTEGER, DIMENSION(NUMBER_UNKNOWN) :: TRUE_MEMBERSHIP
	REAL(KIND=DBLE_PREC), DIMENSION(2,NUMBER_UNKNOWN) :: UNKNOWN_COORD
	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: COORD !COORD(LONG=1;LAT=2,POPULATION)
	REAL(KIND=DBLE_PREC) :: TEMP_REAL

	TRUE_MEMBERSHIP=0
	NUM_FOUND=0

	DO I=1,NUMBER_UNKNOWN
		DO J=1,SAMPLE_SITES
			IF(UNKNOWN_COORD(1,I).EQ.COORD(1,J)) THEN
				IF(UNKNOWN_COORD(2,I).EQ.COORD(2,J)) THEN
					TRUE_MEMBERSHIP(I)=J
					NUM_FOUND=NUM_FOUND+1
				END IF
			END IF
		END DO
	END DO

!	PRINT*, "NUMBER NOT FOUND: ",NUMBER_UNKNOWN-NUM_FOUND

	END SUBROUTINE UPDATE_TRUE_MEMBERSHIPS




	FUNCTION CALC_DISTANCE(COORD1,COORD2)
	!(LONG,LAT)
	!CALCULATES THE GREAT CIRCLE DISTANCE BETWEEN THE TWO COORDINATES
	!NOTE 3959 IS THE RADIUS OF THE EARTH in miles
	!SEE http://mathforum.org/library/drmath/view/51722.html FOR MORE INFO

	IMPLICIT NONE

	REAL(KIND=DBLE_PREC), DIMENSION(2) :: COORD1,COORD2
	REAL(KIND=DBLE_PREC) :: CABC,SAB,PI,CALC_DISTANCE

	IF((COORD1(1).EQ.COORD2(1)).AND.(COORD1(2).EQ.COORD2(2))) THEN
		CALC_DISTANCE=0.
	ELSE
		PI = 3.1415926535d+0
		CABC=0.0
		SAB=0.0

		CABC = COS(PI*COORD1(1)/180)*COS(PI*COORD2(1)/180)*COS(PI*(COORD1(2)-COORD2(2))/180)
		SAB = SIN(PI*COORD1(1)/180)*SIN(PI*COORD2(1)/180)
		CALC_DISTANCE=(6378.1*ACOS(CABC+SAB))!6378.1 IS THE RADIUS OF THE EARTH IN km
	END IF

	END FUNCTION CALC_DISTANCE



	
	
	FUNCTION CALC_FRACTIONS_LOGLIK(UNKNOWN_INDIV,GENOME_FRACTIONS,&
		ALLELE_FREQUENCY_POP,LAMBDA,ALLELES,SAMPLE_SITES,NUMBER_SNPS)
		!THIS IS THE LOGLIKELIHOOD...
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,ALLELES,SAMPLE_SITES,NUMBER_SNPS
	REAL(KIND=DBLE_PREC), DIMENSION(NUMBER_SNPS,SAMPLE_SITES) :: ALLELE_FREQUENCY_POP
	REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES) :: GENOME_FRACTIONS
	INTEGER, DIMENSION(NUMBER_SNPS) :: UNKNOWN_INDIV
	REAL(KIND=DBLE_PREC) :: LAMBDA,CALC_FRACTIONS_LOGLIK,TEMP_REAL1,TEMP_REAL2
	
	!CALCULATE THE MODEL PART
	CALC_FRACTIONS_LOGLIK=0.
	DO I=1,NUMBER_SNPS
		TEMP_REAL1=0.
		TEMP_REAL2=0.
		DO J=1,SAMPLE_SITES
			TEMP_REAL1=TEMP_REAL1+GENOME_FRACTIONS(J)*ALLELE_FREQUENCY_POP(I,J)
			TEMP_REAL2=TEMP_REAL2+GENOME_FRACTIONS(J)*(1.-ALLELE_FREQUENCY_POP(I,J))
		END DO
		IF(TEMP_REAL1.GT.ZERO) THEN
			CALC_FRACTIONS_LOGLIK=CALC_FRACTIONS_LOGLIK+UNKNOWN_INDIV(I)*LOG(TEMP_REAL1)
		END IF
		IF(TEMP_REAL2.GT.ZERO) THEN
			CALC_FRACTIONS_LOGLIK=CALC_FRACTIONS_LOGLIK+(2.-UNKNOWN_INDIV(I))*LOG(TEMP_REAL2)
		END IF
	END DO
	
	!CALCULATE THE PENALTY PART...
	DO I=1,SAMPLE_SITES
		IF(GENOME_FRACTIONS(I)<DELTA) THEN
			CALC_FRACTIONS_LOGLIK=CALC_FRACTIONS_LOGLIK-GENOME_FRACTIONS(I)*LAMBDA
		ELSE
			CALC_FRACTIONS_LOGLIK=CALC_FRACTIONS_LOGLIK-DELTA*LAMBDA
		END IF
	END DO
	
	END FUNCTION CALC_FRACTIONS_LOGLIK
	
	
	
	
	FUNCTION RESHAPE_MATRIX_MASK(MATRIX1,MASK1,MATRIX_LENGTH,VECTOR_LENGTH)
	!THIS FUNCTION RESHAPES A MATRIX TO A VECTOR WITH A MASK
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,MATRIX_LENGTH,VECTOR_LENGTH,TEMP_COUNT
	REAL(KIND=DBLE_PREC), DIMENSION(MATRIX_LENGTH,MATRIX_LENGTH) :: MATRIX1
	REAL(KIND=DBLE_PREC), DIMENSION(VECTOR_LENGTH) :: RESHAPE_MATRIX_MASK
	LOGICAL, DIMENSION(MATRIX_LENGTH,MATRIX_LENGTH) :: MASK1
	
	RESHAPE_MATRIX_MASK=0.
	TEMP_COUNT=0
	DO I=1,MATRIX_LENGTH
		DO J=1,MATRIX_LENGTH
			IF(MASK1(J,I)) THEN
				TEMP_COUNT=TEMP_COUNT+1
				RESHAPE_MATRIX_MASK(TEMP_COUNT)=MATRIX1(J,I)
			END IF
		END DO
	END DO
	
	END FUNCTION RESHAPE_MATRIX_MASK
	
	
	
	FUNCTION RESHAPE_MATRIX_MASK_SQUARE(MATRIX1,MASK1,MATRIX_LENGTHS,VECTOR_LENGTH)
	!THIS FUNCTION RESHAPES A MATRIX TO A VECTOR WITH A MASK
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,VECTOR_LENGTH,TEMP_COUNT
	INTEGER, DIMENSION(2) :: MATRIX_LENGTHS
	REAL(KIND=DBLE_PREC), DIMENSION(MATRIX_LENGTHS(1),MATRIX_LENGTHS(2)) :: MATRIX1
	REAL(KIND=DBLE_PREC), DIMENSION(VECTOR_LENGTH) :: RESHAPE_MATRIX_MASK_SQUARE
	LOGICAL, DIMENSION(MATRIX_LENGTHS(1),MATRIX_LENGTHS(2)) :: MASK1
	
	RESHAPE_MATRIX_MASK_SQUARE=0.
	TEMP_COUNT=0
	DO I=1,MATRIX_LENGTHS(2)
		DO J=1,MATRIX_LENGTHS(1)
			IF(MASK1(J,I)) THEN
				TEMP_COUNT=TEMP_COUNT+1
				RESHAPE_MATRIX_MASK_SQUARE(TEMP_COUNT)=MATRIX1(J,I)
			END IF
		END DO
	END DO
	
	END FUNCTION RESHAPE_MATRIX_MASK_SQUARE
	
	
	
	
	FUNCTION RESHAPE_VECTOR_MASK(VECTOR1,MASK1,MATRIX_LENGTH,VECTOR_LENGTH)
	!THIS FUNCTION RESHAPES A MATRIX TO A VECTOR WITH A MASK
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,MATRIX_LENGTH,VECTOR_LENGTH,TEMP_COUNT
	REAL(KIND=DBLE_PREC), DIMENSION(VECTOR_LENGTH) :: VECTOR1
	REAL(KIND=DBLE_PREC), DIMENSION(MATRIX_LENGTH,MATRIX_LENGTH) :: RESHAPE_VECTOR_MASK
	LOGICAL, DIMENSION(MATRIX_LENGTH,MATRIX_LENGTH) :: MASK1
	
	RESHAPE_VECTOR_MASK=0.
	TEMP_COUNT=0
	DO I=1,MATRIX_LENGTH
		DO J=1,MATRIX_LENGTH
			IF(MASK1(J,I)) THEN
				TEMP_COUNT=TEMP_COUNT+1
				RESHAPE_VECTOR_MASK(J,I)=VECTOR1(TEMP_COUNT)
			END IF
		END DO
	END DO
	
	END FUNCTION RESHAPE_VECTOR_MASK
	
	
	
	FUNCTION RESHAPE_VECTOR_MASK_SQUARE(VECTOR1,MASK1,MATRIX_LENGTHS,VECTOR_LENGTH)
	!THIS FUNCTION RESHAPES A MATRIX TO A VECTOR WITH A MASK
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,VECTOR_LENGTH,TEMP_COUNT
	INTEGER, DIMENSION(2) :: MATRIX_LENGTHS
	REAL(KIND=DBLE_PREC), DIMENSION(VECTOR_LENGTH) :: VECTOR1
	REAL(KIND=DBLE_PREC), DIMENSION(MATRIX_LENGTHS(1),MATRIX_LENGTHS(2)) :: RESHAPE_VECTOR_MASK_SQUARE
	LOGICAL, DIMENSION(MATRIX_LENGTHS(1),MATRIX_LENGTHS(2)) :: MASK1
	
	RESHAPE_VECTOR_MASK_SQUARE=0.
	TEMP_COUNT=0
	DO I=1,MATRIX_LENGTHS(2)
		DO J=1,MATRIX_LENGTHS(1)
			IF(MASK1(J,I)) THEN
				TEMP_COUNT=TEMP_COUNT+1
				RESHAPE_VECTOR_MASK_SQUARE(J,I)=VECTOR1(TEMP_COUNT)
			END IF
		END DO
	END DO
	
	END FUNCTION RESHAPE_VECTOR_MASK_SQUARE
	
	
	
	
	
	
	SUBROUTINE UPDATE_FRACTION_PARS_ONCE(GENOME_FRACTIONS,ALLELE_FREQUENCY,&
			UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,SAMPLE_SITES)
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,ALLELES,SAMPLE_SITES,NUMBER_SNPS
	REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES) :: C,GENOME_FRACTIONS
	REAL(KIND=DBLE_PREC) :: E,D,T,LAMBDA,TEMP_DEN1,TEMP_DEN2,SUMC
	REAL(KIND=DBLE_PREC), DIMENSION(NUMBER_SNPS,SAMPLE_SITES) :: ALLELE_FREQUENCY
	INTEGER, DIMENSION(NUMBER_SNPS) :: UNKNOWN_INDIV
				
	ALLELES=2
!	CALL UPDATE_C(C,GENOME_FRACTIONS,ALLELE_FREQUENCY_POP,UNKNOWN_INDIV, &
!				ALLELES,SAMPLE_SITES,NUMBER_SNPS)
	!!UPDATES THE C VARIABLE
	C=0.
	DO J=1,NUMBER_SNPS
		TEMP_DEN1=0.
		TEMP_DEN2=0.
		DO I=1,SAMPLE_SITES
			TEMP_DEN1=TEMP_DEN1+GENOME_FRACTIONS(I)*ALLELE_FREQUENCY(J,I)
			TEMP_DEN2=TEMP_DEN2+GENOME_FRACTIONS(I)*(1-ALLELE_FREQUENCY(J,I))
		END DO
		DO K=1,SAMPLE_SITES
			C(K)=C(K)+UNKNOWN_INDIV(J)*GENOME_FRACTIONS(K)*ALLELE_FREQUENCY(J,K)/TEMP_DEN1 &
					+(2-UNKNOWN_INDIV(J))*GENOME_FRACTIONS(K)*(1-ALLELE_FREQUENCY(J,K))/TEMP_DEN2
		END DO
	END DO 
	
!	CALL UPDATE_EDT(E,D,T,C,GENOME_FRACTIONS,ALLELES,SAMPLE_SITES,NUMBER_SNPS, &
!			DELTA,LAMBDA)
	!!UPDATES E,D, AND T
	D = ZERO
	E = ZERO
	DO I = 1,SAMPLE_SITES
		IF (GENOME_FRACTIONS(I)>DELTA) THEN
			D = D+C(I)
		ELSE
			E = E+C(I)
		END IF
	END DO
	
	T=(D+E+LAMBDA-SQRT((D+E+LAMBDA)**2-4*LAMBDA*E))/(2.0*LAMBDA)
	
	
!	CALL UPDATE_GENOME_FRACTIONS(T,C,GENOME_FRACTIONS, &
!			SAMPLE_SITES,DELTA,LAMBDA)
	!!UPDATES THE GENOME FRACTIONS
	
	SUMC=SUM(C)
	DO K=1,SAMPLE_SITES
		IF(GENOME_FRACTIONS(K)>DELTA) THEN
			GENOME_FRACTIONS(K)=C(K)/(SUMC-LAMBDA*T)
		ELSE
			GENOME_FRACTIONS(K)=C(K)/(SUMC-LAMBDA*T+LAMBDA)
		END IF
	END DO
	
	END SUBROUTINE UPDATE_FRACTION_PARS_ONCE
	
	
	
	
	SUBROUTINE FIT_FRACTIONS_MODEL(GENOME_FRACTIONS,ALLELE_FREQUENCY_POP,&
			UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,SAMPLE_SITES)
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,ALLELES,SAMPLE_SITES,NUMBER_SNPS,ITERATIONS
	REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES) :: GENOME_FRACTIONS
	REAL(KIND=DBLE_PREC) :: LAMBDA,LOGLIK,NEW_LOGLIK
	REAL(KIND=DBLE_PREC), DIMENSION(NUMBER_SNPS,SAMPLE_SITES) :: ALLELE_FREQUENCY_POP
	INTEGER, DIMENSION(NUMBER_SNPS) :: UNKNOWN_INDIV
	LOGICAL :: NOT_CONVERGED
	
	ALLELES=2
	LOGLIK=0.
	NEW_LOGLIK=0.
	NOT_CONVERGED=.TRUE.
	ITERATIONS=0
	!need to initialize elsewhere...
!	GENOME_FRACTIONS=ONE/(SAMPLE_SITES+ZERO)
	DO WHILE(NOT_CONVERGED)
		ITERATIONS=ITERATIONS+1
		
		CALL UPDATE_FRACTION_PARS_ONCE(GENOME_FRACTIONS,ALLELE_FREQUENCY_POP,&
			UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,SAMPLE_SITES)
			
!		TEMP_REAL2=CALC_FRACTIONS_MODEL(C,GENOME_FRACTIONS,LAMBDA,ALLELES,SAMPLE_SITES,DELTA)
		NEW_LOGLIK=CALC_FRACTIONS_LOGLIK(UNKNOWN_INDIV,GENOME_FRACTIONS,ALLELE_FREQUENCY_POP, &
				LAMBDA,ALLELES,SAMPLE_SITES,NUMBER_SNPS)		
		IF(ITERATIONS>5) THEN
			IF(NEW_LOGLIK-LOGLIK<ZERO) THEN
!				PRINT*, "ASCENT CRITERION FAIL: NEW, DIFF:",NEW_LOGLIK, NEW_LOGLIK-LOGLIK 
				CALL DBLEPR("ASCENT CRITERION FAIL(THIS IS OKAY IF SMALL)",-1,NEW_LOGLIK-LOGLIK,0)
			END IF
			CALL CHECK_CONVERGENCE(LOGLIK,NEW_LOGLIK,ITERATIONS,NOT_CONVERGED)
		END IF
		LOGLIK=NEW_LOGLIK
!		IF(MOD(ITERATIONS,500).EQ.0) THEN
!			PRINT*, "ITERATIONS: ",ITERATIONS, "LOGLIK: ", NEW_LOGLIK
!		END IF
	END DO
!	PRINT*, "ITERATIONS: ",ITERATIONS, "LOGLIK: ", NEW_LOGLIK
	
	END SUBROUTINE FIT_FRACTIONS_MODEL
	
	
	
	
	
	SUBROUTINE FIT_FRACTIONS_MODEL_QNEWT(GENOME_FRACTIONS,&
			ALLELE_FREQUENCY_NODES,UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,NODES)
	!HERE SAMPLE_SITES IS THE NUMBER OF EFFECTIVE "SAMPLE_SITES" TO CHECK
	!IT IS EITHER EQUAL TO SAMPLE_SITES OR GRID_LENGTH*GRID_LENGTH
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,ALLELES,NODES,NUMBER_SNPS,ITERATIONS
	REAL(KIND=DBLE_PREC), DIMENSION(NODES) :: GENOME_FRACTIONS,OLD_FRACTIONS
	REAL(KIND=DBLE_PREC) :: LAMBDA,TEMP_REAL1,TEMP_REAL2
	REAL(KIND=DBLE_PREC), DIMENSION(NUMBER_SNPS,NODES) :: ALLELE_FREQUENCY_NODES
	INTEGER, DIMENSION(NUMBER_SNPS) :: UNKNOWN_INDIV
	LOGICAL :: NOT_CONVERGED
	INTEGER :: QSEC=6,ERROR_INT
	REAL(KIND=DBLE_PREC) :: LOGLIK,NEW_LOGLIK
	!QUASI NEWTON ELEMENTS
	REAL(KIND=DBLE_PREC), DIMENSION(NODES) :: TEMP_VEC,TEMP_F,TEMP_F2,TEMP_NEW
	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: MAP_U,MAP_V
	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: TEMP_MAT,TEMPINV_MAT
	
	
	QSEC=6
	ALLOCATE(MAP_U(NODES,QSEC),MAP_V(NODES,QSEC))
	ALLOCATE(TEMP_MAT(QSEC,QSEC),TEMPINV_MAT(QSEC,QSEC))
!	PRINT*, "BEGIN FIT MODEL"
	!INITIALIZATION
	
	!!NEED TO UNCOMMENT THIS!!!!!!!!!!!!---------------------------------
	!GENOME FRACTIONS HAS BEEN INITIALIZED
!	GENOME_FRACTIONS=ONE/(NODES+ZERO)
	OLD_FRACTIONS=GENOME_FRACTIONS
	
	!HERE WE PERFORM NORMAL MM UPDATES TO STABALIZE THE QUASI-NEWTON UPDATES
	DO I=1,20
		CALL UPDATE_FRACTION_PARS_ONCE(GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES,&
			UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,NODES)
	END DO
	
	OLD_FRACTIONS=GENOME_FRACTIONS
	CALL UPDATE_FRACTION_PARS_ONCE(GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES,&
			UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,NODES)
	NOT_CONVERGED=.TRUE.
	
	TEMP_VEC=GENOME_FRACTIONS
	DO I=1,QSEC+1
		CALL UPDATE_FRACTION_PARS_ONCE(GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES,&
			UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,NODES)
		TEMP_F=GENOME_FRACTIONS
		IF(I.LE.QSEC) THEN
			MAP_U(:,I)=TEMP_F-TEMP_VEC
		END IF
		IF(I.GE.2) THEN
			MAP_V(:,I-1)=TEMP_F-TEMP_VEC
		END IF
		TEMP_VEC=TEMP_F
	END DO
	
	ITERATIONS=0
	DO WHILE(NOT_CONVERGED)
		ITERATIONS=ITERATIONS+1
		
		TEMP_VEC=GENOME_FRACTIONS
		CALL UPDATE_FRACTION_PARS_ONCE(GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES,&
			UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,NODES)
		TEMP_F=GENOME_FRACTIONS
		CALL UPDATE_FRACTION_PARS_ONCE(GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES,&
			UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,NODES)
		TEMP_F2=GENOME_FRACTIONS
		
		CALL SHIFT_MAT_LEFT(MAP_U,NODES,QSEC)
		MAP_U(:,QSEC)=TEMP_F-TEMP_VEC
		CALL SHIFT_MAT_LEFT(MAP_V,NODES,QSEC)
		MAP_V(:,QSEC)=TEMP_F2-TEMP_F

		TEMP_MAT=FIRST_MATMULT(MAP_U,MAP_V,NODES,QSEC)

		IF(IEEE_IS_NAN(SUM(TEMP_MAT))) THEN
			ERROR_INT=-1
		ELSE
			CALL FINDINV(TEMP_MAT,TEMPINV_MAT,QSEC,ERROR_INT)
		END IF
		IF(IEEE_IS_NAN(SUM(TEMPINV_MAT))) THEN
			ERROR_INT=-1
		END IF
	
		IF (ERROR_INT.LE.-1) THEN
			TEMP_NEW=TEMP_F2
			!CHOOSING TO BAIL ON THE QNEWT UPDATES IF SOMETHING GOES WRONG...
!			PRINT*, "FOUND NANS IN QNEWT UPDATE... UPDATED NORMALLY"
			CALL INTPR("FOUND NANS IN QNEWT UPDATE...UPDATED NORMALLY",-1,1,0)
			GENOME_FRACTIONS=TEMP_NEW
			CALL FIT_FRACTIONS_MODEL(GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES,&
					UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,NODES)
			RETURN
		ELSE
			TEMP_NEW=TEMP_F-MATMUL(MAP_V,MATMUL(TEMPINV_MAT,MATMUL(TRANSPOSE(MAP_U),TEMP_VEC-TEMP_F)))
			IF(IEEE_IS_NAN(SUM(TEMP_NEW))) THEN
!				PRINT*, "NAN FOUND IN UPDATE PARS QNEWT"
!				PRINT*, "TEMP_NEW"
!				PRINT*, TEMP_NEW
!				PRINT*, "MAP_V"
!				PRINT*, MAP_V
!				PRINT*, "MAP_U"
!				PRINT*, MAP_U
!				PRINT*, "TEMP_MAT"
!				PRINT*, TEMP_MAT
!				PRINT*, "TEMPINV_MAT"
!				PRINT*, TEMPINV_MAT
!				PRINT*, "TEMP_VEC"
!				PRINT*, TEMP_VEC
!				PRINT*, "TEMP_F"
!				PRINT*, TEMP_F
!				PRINT*, "NAN FOUND IN UPDATE PARS QNEWT...UPDATED PARS NORMALLY"
				CALL INTPR("NAN FOUND IN UPDATE PARS QNEWT...UPDATED NORMALLY",-1,1,0)
				TEMP_NEW=TEMP_F2
			END IF
			
			!THIS ENSURES THAT OUR QUASI NEWTON UPDATES DO NOT MEET OR PASS OUR 
			!BOUNDARIES OF (0,1)
			DO I=1,NODES
				IF(TEMP_NEW(I).LE.0) THEN
					TEMP_NEW(I)=TEMP_VEC(I)/2.
				ELSE IF(TEMP_NEW(I).GE.1.) THEN
					TEMP_NEW(I)=(TEMP_VEC(I)+1.)/2.
				END IF
			END DO
			TEMP_REAL1=SUM(TEMP_NEW)
			GENOME_FRACTIONS=TEMP_NEW/TEMP_REAL1
			
		END IF
		
		!CHECKS FOR CONVERGENCE
		IF((ITERATIONS.GE.20)) THEN
			OLD_FRACTIONS=GENOME_FRACTIONS
			LOGLIK=CALC_FRACTIONS_LOGLIK(UNKNOWN_INDIV,GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES, &
				LAMBDA,ALLELES,NODES,NUMBER_SNPS)
			CALL UPDATE_FRACTION_PARS_ONCE(GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES,&
					UNKNOWN_INDIV,LAMBDA,NUMBER_SNPS,NODES)
			NEW_LOGLIK=CALC_FRACTIONS_LOGLIK(UNKNOWN_INDIV,GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES, &
				LAMBDA,ALLELES,NODES,NUMBER_SNPS)
			CALL CHECK_CONVERGENCE(LOGLIK,NEW_LOGLIK,ITERATIONS,NOT_CONVERGED)
		END IF
		
		IF((MOD(ITERATIONS,1).EQ.0).OR.(.NOT.NOT_CONVERGED)) THEN
			IF(ITERATIONS.LT.20) THEN
				NEW_LOGLIK=CALC_FRACTIONS_LOGLIK(UNKNOWN_INDIV,GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES, &
					LAMBDA,ALLELES,NODES,NUMBER_SNPS)
			END IF
!			PRINT*, "QNEWT ITERATIONS: ", ITERATIONS, "LOGLIK: ", NEW_LOGLIK, "Sum to 1: ", SUM(GENOME_FRACTIONS)
!			IF(NEW_LOGLIK.GT.ZERO) THEN
!				PRINT*, "GENOME_FRACTIONS   ", "ALLELE_FREQUENCY"
!				DO J=1,MIN(50,NODES)
!					WRITE(*,"(100F14.6)") GENOME_FRACTIONS(J), ALLELE_FREQUENCY_NODES(1,J), ALLELE_FREQUENCY_NODES(2,J), &
!							ALLELE_FREQUENCY_NODES(1,J)
!!					PRINT 1000, GENOME_FRACTIONS(J), ALLELE_FREQUENCY_NODES(1,J), ALLELE_FREQUENCY_NODES(2,J), &
!!							ALLELE_FREQUENCY_NODES(1,J)
!				END DO
!!				1000 FORMAT (100F14.6)
!			END IF
		END IF
	END DO

	IF(ITERATIONS.EQ.0) THEN
		LOGLIK=CALC_FRACTIONS_LOGLIK(UNKNOWN_INDIV,GENOME_FRACTIONS,ALLELE_FREQUENCY_NODES, &
			LAMBDA,ALLELES,NODES,NUMBER_SNPS)
!		PRINT*, "QNEWT ITERATIONS: ", ITERATIONS, "LOGLIK: ", LOGLIK
	END IF
	
	END SUBROUTINE FIT_FRACTIONS_MODEL_QNEWT
	
	
	
END MODULE ALLELEFREQUENCIES
!THE END OF THE MODULE ---------------------------------------------------------------------








!BEGIN ORIGEN MODULE ---------------------------------------------------------------------
MODULE ORIGENMOD

	USE CONSTANTS
	USE QUASINEWTON
	USE ALLELEFREQUENCIES
	
	IMPLICIT NONE

	CONTAINS
	
!	SUBROUTINE COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	!THIS SUBROUTINE COUNTS THE NUMBER OF SAMPLE_SITES SO THAT VECTORS
!	!CAN BE ALLOCATED
!	
!	IMPLICIT NONE
!	
!	INTEGER :: SAMPLE_SITES,LINES,NUMBER_SNPS,INPUT_UNIT=7
!	CHARACTER(LEN=800) :: DATA_FILE,LINE,TEXT_END,INPUT_FILE
!	
!!	PRINT*, "START COUNT SAMPLE_SITES"
!	
!	TEXT_END=" "
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	INPUT_FILE="Input"//TRIM(DATA_FILE)//TRIM(TEXT_END)//".txt"
!	
!	CALL INPUT_DATA(INPUT_FILE,LINE,INPUT_UNIT,LINES)
!	CLOSE(INPUT_UNIT)
!	SAMPLE_SITES=LINES-1
!	
!!	PRINT*, "END COUNT SAMPLE_SITES"
!	
!	END SUBROUTINE COUNT_SAMPLE_SITES
!	
!	
!	
!	SUBROUTINE COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	
!	IMPLICIT NONE
!	
!	INTEGER :: SAMPLE_SITES,LINES,NUMBER_SNPS,INPUT_UNIT=7,NUMBER_UNKNOWN
!	CHARACTER(LEN=800) :: DATA_FILE,LINE,TEXT_END,INPUT_FILE
!	
!	TEXT_END=" "
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	INPUT_FILE="Unknown"//TRIM(DATA_FILE)//TRIM(TEXT_END)//".txt"
!	
!	CALL INPUT_DATA(INPUT_FILE,LINE,INPUT_UNIT,LINES)
!	CLOSE(INPUT_UNIT)
!	NUMBER_UNKNOWN=LINES-1
!	
!	END SUBROUTINE COUNT_UNKNOWN_INDIVIDUALS
	!
!	
!	
!	SUBROUTINE READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	!THIS SUBROUTINE READS IN DATA AND OUTPUTS IT TO DATA_VECTOR AND DATA_LOCATIONS
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,INPUT_UNIT=8,LINES,TEMP_INT,NUMBER_SNPS
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_END,LINE,INPUT_FILE
!	INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: DATA_COORD !(LONG,LAT)
!	
!!	PRINT*, "START READ DATA"
!	
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	
!	TEXT_END=" "
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	INPUT_FILE="Input"//TRIM(DATA_FILE)//TRIM(TEXT_END)//".txt"
!
!	!READS IN DATA
!	!DATA SHOULD BE IN THE FOLLOWING FORM
!	!ALLELES A LOCI L
!	!POP1 LONG1 LAT1 L1A1 L1A2 L2A1 L2A2
!	!POP2 LONG2 LAT2 L1A1 L1A2 L2A1 L2A2
!	CALL INPUT_DATA(INPUT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!	READ(INPUT_UNIT,*) LINE, TEMP_INT, LINE, TEMP_INT
!	
!	DO I=1,SAMPLE_SITES
!		READ(INPUT_UNIT,*) TEMP_INT, DATA_COORD(1,I), DATA_COORD(2,I), &
!						 (DATA_MATRIX(1,I,J), DATA_MATRIX(2,I,J), J=1,NUMBER_SNPS)
!	END DO
!	CLOSE(INPUT_UNIT)
!	
!!	CALL PRINT_INT_MATRIX(DATA_MATRIX(:,:,1))
!!	
!!	PRINT*, "LOCUS 2"
!!	CALL PRINT_INT_MATRIX(DATA_MATRIX(:,:,2))
!!	
!	END SUBROUTINE READ_GENO_DATA
!	
!	
!	
!	
!	
!	
!	SUBROUTINE READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP, &
!					NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,INPUT_UNIT=8,LINES,TEMP_INT,NUMBER_SNPS,NUMBER_UNKNOWN
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_END,LINE,INPUT_FILE
!	INTEGER, DIMENSION(NUMBER_UNKNOWN,NUMBER_SNPS) :: UNKNOWN_OBS !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,NUMBER_UNKNOWN) :: UNKNOWN_COORD !(LONG,LAT)
!	INTEGER, DIMENSION(2,NUMBER_UNKNOWN) :: ADMIXED_MEMBERSHIP
!	
!!	PRINT*, "START READ DATA"
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	
!	TEXT_END=" "
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	INPUT_FILE="Unknown"//TRIM(DATA_FILE)//TRIM(TEXT_END)//".txt"
!
!	!READS IN UNKNOWN DATA
!	!DATA SHOULD BE IN THE FOLLOWING FORM
!	!ALLELES A LOCI L
!	!TRUEPOP1 LONG1 LAT1 L1A1 L1A2 L2A1 L2A2
!	!TRUEPOP2 LONG1 LAT1 L1A1 L1A2 L2A1 L2A2
!	CALL INPUT_DATA(INPUT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!	READ(INPUT_UNIT,*) LINE, TEMP_INT, LINE, TEMP_INT, LINE, TEMP_INT
!	DO I=1,NUMBER_UNKNOWN
!		READ(INPUT_UNIT,*) ADMIXED_MEMBERSHIP(1,I), ADMIXED_MEMBERSHIP(2,I), &
!						(UNKNOWN_OBS(I,J), TEMP_INT, J=1,NUMBER_SNPS)
!	END DO
!	CLOSE(INPUT_UNIT)
!	
!!	PRINT*, "END READ DATA"
!	
!	END SUBROUTINE READ_UNKNOWN_DATA
!	
!	
!	
!	
!	SUBROUTINE READ_UNKNOWN_DATAG(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP, &
!					NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,INPUT_UNIT=8,LINES,TEMP_INT,NUMBER_SNPS,NUMBER_UNKNOWN
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_END,LINE,INPUT_FILE
!	INTEGER, DIMENSION(NUMBER_UNKNOWN,NUMBER_SNPS) :: UNKNOWN_OBS !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,NUMBER_UNKNOWN) :: UNKNOWN_COORD !(LONG,LAT)
!	INTEGER, DIMENSION(4,NUMBER_UNKNOWN) :: ADMIXED_MEMBERSHIP
!	
!!	PRINT*, "START READ DATA"
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	
!	TEXT_END=" "
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	INPUT_FILE="Unknown"//TRIM(DATA_FILE)//TRIM(TEXT_END)//".txt"
!
!	!READS IN UNKNOWN DATA
!	!DATA SHOULD BE IN THE FOLLOWING FORM
!	!ALLELES A LOCI L
!	!TRUEPOP1 LONG1 LAT1 L1A1 L1A2 L2A1 L2A2
!	!TRUEPOP2 LONG1 LAT1 L1A1 L1A2 L2A1 L2A2
!	CALL INPUT_DATA(INPUT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!	READ(INPUT_UNIT,*) LINE, TEMP_INT, LINE, TEMP_INT, LINE, TEMP_INT
!	DO I=1,NUMBER_UNKNOWN
!		READ(INPUT_UNIT,*) ADMIXED_MEMBERSHIP(1,I), ADMIXED_MEMBERSHIP(2,I), &
!						ADMIXED_MEMBERSHIP(3,I), ADMIXED_MEMBERSHIP(4,I), &
!						(UNKNOWN_OBS(I,J), TEMP_INT, J=1,NUMBER_SNPS)
!	END DO
!	CLOSE(INPUT_UNIT)
!	
!!	PRINT*, "END READ DATA"
!	
!	END SUBROUTINE READ_UNKNOWN_DATAG
!	
!	

	
		
	SUBROUTINE POOL_GENO_DATA_SQUARE(DATA_VECTOR,COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,&
			SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
	!THIS SUBROUTINE TAKES IN DATA AND POOLS THE INFORMATION INTO A MATRIX(OF MOSTLY ZEROES)
	!FOR ANALYSIS.
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,SAMPLE_SITES,MAX_GRID_LENGTH,LOC1,LOC2
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(2,SAMPLE_SITES) :: DATA_VECTOR !(FIRST IS ALLELE1, SECOND IS ALLELE2)
	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: COORD !(LONG,LAT)
	INTEGER, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_ALLELE1,DATA_TOTAL
	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD
	REAL(KIND=DBLE_PREC) :: DIST1, DIST2, TEMP_DIST
	
!	PRINT*, "START POOL DATA"
	
	DATA_ALLELE1=0
	DATA_TOTAL=0
	DO I=1,SAMPLE_SITES
		LOC1=1
		DIST1=ABS(COORD(1,I)-GRID_COORD(1,LOC1))
		LOC2=1
		DIST2=ABS(COORD(2,I)-GRID_COORD(2,LOC2))
		DO J=2,GRID_LENGTH(1)
			TEMP_DIST=ABS(COORD(1,I)-GRID_COORD(1,J))
			IF(TEMP_DIST<DIST1) THEN
				LOC1=J
				DIST1=TEMP_DIST
			END IF
		END DO	
		
		DO J=2,GRID_LENGTH(2)
			TEMP_DIST=ABS(COORD(2,I)-GRID_COORD(2,J))
			IF(TEMP_DIST<DIST2) THEN
				LOC2=J
				DIST2=TEMP_DIST
			END IF
		END DO
		DATA_ALLELE1(LOC1,LOC2)=DATA_ALLELE1(LOC1,LOC2)+DATA_VECTOR(1,I)
		DATA_TOTAL(LOC1,LOC2)=DATA_TOTAL(LOC1,LOC2)+DATA_VECTOR(2,I)+DATA_VECTOR(1,I)
	END DO
		
!	PRINT*, "END POOL DATA"
	
	END SUBROUTINE POOL_GENO_DATA_SQUARE
	

	
	
	SUBROUTINE INITIALIZE_ALLELE_FREQUENCY_SQUARE(DATA_ALLELE1,DATA_TOTAL,&
			ALLELE_FREQUENCY,GRID_LENGTH,MAX_GRID_LENGTH)
	!INITIALIZES THE ALLELE FREQUENCY MATRIX BY TAKING WEIGHTED AVERAGES
	!OF THE LOCATIONS WITH DATA (SIMILAR TO INVERSE DISTANCE WEIGHTING)
	!UPDATED: THIS NOW AVERAGES THE INVERSE DISTANCE WEIGHTED VALUES
	!WITH THE AVERAGE ALLELE FREQUENCY DISREGARDING POPULATION STRUCTURE
	
	IMPLICIT NONE
	
	INTEGER :: I,J,K,MAX_GRID_LENGTH,NON_ZERO
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_ALLELE1,DATA_TOTAL
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY,A1,A2,A3
!	INTEGER, DIMENSION(2,GRID_LENGTH*GRID_LENGTH) :: DATA_LOC
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH*GRID_LENGTH) :: DATA_VEC,WEIGHTS
	REAL(KIND=DBLE_PREC) :: AVG_FREQ
	
	
!	PRINT*, "INITIALIZED ALLELE FREQ TO AVERAGE OF AVERAGE AND INVERSE DISTANCE"
	A1=0.
	A2=0.
	A3=0.
	ALLELE_FREQUENCY=0.
	
	AVG_FREQ=(ZERO+SUM(DATA_ALLELE1))/SUM(DATA_TOTAL)
	
!	CALL INITIALIZE_ALLELE_FREQUENCY2(DATA_ALLELE1,DATA_TOTAL,A2,GRID_LENGTH)
	A3=AVG_FREQ
	CALL INITIALIZE_ALLELE_FREQUENCY4_SQUARE(DATA_ALLELE1,DATA_TOTAL,A1,GRID_LENGTH,MAX_GRID_LENGTH)
	
	ALLELE_FREQUENCY=(6.*A1+4.*A3+PSEUDO_COUNT)/(10.+2*PSEUDO_COUNT)
!	ALLELE_FREQUENCY=A3
	
!	DO J=1,GRID_LENGTH(2)
!		DO I=1,GRID_LENGTH(1)
!			IF((ALLELE_FREQUENCY(I,J).EQ.0).OR.(ALLELE_FREQUENCY(I,J).EQ.1)) THEN
!				PRINT*, "INITIALIZED TO ZERO!!!!! "
!				STOP
!			END IF
!		END DO
!	END DO
	
	END SUBROUTINE INITIALIZE_ALLELE_FREQUENCY_SQUARE
	
	
	
!	
!	SUBROUTINE INITIALIZE_ALLELE_FREQUENCY2(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,GRID_LENGTH)
!	!INITIALIZES THE ALLELE FREQUENCY MATRIX BY TAKING THE VALUE OF THE
!	!NEAREST PIXEL WITH DATA (NEAREST NEIGHBOR)
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,GRID_LENGTH,NON_ZERO
!	INTEGER, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY
!	INTEGER, DIMENSION(2,GRID_LENGTH*GRID_LENGTH) :: DATA_LOC
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH*GRID_LENGTH) :: DATA_VEC,WEIGHTS
!	
!!	PRINT*, "INITIALIZE ALLELE FREQUENCY"
!	
!	!COUNT THE NUMBER OF NON_ZERO CELLS
!	NON_ZERO=0
!	DATA_LOC=0
!	DATA_VEC=0.
!	DO J=1,GRID_LENGTH
!		DO I=1,GRID_LENGTH
!			IF(DATA_TOTAL(I,J)>0) THEN
!				NON_ZERO=NON_ZERO+1
!				IF(DATA_ALLELE1(I,J).EQ.0) THEN
!					DATA_VEC(NON_ZERO)=EPS
!				ELSE IF(DATA_ALLELE1(I,J).EQ.DATA_TOTAL(I,J)) THEN
!					DATA_VEC(NON_ZERO)=1.-EPS
!				ELSE
!					DATA_VEC(NON_ZERO)=(DATA_ALLELE1(I,J)+ZERO)/DATA_TOTAL(I,J)
!				END IF
!				DATA_LOC(1,NON_ZERO)=I
!				DATA_LOC(2,NON_ZERO)=J
!			END IF
!		END DO
!	END DO
!	
!	DO J=1,GRID_LENGTH
!		DO I=1,GRID_LENGTH
!			IF(DATA_TOTAL(I,J)>0) THEN
!				!INITIAL VALUE IS THE MLE FOR THE LOCATION
!				IF(DATA_ALLELE1(I,J).EQ.0) THEN
!					ALLELE_FREQUENCY(I,J)=EPS
!				ELSE IF(DATA_ALLELE1(I,J).EQ.DATA_TOTAL(I,J)) THEN
!					ALLELE_FREQUENCY(I,J)=1.-EPS
!				ELSE
!					ALLELE_FREQUENCY(I,J)=(DATA_ALLELE1(I,J)+ZERO)/DATA_TOTAL(I,J)
!				END IF
!			ELSE
!				!INITIAL VALUE IS THE VALUE OF THE NEAREST DATA POINT
!				WEIGHTS=0
!				DO K=1,NON_ZERO
!					WEIGHTS(K)=ONE/SQRT((I-DATA_LOC(1,K))**2+(J-DATA_LOC(2,K))**2+ZERO)
!				END DO
!				ALLELE_FREQUENCY(I,J)=DATA_VEC(MAXLOC(WEIGHTS,1))
!			END IF
!		END DO
!	END DO
!
!!	PRINT*, "END ALLELE FREQUENCY"
!	
!	END SUBROUTINE INITIALIZE_ALLELE_FREQUENCY2
!	
!	
!	
!	
!	SUBROUTINE INITIALIZE_ALLELE_FREQUENCY3(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,GRID_LENGTH)
!	!INITIALIZES THE ALLELE FREQUENCY MATRIX TO A FIXED VALUE WHICH IS THE
!	!AVERAGE ALLELE FREQUENCY ACROSS ALL SAMPLE_SITES
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,GRID_LENGTH
!	INTEGER, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY
!	
!	ALLELE_FREQUENCY=(ZERO+SUM(DATA_ALLELE1))/SUM(DATA_TOTAL)
!	PRINT*, "INITIAL FREQUENCY: ", (ZERO+SUM(DATA_ALLELE1))/SUM(DATA_TOTAL)
!	
!	END SUBROUTINE INITIALIZE_ALLELE_FREQUENCY3
!	
!	
!	
!	
!	SUBROUTINE INITIALIZE_ALLELE_FREQUENCY4(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,GRID_LENGTH)
!	!INITIALIZES THE ALLELE FREQUENCY MATRIX BY TAKING WEIGHTED AVERAGES
!	!OF THE LOCATIONS WITH DATA (SIMILAR TO INVERSE DISTANCE WEIGHTING)
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,GRID_LENGTH,NON_ZERO
!	INTEGER, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY
!	INTEGER, DIMENSION(2,GRID_LENGTH*GRID_LENGTH) :: DATA_LOC
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH*GRID_LENGTH) :: DATA_VEC,WEIGHTS
!	REAL(KIND=DBLE_PREC) :: AVG_FREQ
!		
!	AVG_FREQ=(ZERO+SUM(DATA_ALLELE1))/SUM(DATA_TOTAL)
!	
!	!COUNT THE NUMBER OF NON_ZERO CELLS
!	NON_ZERO=0
!	DATA_LOC=0
!	DATA_VEC=0.
!	DO J=1,GRID_LENGTH
!		DO I=1,GRID_LENGTH
!			IF(DATA_TOTAL(I,J)>0) THEN
!				NON_ZERO=NON_ZERO+1
!				DATA_VEC(NON_ZERO)=(DATA_ALLELE1(I,J)+ZERO)/DATA_TOTAL(I,J)
!				DATA_LOC(1,NON_ZERO)=I
!				DATA_LOC(2,NON_ZERO)=J
!			END IF
!		END DO
!	END DO
!	
!	DO J=1,GRID_LENGTH
!		DO I=1,GRID_LENGTH
!			IF(DATA_TOTAL(I,J)>0) THEN
!				!INITIAL VALUE IS THE MLE FOR THE LOCATION
!				ALLELE_FREQUENCY(I,J)=((DATA_ALLELE1(I,J)+ZERO)/DATA_TOTAL(I,J))
!			ELSE
!				!INITIAL VALUE IS THE VALUE OBTAINED FROM INVERSE DISTANCE WEIGHTING
!				WEIGHTS=0
!				DO K=1,NON_ZERO
!					WEIGHTS(K)=ONE/SQRT((I-DATA_LOC(1,K))**2+(J-DATA_LOC(2,K))**2+ZERO)
!				END DO
!				ALLELE_FREQUENCY(I,J)=(DOT_PRODUCT(WEIGHTS,DATA_VEC)/SUM(WEIGHTS))
!			END IF
!		END DO
!	END DO
!	
!	END SUBROUTINE INITIALIZE_ALLELE_FREQUENCY4
!	
!	
!	
!	
	SUBROUTINE INITIALIZE_ALLELE_FREQUENCY4_SQUARE(DATA_ALLELE1,DATA_TOTAL,&
				ALLELE_FREQUENCY,GRID_LENGTH,MAX_GRID_LENGTH)
	!INITIALIZES THE ALLELE FREQUENCY MATRIX BY TAKING WEIGHTED AVERAGES
	!OF THE LOCATIONS WITH DATA (SIMILAR TO INVERSE DISTANCE WEIGHTING)
	
	IMPLICIT NONE
	
	INTEGER :: I,J,K,MAX_GRID_LENGTH,NON_ZERO
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_ALLELE1,DATA_TOTAL
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY
	INTEGER, DIMENSION(2,GRID_LENGTH(1)*GRID_LENGTH(2)) :: DATA_LOC
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1)*GRID_LENGTH(2)) :: DATA_VEC,WEIGHTS
	REAL(KIND=DBLE_PREC) :: AVG_FREQ
		
	AVG_FREQ=(ZERO+SUM(DATA_ALLELE1))/SUM(DATA_TOTAL)
	
	!COUNT THE NUMBER OF NON_ZERO CELLS
	NON_ZERO=0
	DATA_LOC=0
	DATA_VEC=0.
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			IF(DATA_TOTAL(I,J)>0) THEN
				NON_ZERO=NON_ZERO+1
				DATA_VEC(NON_ZERO)=(DATA_ALLELE1(I,J)+ZERO)/DATA_TOTAL(I,J)
				DATA_LOC(1,NON_ZERO)=I
				DATA_LOC(2,NON_ZERO)=J
			END IF
		END DO
	END DO
	
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			IF(DATA_TOTAL(I,J)>0) THEN
				!INITIAL VALUE IS THE MLE FOR THE LOCATION
				ALLELE_FREQUENCY(I,J)=((DATA_ALLELE1(I,J)+ZERO)/DATA_TOTAL(I,J))
			ELSE
				!INITIAL VALUE IS THE VALUE OBTAINED FROM INVERSE DISTANCE WEIGHTING
				WEIGHTS=0
				DO K=1,NON_ZERO
					WEIGHTS(K)=ONE/SQRT((I-DATA_LOC(1,K))**2+(J-DATA_LOC(2,K))**2+ZERO)
				END DO
				ALLELE_FREQUENCY(I,J)=(DOT_PRODUCT(WEIGHTS,DATA_VEC)/SUM(WEIGHTS))
			END IF
		END DO
	END DO
	
	END SUBROUTINE INITIALIZE_ALLELE_FREQUENCY4_SQUARE
!	
!	
!
!	
!	
!	SUBROUTINE WRITE_ALLELE_SURFACE_SQUARE(ALLELE_FREQUENCY,GRID_COORD,&
!			DATA_COORD,DATA_MATRIX_LOCUS,SAMPLE_SITES,LOCUS,&
!			GRID_LENGTH,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!	!THIS SUBROUTINE WRITES OUT AN ALLELE FREQUENCY SURFACE,
!	!A FEW UNKNOWN HEAT MAPS, AND THE LOCATIONS OF ALL UNKNOWN INDIVIDUALS
!	!IN THREE SEPARATE FILES FOR PLOTTING IN R
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,NUMBER_UNKNOWN,MAX_GRID_LENGTH,LOCUS,SAMPLE_SITES
!	INTEGER :: MEMBERSHIP,OUTPUT_UNIT
!	INTEGER, DIMENSION(2) :: GRID_LENGTH
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY
!	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: DATA_COORD !(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(2,SAMPLE_SITES) :: DATA_MATRIX_LOCUS
!	REAL(KIND=DBLE_PREC) :: LONG,LAT,PROB
!	
!	!WRITES AN ALLELE FREQUENCY SURFACE
!!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","DataType","Long","Lat","AlleleFreq"
!	DO I=1,SAMPLE_SITES
!		PROB=(DATA_MATRIX_LOCUS(1,I)+ZERO)/SUM(DATA_MATRIX_LOCUS(:,I))
!		WRITE(OUTPUT_UNIT,'(I0,3X,A,3X,F10.4,3X,F10.4,3X,F10.4)') LOCUS, " Original ",DATA_COORD(1,I),&
!						DATA_COORD(2,I),PROB
!	END DO
!
!	DO I=1,GRID_LENGTH(1)
!		LONG=GRID_COORD(1,I)
!		DO J=1,GRID_LENGTH(2)
!			LAT=GRID_COORD(2,J)
!			PROB=ALLELE_FREQUENCY(I,J)
!			WRITE(OUTPUT_UNIT,'(I0,3X,A,3X,F10.4,3X,F10.4,3X,F10.4)') LOCUS," Interpolated ",LONG,LAT,PROB
!		END DO
!	END DO
!	
!	END SUBROUTINE WRITE_ALLELE_SURFACE_SQUARE
!	
!	
!	
!	SUBROUTINE WRITE_GENOME_FRACTIONS(GENOME_FRACTIONS,GRID_COORD,&
!			ADMIXED_MEMBERSHIP,DATA_COORD,SAMPLE_SITES,INDIVIDUAL,&
!			GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!	!THIS SUBROUTINE WRITES OUT AN ALLELE FREQUENCY SURFACE,
!	!A FEW UNKNOWN HEAT MAPS, AND THE LOCATIONS OF ALL UNKNOWN INDIVIDUALS
!	!IN THREE SEPARATE FILES FOR PLOTTING IN R
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,NUMBER_UNKNOWN,GRID_LENGTH,LOCUS,SAMPLE_SITES
!	INTEGER :: OUTPUT_UNIT,INDIVIDUAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: GENOME_FRACTIONS
!	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: DATA_COORD !(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(2,NUMBER_UNKNOWN) :: ADMIXED_MEMBERSHIP
!	REAL(KIND=DBLE_PREC) :: LONG,LAT,PROB
!	
!	PROB=ONE/TWO
!	LONG=DATA_COORD(1,ADMIXED_MEMBERSHIP(1,INDIVIDUAL))
!	LAT=DATA_COORD(2,ADMIXED_MEMBERSHIP(1,INDIVIDUAL))
!	WRITE(OUTPUT_UNIT,'(I0,3X,A,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F12.6)') INDIVIDUAL," True ",&
!					ADMIXED_MEMBERSHIP(1,INDIVIDUAL),ADMIXED_MEMBERSHIP(2,INDIVIDUAL),LONG,LAT,PROB
!	LONG=DATA_COORD(1,ADMIXED_MEMBERSHIP(2,INDIVIDUAL))
!	LAT=DATA_COORD(2,ADMIXED_MEMBERSHIP(2,INDIVIDUAL))
!	WRITE(OUTPUT_UNIT,'(I0,3X,A,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F12.6)') INDIVIDUAL," True ",&
!					ADMIXED_MEMBERSHIP(1,INDIVIDUAL),ADMIXED_MEMBERSHIP(2,INDIVIDUAL),LONG,LAT,PROB
!	
!	!WRITES AN GENOME_FRACTIONS SURFACE
!	DO I=1,GRID_LENGTH
!		LONG=GRID_COORD(1,I)
!		DO J=1,GRID_LENGTH
!			LAT=GRID_COORD(2,J)
!			PROB=GENOME_FRACTIONS(I,J)
!			WRITE(OUTPUT_UNIT,'(I0,3X,A,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F12.6)') INDIVIDUAL," Interpolated ",&
!					ADMIXED_MEMBERSHIP(1,INDIVIDUAL),ADMIXED_MEMBERSHIP(2,INDIVIDUAL),LONG,LAT,PROB
!		END DO
!	END DO
!	
!	END SUBROUTINE WRITE_GENOME_FRACTIONS
!	
!	
!	
!	
!	SUBROUTINE WRITE_GENOME_FRACTIONSG(GENOME_FRACTIONS,GRID_COORD,&
!			ADMIXED_MEMBERSHIP,DATA_COORD,SAMPLE_SITES,INDIVIDUAL,&
!			GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!	!THIS SUBROUTINE WRITES OUT AN ALLELE FREQUENCY SURFACE,
!	!A FEW UNKNOWN HEAT MAPS, AND THE LOCATIONS OF ALL UNKNOWN INDIVIDUALS
!	!IN THREE SEPARATE FILES FOR PLOTTING IN R
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,NUMBER_UNKNOWN,GRID_LENGTH,LOCUS,SAMPLE_SITES
!	INTEGER :: OUTPUT_UNIT,INDIVIDUAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: GENOME_FRACTIONS
!	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: DATA_COORD !(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(4,NUMBER_UNKNOWN) :: ADMIXED_MEMBERSHIP
!	REAL(KIND=DBLE_PREC) :: LONG,LAT,PROB
!	
!	PROB=ONE/4.
!	DO I=1,4
!		LONG=DATA_COORD(1,ADMIXED_MEMBERSHIP(I,INDIVIDUAL))
!		LAT=DATA_COORD(2,ADMIXED_MEMBERSHIP(I,INDIVIDUAL))
!		WRITE(OUTPUT_UNIT,'(I0,3X,A,3X,I0,3X,I0,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F12.6)') INDIVIDUAL," True ",&
!						ADMIXED_MEMBERSHIP(1,INDIVIDUAL),ADMIXED_MEMBERSHIP(2,INDIVIDUAL), &
!						ADMIXED_MEMBERSHIP(3,INDIVIDUAL),ADMIXED_MEMBERSHIP(4,INDIVIDUAL),LONG,LAT,PROB
!	END DO
!!	LONG=DATA_COORD(1,ADMIXED_MEMBERSHIP(2,INDIVIDUAL))
!!	LAT=DATA_COORD(2,ADMIXED_MEMBERSHIP(2,INDIVIDUAL))
!!	WRITE(OUTPUT_UNIT,'(I0,3X,A,3X,I0,3X,I0,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F12.6)') INDIVIDUAL," True ",&
!!					ADMIXED_MEMBERSHIP(1,INDIVIDUAL),ADMIXED_MEMBERSHIP(2,INDIVIDUAL),&
!!					ADMIXED_MEMBERSHIP(3,INDIVIDUAL),ADMIXED_MEMBERSHIP(4,INDIVIDUAL),LONG,LAT,PROB
!	
!	!WRITES AN GENOME_FRACTIONS SURFACE
!	DO I=1,GRID_LENGTH
!		LONG=GRID_COORD(1,I)
!		DO J=1,GRID_LENGTH
!			LAT=GRID_COORD(2,J)
!			PROB=GENOME_FRACTIONS(I,J)
!			WRITE(OUTPUT_UNIT,'(I0,3X,A,3X,I0,3X,I0,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F12.6)') INDIVIDUAL," Interpolated ",&
!					ADMIXED_MEMBERSHIP(1,INDIVIDUAL),ADMIXED_MEMBERSHIP(2,INDIVIDUAL),&
!					ADMIXED_MEMBERSHIP(3,INDIVIDUAL),ADMIXED_MEMBERSHIP(4,INDIVIDUAL),LONG,LAT,PROB
!		END DO
!	END DO
!	
!	END SUBROUTINE WRITE_GENOME_FRACTIONSG
!	
!	
!	
!	
!	SUBROUTINE WRITE_UNKNOWN_HEATMAP_SQUARE(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!			DATA_COORD,DATA_MATRIX_LOCUS,ADMIXED_MEMBERSHIP,SAMPLE_SITES,LOCUS,&
!			GRID_LENGTH,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!	!THIS SUBROUTINE WRITES OUT AN ALLELE FREQUENCY SURFACE,
!	!A FEW UNKNOWN HEAT MAPS, AND THE LOCATIONS OF ALL UNKNOWN INDIVIDUALS
!	!IN THREE SEPARATE FILES FOR PLOTTING IN R
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,NUMBER_UNKNOWN,MAX_GRID_LENGTH,LOCUS,SAMPLE_SITES
!	INTEGER :: MEMBERSHIP,OUTPUT_UNIT
!	INTEGER, DIMENSION(2) :: GRID_LENGTH
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN) :: UNKNOWN_GRID
!	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: DATA_COORD !(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(2,SAMPLE_SITES) :: DATA_MATRIX_LOCUS
!	INTEGER, DIMENSION(2,NUMBER_UNKNOWN) :: ADMIXED_MEMBERSHIP
!	REAL(KIND=DBLE_PREC) :: LONG,LAT,PROB
!	
!	!WRITES AN ALLELE FREQUENCY SURFACE
!!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","Individual","Long","Lat","Log(P)"
!
!	DO I=1,MIN(NUMBER_UNKNOWN,40)
!		DO J=1,GRID_LENGTH(1)
!			LONG=GRID_COORD(1,J)
!			DO K=1,GRID_LENGTH(2)
!				LAT=GRID_COORD(2,K)
!				PROB=UNKNOWN_GRID(J,K,I)
!				WRITE(OUTPUT_UNIT,'(I0,3X,I0,3X,F10.4,3X,F10.4,3X,F16.6)') LOCUS,I,LONG,LAT,PROB
!			END DO
!		END DO
!	END DO
!	
!	END SUBROUTINE WRITE_UNKNOWN_HEATMAP_SQUARE
!	
!	
!	
!	
!	SUBROUTINE WRITE_UNKNOWN_LOCATIONS_SQUARE(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!			DATA_COORD,DATA_MATRIX_LOCUS,ADMIXED_MEMBERSHIP,IS_LAND,SAMPLE_SITES,LOCUS,&
!			GRID_LENGTH,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,NUMBER_UNKNOWN,MAX_GRID_LENGTH,LOCUS,SAMPLE_SITES
!	INTEGER :: MEMBERSHIP,OUTPUT_UNIT
!	INTEGER, DIMENSION(2) :: GRID_LENGTH
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN) :: UNKNOWN_GRID
!	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: DATA_COORD !(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	LOGICAL, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: IS_LAND
!	INTEGER, DIMENSION(2,SAMPLE_SITES) :: DATA_MATRIX_LOCUS
!	INTEGER, DIMENSION(2,NUMBER_UNKNOWN) :: ADMIXED_MEMBERSHIP
!	REAL(KIND=DBLE_PREC) :: LONG,LAT,PROB
!	
!	DO I=1,NUMBER_UNKNOWN
!		CALL FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID(:,:,I),LONG,LAT,&
!					MAX_GRID_LENGTH,GRID_COORD,IS_LAND)
!		MEMBERSHIP=ADMIXED_MEMBERSHIP(1,I)
!		WRITE(OUTPUT_UNIT,'(I0,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F10.4)') I,LOCUS,MEMBERSHIP,LONG,LAT,PROB
!	END DO
!	
!	END SUBROUTINE WRITE_UNKNOWN_LOCATIONS_SQUARE
!	
!	
!	
!	SUBROUTINE WRITE_FIGURE_DATA(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!			DATA_COORD,DATA_MATRIX_LOCUS,ADMIXED_MEMBERSHIP,IS_LAND,SAMPLE_SITES,NUMBER_SNPS,&
!			GRID_LENGTH,NUMBER_UNKNOWN)
!	!THIS SUBROUTINE WRITES OUT AN ALLELE FREQUENCY SURFACE,
!	!A FEW UNKNOWN HEAT MAPS, AND THE LOCATIONS OF ALL UNKNOWN INDIVIDUALS
!	!IN THREE SEPARATE FILES FOR PLOTTING IN R
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,NUMBER_UNKNOWN,GRID_LENGTH,NUMBER_SNPS,SAMPLE_SITES,OUTPUT_UNIT=9
!	INTEGER :: MEMBERSHIP
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH,NUMBER_UNKNOWN) :: UNKNOWN_GRID
!	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: DATA_COORD !(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(2,SAMPLE_SITES) :: DATA_MATRIX_LOCUS
!	INTEGER, DIMENSION(2,NUMBER_UNKNOWN) :: ADMIXED_MEMBERSHIP
!	CHARACTER(LEN=800) :: TEXT_FILE,TEXT_END
!	REAL(KIND=DBLE_PREC) :: LONG,LAT,PROB
!	LOGICAL, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: IS_LAND
!	
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	
!	!WRITES AN ALLELE FREQUENCY SURFACE
!	TEXT_FILE="OneAlleleSurface"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","DataType","Long","Lat","AlleleFreq"
!	CALL WRITE_ALLELE_SURFACE(ALLELE_FREQUENCY,GRID_COORD,&
!			DATA_COORD,DATA_MATRIX_LOCUS,SAMPLE_SITES,NUMBER_SNPS,&
!			GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!	CLOSE(OUTPUT_UNIT)
!	
!	
!	!WRITES A FEW UNKNOWN HEAT MAPS
!	TEXT_FILE="UnknownHeatMaps"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	!TESTING HERE...CHANGE LATER
!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","Individual","Long","Lat","Log(P)"
!	CALL WRITE_UNKNOWN_HEATMAP(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!			DATA_COORD,DATA_MATRIX_LOCUS,ADMIXED_MEMBERSHIP,SAMPLE_SITES,NUMBER_SNPS,&
!			GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!	CLOSE(OUTPUT_UNIT)
!	
!	
!	!WRITES ALL OF THE UNKNOWN'S ESTIMATED LOCATIONS
!	TEXT_FILE="UnknownPlot"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A,3X,A)') "Individual","NumberSNPs","Membership","Long","Lat","Log(P)"
!	CALL WRITE_UNKNOWN_LOCATIONS(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!			DATA_COORD,DATA_MATRIX_LOCUS,ADMIXED_MEMBERSHIP,IS_LAND,SAMPLE_SITES,NUMBER_SNPS,&
!			GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!	CLOSE(OUTPUT_UNIT)
!	
!	END SUBROUTINE WRITE_FIGURE_DATA
!	
!	
!	
!	
!
!	
!	SUBROUTINE CUBIC_SOLVER(COEFF,SOLS,NUM_SOLS)
!		
!	IMPLICIT NONE
!	
!	REAL(KIND=DBLE_PREC), DIMENSION(4) :: COEFF
!	REAL(KIND=DBLE_PREC), DIMENSION(3) :: SOLS
!	INTEGER :: NUM_SOLS,I,J,K
!	COMPLEX(KIND=DBLE_PREC) :: U1,U2,U3,C1,C2,A,B,C,D,T0,T1,CVAR,S1,S2,S3
!	COMPLEX(KIND=DBLE_PREC), DIMENSION(3) :: U,S
!	
!	A=CMPLX(COEFF(1),0)
!	B=CMPLX(COEFF(2),0)
!	C=CMPLX(COEFF(3),0)
!	D=CMPLX(COEFF(4),0)
!	SOLS=0.
!	U(1)=CMPLX(1,0)
!	U(2)=CMPLX(-0.5,SQRT(3.)/2.)
!	U(3)=CMPLX(-0.5,-SQRT(3.)/2.)
!	
!	IF(REAL(A).EQ.0) THEN
!		S(1)=(-C+SQRT(C**2-4*B*D))/(2.*B)
!		S(2)=(-C-SQRT(C**2-4*B*D))/(2.*B)
!		S(3)=CMPLX(-20,-20)
!	ELSE
!		C1=18*A*B*C*D-4*(B**3)*D +(B**2)*(C**2)-4*A*(C**3)-27*(A**2)*(D**2)
!		T0=B**2-3*A*C
!		T1=2*(B**3)-9*A*B*C+27*(A**2)*D
!		CVAR=((T1-SQRT(-27*(A**2)*C1))/2.)**(1.0/3.0)
!		DO I=1,3
!			S(I)=(-1./(3.*A))*(B+U(I)*CVAR+T0/(U(I)*CVAR))
!		END DO
!	END IF
!	
!	SOLS=0
!	NUM_SOLS=0
!	DO I=1,3
!		IF(ABS(AIMAG(S(I))).LE.1.E-6) THEN
!			NUM_SOLS=NUM_SOLS+1
!			SOLS(NUM_SOLS)=REAL(S(I))
!		END IF
!	END DO
!	
!!	PRINT*, "NUM_SOLS: ", NUM_SOLS
!!	PRINT*, "C1: ", C1
!!	PRINT*, "COMPLEX SOLUTIONS: "
!!	PRINT*, S
!	
!	END SUBROUTINE CUBIC_SOLVER
!	
	
	
	SUBROUTINE CUBIC_SOLVER_MOD(COEFF,SOLS,NUM_SOLS)
	!SINCE OUR MODEL GUARANTEES 3 REAL SOLUTIONS WITH EXACTLY 1
	!BEING IN [0,1] WE HAVE MODIFIED THIS CUBIC SOLVER TO RETURN ALL
	!3 REAL SOLUTIONS
		
	IMPLICIT NONE
	
	REAL(KIND=DBLE_PREC), DIMENSION(4) :: COEFF
	REAL(KIND=DBLE_PREC), DIMENSION(3) :: SOLS
	INTEGER :: NUM_SOLS,I,J,K
	COMPLEX(KIND=DBLE_PREC) :: U1,U2,U3,C1,C2,A,B,C,D,T0,T1,CVAR,S1,S2,S3
	COMPLEX(KIND=DBLE_PREC), DIMENSION(3) :: U,S
	
	A=CMPLX(COEFF(1),0)
	B=CMPLX(COEFF(2),0)
	C=CMPLX(COEFF(3),0)
	D=CMPLX(COEFF(4),0)
	SOLS=0.
	U(1)=CMPLX(1,0)
	U(2)=CMPLX(-0.5,SQRT(3.)/2.)
	U(3)=CMPLX(-0.5,-SQRT(3.)/2.)
	
	IF(REAL(A).EQ.0) THEN
		S(1)=(-C+SQRT(C**2-4*B*D))/(2.*B)
		S(2)=(-C-SQRT(C**2-4*B*D))/(2.*B)
		S(3)=CMPLX(-20,-20)
	ELSE
		C1=18*A*B*C*D-4*(B**3)*D +(B**2)*(C**2)-4*A*(C**3)-27*(A**2)*(D**2)
		T0=B**2-3*A*C
		T1=2*(B**3)-9*A*B*C+27*(A**2)*D
		CVAR=((T1-SQRT(-27*(A**2)*C1))/2.)**(1.0/3.0)
		DO I=1,3
			S(I)=(-1./(3.*A))*(B+U(I)*CVAR+T0/(U(I)*CVAR))
		END DO
	END IF
	
	SOLS=0
	NUM_SOLS=3
	DO I=1,3
		SOLS(I)=REAL(S(I))
	END DO
	
!	PRINT*, "NUM_SOLS: ", NUM_SOLS
!	PRINT*, "C1: ", C1
!	PRINT*, "COMPLEX SOLUTIONS: "
!	PRINT*, S
	
	END SUBROUTINE CUBIC_SOLVER_MOD
	
	
	
	
	SUBROUTINE UPDATE_L2_PARS_ONCE_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
	!THIS SUBROUTINE UPDATES THE ALLELE_FREQUENCY MATRIX ONCE USING MM UPDATES
	
	IMPLICIT NONE
	
	INTEGER :: I,J,K,MAX_GRID_LENGTH,NUM_SOLS,TEMP_SOLS
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_ALLELE1,DATA_TOTAL
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY, TEMP_FREQUENCY
	REAL(KIND=DBLE_PREC) :: RHO,DIAGWEIGHT,LONG_WEIGHT,LAT_WEIGHT,U,V,R1
	REAL(KIND=DBLE_PREC), DIMENSION(3,3) :: MINI_WEIGHTS,MINI_FREQUENCY
	REAL(KIND=DBLE_PREC), DIMENSION(4) :: COEFF
	REAL(KIND=DBLE_PREC), DIMENSION(3) :: SOLS
	
	DIAGWEIGHT=ONE/SQRT((ONE/LONG_WEIGHT)**2+(ONE/LAT_WEIGHT)**2)
	TEMP_FREQUENCY=ALLELE_FREQUENCY
	
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			MINI_WEIGHTS=0
			MINI_FREQUENCY=0
			MINI_FREQUENCY(2,2)=TEMP_FREQUENCY(I,J)
			IF(I>1) THEN
!				PRINT*, I,J,ALLELE_FREQUENCY(I-1,J)
				MINI_FREQUENCY(1,2)=TEMP_FREQUENCY(I-1,J)
				MINI_WEIGHTS(1,2)=LONG_WEIGHT
				IF(J>1) THEN
					MINI_FREQUENCY(1,1)=TEMP_FREQUENCY(I-1,J-1)
					MINI_WEIGHTS(1,1)=DIAGWEIGHT
				END IF
				IF(J<GRID_LENGTH(2)) THEN
					MINI_FREQUENCY(1,3)=TEMP_FREQUENCY(I-1,J+1)
					MINI_WEIGHTS(1,3)=DIAGWEIGHT
				END IF
			END IF
			IF(I<GRID_LENGTH(1)) THEN
				MINI_FREQUENCY(3,2)=TEMP_FREQUENCY(I+1,J)
				MINI_WEIGHTS(3,2)=LONG_WEIGHT
				IF(J>1) THEN
					MINI_FREQUENCY(3,1)=TEMP_FREQUENCY(I+1,J-1)
					MINI_WEIGHTS(3,1)=DIAGWEIGHT
				END IF
				IF(J<GRID_LENGTH(2)) THEN
					MINI_FREQUENCY(3,3)=TEMP_FREQUENCY(I+1,J+1)
					MINI_WEIGHTS(3,3)=DIAGWEIGHT
				END IF
			END IF
			IF(J>1) THEN
				MINI_FREQUENCY(2,1)=TEMP_FREQUENCY(I,J-1)
				MINI_WEIGHTS(2,1)=LAT_WEIGHT
			END IF
			IF(J<GRID_LENGTH(2)) THEN
				MINI_FREQUENCY(2,3)=TEMP_FREQUENCY(I,J+1)
				MINI_WEIGHTS(2,3)=LAT_WEIGHT
			END IF
			
			
			IF(IEEE_IS_NAN(SUM(MINI_WEIGHTS)).OR.IEEE_IS_NAN(SUM(MINI_FREQUENCY))) THEN
!				PRINT*, "MINI_WEIGHTS: "
!				DO K=1,SIZE(MINI_WEIGHTS(1,:))
!					PRINT*, MINI_WEIGHTS(:,K)
!				END DO
!				PRINT*, "MINI_FREQUENCY: "
!				DO K=1,SIZE(MINI_FREQUENCY(1,:))
!					PRINT*, MINI_FREQUENCY(:,K)
!				END DO
!				PRINT*, "RHO: ", RHO
!				PRINT*, "DATA_ALLELE1(I,J): ", DATA_ALLELE1(I,J)
!				PRINT*, "DATA_TOTAL(I,J): ", DATA_TOTAL(I,J)
!				STOP
				CALL INTPR("SOMETHING WENT WRONG...NAN WEIGHT SUM",-1,1,0)
				CALL INTPR("RESULTS ARE SPURIOUS",-1,1,0)
			END IF
			
			U=2*RHO*SUM(MINI_WEIGHTS)
			V=2*RHO*TEMP_FREQUENCY(I,J)*SUM(MINI_WEIGHTS*MINI_FREQUENCY)
			COEFF(1)=U
			COEFF(2)=-U
			!PSEUDO_COUNTS ADDED HERE...
			IF(DATA_TOTAL(I,J)>0) THEN
				COEFF(3)=-DATA_TOTAL(I,J)-2*PSEUDO_COUNT-V
				COEFF(4)=DATA_ALLELE1(I,J)+PSEUDO_COUNT+V
			ELSE
				COEFF(3)=-V
				COEFF(4)=V
			END IF
			SOLS=0
			NUM_SOLS=0
!			PRINT*, "MINI WEIGHTS:"
!			CALL PRINT_MATRIX(MINI_WEIGHTS)
!			PRINT*, "MINI FREQUENCY:"
!			CALL PRINT_MATRIX(MINI_FREQUENCY)
!			PRINT*, "U,V",U,V
			CALL CUBIC_SOLVER_MOD(COEFF,SOLS,NUM_SOLS)
!			PRINT*, "SOLS: ", SOLS(1:3)
!			STOP
			!FIGURE OUT HOW MANY SOLUTIONS CAME OUT OF THE SOLVER AND DEAL ACCORDINGLY
			!IF 1, THEN TAKE IT
!			PRINT*, "NUM_SOLS: ", NUM_SOLS
			TEMP_SOLS=MINLOC(ABS(0.5-SOLS),1)
			R1=SOLS(TEMP_SOLS)
			IF(R1.GE.1) THEN
				ALLELE_FREQUENCY(I,J)=(ALLELE_FREQUENCY(I,J)+1.)/2.
			ELSE IF(R1.LE.0) THEN
				ALLELE_FREQUENCY(I,J)=(ALLELE_FREQUENCY(I,J))/2.
			ELSE
				ALLELE_FREQUENCY(I,J)=R1
			END IF
		END DO
	END DO
		
	END SUBROUTINE UPDATE_L2_PARS_ONCE_SQUARE
	
	
	
	
	
	FUNCTION CALC_LOGLIK_L2_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
		LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
		
	IMPLICIT NONE

	INTEGER :: I,J,K,MAX_GRID_LENGTH
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_ALLELE1,DATA_TOTAL
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY
	REAL(KIND=DBLE_PREC) :: RHO,CALC_LOGLIK_L2_SQUARE,PENALTY,LONG_WEIGHT,LAT_WEIGHT,DIAG_WEIGHT

	CALC_LOGLIK_L2_SQUARE=0.
	PENALTY=0.
	DIAG_WEIGHT=ONE/SQRT((ONE/LONG_WEIGHT)**2+(ONE/LAT_WEIGHT)**2)
	
	!Binomial model
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			IF(DATA_TOTAL(I,J)>0) THEN
				IF(ALLELE_FREQUENCY(I,J)>0 .AND. ALLELE_FREQUENCY(I,J)<1) THEN
					CALC_LOGLIK_L2_SQUARE=CALC_LOGLIK_L2_SQUARE+(DATA_ALLELE1(I,J)+PSEUDO_COUNT)*LOG(ALLELE_FREQUENCY(I,J))&
						+(DATA_TOTAL(I,J)+PSEUDO_COUNT-DATA_ALLELE1(I,J))*LOG(1.-ALLELE_FREQUENCY(I,J))
				END IF
			END IF
		END DO
	END DO
!	PRINT*, "CALC_LOGLIK_GENO: ", CALC_LOGLIK_GENO
	
	!penalty
	DO J=1,GRID_LENGTH(2)-1
		DO I=1,GRID_LENGTH(1)
			!BOTTOM
			PENALTY=PENALTY+LAT_WEIGHT*(ALLELE_FREQUENCY(I,J)-ALLELE_FREQUENCY(I,J+1))**2
!			PRINT*, "PENALTY: ", PENALTY, I,J,ALLELE_FREQUENCY(I,J),ALLELE_FREQUENCY(I,J+1)
			
			!BOTTOM LEFT
			IF(I>1) THEN
				PENALTY=PENALTY+DIAG_WEIGHT*(ALLELE_FREQUENCY(I,J)-ALLELE_FREQUENCY(I-1,J+1))**2
			END IF
			
			!RIGHT(PARTIAL) AND BOTTOM RIGHT
			IF(I<GRID_LENGTH(1)) THEN
			!BOTTOM RIGHT
				PENALTY=PENALTY+DIAG_WEIGHT*(ALLELE_FREQUENCY(I,J)-ALLELE_FREQUENCY(I+1,J+1))**2
			!RIGHT SIDE !THIS IS INCOMPLETE... MISSING WHOLE BOTTOM ROW
				PENALTY=PENALTY+LONG_WEIGHT*(ALLELE_FREQUENCY(I,J)-ALLELE_FREQUENCY(I+1,J))**2
			END IF
		END DO
	END DO
!	PRINT*, "PENALTY: ", PENALTY
	
	!FINISHING RIGHT SIDE PENALTY(BOTTOM ROW)
	DO I=1,GRID_LENGTH(1)-1
		PENALTY=PENALTY+LONG_WEIGHT*(ALLELE_FREQUENCY(I,GRID_LENGTH(2))-ALLELE_FREQUENCY(I+1,GRID_LENGTH(2)))**2
	END DO
	
	CALC_LOGLIK_L2_SQUARE=CALC_LOGLIK_L2_SQUARE-RHO*PENALTY
	
	IF(IEEE_IS_NAN(CALC_LOGLIK_L2_SQUARE)) THEN
!		PRINT*, "PENALTY ", PENALTY
!		STOP
		CALL INTPR("NAN LOGLIK FOUND",-1,1,0)
		CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
	END IF

	END FUNCTION CALC_LOGLIK_L2_SQUARE
	
	
	

	
	
	SUBROUTINE FIT_L2_MODEL_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,INIT_BOOL)
	!THIS SUBROUTINE FITS THE MODEL...IT UPDATES THE MATRIX UNTIL THE CONVERGENCE
	!CRITERIA IS MET
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,MAX_GRID_LENGTH,ITERATIONS
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_ALLELE1,DATA_TOTAL
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY,OLD_ALLELE_FREQ
	REAL(KIND=DBLE_PREC) :: RHO
	LOGICAL :: NOT_CONVERGED,INIT_BOOL
	REAL(KIND=DBLE_PREC) :: LOGLIK,NEW_LOGLIK,LONG_WEIGHT,LAT_WEIGHT
	
	
!	PRINT*, "BEGIN FIT MODEL"
	IF(INIT_BOOL) THEN
	!	CALL INITIALIZE_ALLELE_FREQUENCY2(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,GRID_LENGTH)
		CALL INITIALIZE_ALLELE_FREQUENCY_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,GRID_LENGTH,MAX_GRID_LENGTH)
	END IF
	OLD_ALLELE_FREQ=ALLELE_FREQUENCY
	
!	PRINT*, "ALLELE FREQUENCY: ", ALLELE_FREQUENCY(1:5,1:5)
	
	LOGLIK=CALC_LOGLIK_L2_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!	PRINT*, "LOGLIK: ",  LOGLIK
	
	NOT_CONVERGED=.TRUE.
	ITERATIONS=0
	DO WHILE(NOT_CONVERGED)
		ITERATIONS=ITERATIONS+1
		CALL UPDATE_L2_PARS_ONCE_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
		NEW_LOGLIK=CALC_LOGLIK_L2_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
		IF(IEEE_IS_NAN(SUM(ALLELE_FREQUENCY))) THEN
!			PRINT*, "ALLELE_FREQUENCY IS NAN"
!			STOP
			CALL INTPR("NAN ALLELE FREQUENCY",-1,1,0)
			CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
		END IF
		
		!MODEL CHECK
!		IF(NEW_LOGLIK-LOGLIK<-1.E-7) THEN
!			PRINT*, "DECREASE IN LOGLIK(IT,DIFF,LOGLIK,NEW_LOGLIK): ", ITERATIONS, (NEW_LOGLIK-LOGLIK), LOGLIK, NEW_LOGLIK
!		END IF
		
		!CHECK CONVERGENCE
!		CALL CHECK_CONVERGENCE(LOGLIK,NEW_LOGLIK,ITERATIONS,NOT_CONVERGED)
		CALL CHECK_CONVERGENCE_MATRIX(OLD_ALLELE_FREQ,ALLELE_FREQUENCY,ITERATIONS,NOT_CONVERGED)
		
!		IF((MOD(ITERATIONS,5000).EQ.0).OR.(.NOT.NOT_CONVERGED)) THEN
!			PRINT*, "ITERATIONS: ", ITERATIONS, "LOGLIK(DIFF,LOGLIK): ", (NEW_LOGLIK-LOGLIK), NEW_LOGLIK
!		END IF
		
		LOGLIK=NEW_LOGLIK
		OLD_ALLELE_FREQ=ALLELE_FREQUENCY
	END DO
	
!	PRINT*, "END FIT MODEL"
	
	END SUBROUTINE FIT_L2_MODEL_SQUARE
	
	
	
	
	
	SUBROUTINE FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
	!THIS SUBROUTINE FITS THE MODEL...IT UPDATES THE MATRIX UNTIL THE CONVERGENCE
	!CRITERIA IS MET
		
	IMPLICIT NONE
	
	INTEGER :: I,J,K,MAX_GRID_LENGTH,ITERATIONS,QSEC=6,ERROR_INT,VEC_LENGTH
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_ALLELE1,DATA_TOTAL
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY,OLD_ALLELE_FREQ
	REAL(KIND=DBLE_PREC) :: RHO,LONG_WEIGHT,LAT_WEIGHT
	LOGICAL :: NOT_CONVERGED
	REAL(KIND=DBLE_PREC) :: LOGLIK,NEW_LOGLIK
	!QUASI NEWTON ELEMENTS
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1)*GRID_LENGTH(2)) :: TEMP_VEC,TEMP_F,TEMP_F2,TEMP_NEW
	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: MAP_U,MAP_V
	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: TEMP_MAT,TEMPINV_MAT
	
!	PRINT*, "RHO: ",RHO
!!	PRINT*, "LONG_WEIGHT: ", LONG_WEIGHT
!!	PRINT*, "LAT_WEIGHT: ", LAT_WEIGHT
!!	PRINT*, "GRID_LENGTH: ", GRID_LENGTH
!	
!	PRINT*, "DATA_ALLELE1"
!	CALL PRINT_INT_MATRIX(DATA_ALLELE1(1:8,1:8))
!	
!	PRINT*, "DATA_TOTAL"
!	CALL PRINT_INT_MATRIX(DATA_TOTAL(1:8,1:8))
!	
!	PRINT*, "ALLELE_FREQUENCY"
!	CALL PRINT_MATRIX(ALLELE_FREQUENCY(1:8,1:8))
	
	QSEC=6
	VEC_LENGTH=GRID_LENGTH(1)*GRID_LENGTH(2)
	ALLOCATE(MAP_U(VEC_LENGTH,QSEC),MAP_V(VEC_LENGTH,QSEC))
	ALLOCATE(TEMP_MAT(QSEC,QSEC),TEMPINV_MAT(QSEC,QSEC))
!	PRINT*, "BEGIN FIT MODEL"
!	CALL INITIALIZE_ALLELE_FREQUENCY(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,GRID_LENGTH)
	CALL INITIALIZE_ALLELE_FREQUENCY_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,&
		GRID_LENGTH,MAX_GRID_LENGTH)
	OLD_ALLELE_FREQ=ALLELE_FREQUENCY
	
	!HERE WE PERFORM NORMAL MM UPDATES TO STABALIZE THE QUASI-NEWTON UPDATES
	DO I=1,20
		CALL UPDATE_L2_PARS_ONCE_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
	END DO
	
	!CHECK CONVERGENCE FIRST JUST IN CASE...
!	LOGLIK=CALC_LOGLIK_GENO(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,GRID_LENGTH)
	OLD_ALLELE_FREQ=ALLELE_FREQUENCY
	CALL UPDATE_L2_PARS_ONCE_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!	NEW_LOGLIK=CALC_LOGLIK_GENO(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,GRID_LENGTH)
	NOT_CONVERGED=.TRUE.
	!CALL CHECK_CONVERGENCE(LOGLIK,NEW_LOGLIK,ITERATIONS,NOT_CONVERGED)
	!DON'T CHECK CONVERGENCE ANYMORE HERE SINCE IT MOVES VERY SLOW...
!	CALL CHECK_CONVERGENCE_MATRIX(OLD_ALLELE_FREQ,ALLELE_FREQUENCY,ITERATIONS,NOT_CONVERGED)
	
	TEMP_VEC=RESHAPE(ALLELE_FREQUENCY,SHAPE(TEMP_VEC))
	DO I=1,QSEC+1
		CALL UPDATE_L2_PARS_ONCE_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
		TEMP_F=RESHAPE(ALLELE_FREQUENCY,SHAPE(TEMP_VEC))
		IF(I.LE.QSEC) THEN
			MAP_U(:,I)=TEMP_F-TEMP_VEC
		END IF
		IF(I.GE.2) THEN
			MAP_V(:,I-1)=TEMP_F-TEMP_VEC
		END IF
		TEMP_VEC=TEMP_F
	END DO
	
	ITERATIONS=0
	DO WHILE(NOT_CONVERGED)
		ITERATIONS=ITERATIONS+1
		
		TEMP_VEC=RESHAPE(ALLELE_FREQUENCY,SHAPE(TEMP_VEC))
		CALL UPDATE_L2_PARS_ONCE_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
		TEMP_F=RESHAPE(ALLELE_FREQUENCY,SHAPE(TEMP_VEC))
		CALL UPDATE_L2_PARS_ONCE_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
		TEMP_F2=RESHAPE(ALLELE_FREQUENCY,SHAPE(TEMP_VEC))
		
		CALL SHIFT_MAT_LEFT(MAP_U,VEC_LENGTH,QSEC)
		MAP_U(:,QSEC)=TEMP_F-TEMP_VEC
		CALL SHIFT_MAT_LEFT(MAP_V,VEC_LENGTH,QSEC)
		MAP_V(:,QSEC)=TEMP_F2-TEMP_F

		TEMP_MAT=FIRST_MATMULT(MAP_U,MAP_V,VEC_LENGTH,QSEC)

		IF(IEEE_IS_NAN(SUM(TEMP_MAT))) THEN
			ERROR_INT=-1
		ELSE
			CALL FINDINV(TEMP_MAT,TEMPINV_MAT,QSEC,ERROR_INT)
		END IF
		IF(IEEE_IS_NAN(SUM(TEMPINV_MAT))) THEN
			ERROR_INT=-1
		END IF
	
		IF (ERROR_INT.LE.-1) THEN
			TEMP_NEW=TEMP_F2
			!CHOOSING TO BAIL ON THE QNEWT UPDATES IF SOMETHING GOES WRONG...
!			PRINT*, "FOUND NANS IN QNEWT UPDATE... CONTINUING WITH NORMAL UPDATES"
			CALL INTPR("NANS FOUND IN QNEWT UPDATE...CONTINUING WITH NORMAL UPDATES",-1,1,0)
			CALL FIT_L2_MODEL_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,.FALSE.)
			RETURN
		ELSE
			TEMP_NEW=TEMP_F-MATMUL(MAP_V,MATMUL(TEMPINV_MAT,MATMUL(TRANSPOSE(MAP_U),TEMP_VEC-TEMP_F)))
			IF(IEEE_IS_NAN(SUM(TEMP_NEW))) THEN
!				PRINT*, "NAN FOUND IN UPDATE PARS QNEWT"
!				PRINT*, "TEMP_NEW"
!				PRINT*, TEMP_NEW
!				PRINT*, "MAP_V"
!				PRINT*, MAP_V
!				PRINT*, "MAP_U"
!				PRINT*, MAP_U
!				PRINT*, "TEMP_MAT"
!				PRINT*, TEMP_MAT
!				PRINT*, "TEMPINV_MAT"
!				PRINT*, TEMPINV_MAT
!				PRINT*, "TEMP_VEC"
!				PRINT*, TEMP_VEC
!				PRINT*, "TEMP_F"
!				PRINT*, TEMP_F
!				PRINT*, "NAN FOUND IN UPDATE PARS QNEWT...UPDATED PARS NORMALLY"
				CALL INTPR("NAN FOUND IN UPDATE PARS QNEWT...UPDATED PARS NORMALLY",-1,1,0)
				TEMP_NEW=TEMP_F2
			END IF
			
			!THIS ENSURES THAT OUR QUASI NEWTON UPDATES DO NOT MEET OR PASS OUR 
			!BOUNDARIES OF (0,1)
			DO I=1,VEC_LENGTH
				IF(TEMP_NEW(I).LE.0) THEN
					TEMP_NEW(I)=TEMP_VEC(I)/2.
				ELSE IF(TEMP_NEW(I).GE.1.) THEN
					TEMP_NEW(I)=(TEMP_VEC(I)+1.)/2.
				END IF
			END DO
!			DO I=1,VEC_LENGTH
!				IF(TEMP_NEW(I).LE.0) THEN
!					TEMP_NEW(I)=EPS
!				ELSE IF(TEMP_NEW(I).GE.1.) THEN
!					TEMP_NEW(I)=1.-EPS
!				END IF
!			END DO
			
			ALLELE_FREQUENCY=RESHAPE(TEMP_NEW,SHAPE(ALLELE_FREQUENCY))
		END IF
		
		!CHECKS FOR CONVERGENCE
		IF((ITERATIONS.GE.20)) THEN
			OLD_ALLELE_FREQ=ALLELE_FREQUENCY
			CALL UPDATE_L2_PARS_ONCE_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
			CALL CHECK_CONVERGENCE_MATRIX(OLD_ALLELE_FREQ,ALLELE_FREQUENCY,ITERATIONS,NOT_CONVERGED)
		END IF
		
		IF((MOD(ITERATIONS,5000).EQ.0).OR.(.NOT.NOT_CONVERGED)) THEN
			LOGLIK=CALC_LOGLIK_L2_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
						LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!			PRINT*, "QNEWT ITERATIONS: ", ITERATIONS, "LOGLIK: ", LOGLIK
		END IF
		
!		DO I=1,GRID_LENGTH
!			DO J=1,GRID_LENGTH
!				IF(ALLELE_FREQUENCY(J,I).LE.0) THEN
!					ALLELE_FREQUENCY(J,I)=EPS
!!					PRINT*, "CHANGED 2"
!				ELSE IF(ALLELE_FREQUENCY(J,I).GE.1.) THEN
!					ALLELE_FREQUENCY(J,I)=1.-EPS
!!					PRINT*, "CHANGED 2"
!				END IF
!			END DO
!		END DO
	END DO
	LOGLIK=CALC_LOGLIK_L2_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!	IF(ITERATIONS.EQ.0) THEN
!		PRINT*, "QNEWT ITERATIONS: ", ITERATIONS, "LOGLIK: ", LOGLIK
!	END IF
	
	!HERE WE PERFORM ADDITIONAL NORMAL MM UPDATES TO FINISH OFF...
	DO I=1,20
		CALL UPDATE_L2_PARS_ONCE_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
	END DO
	
	END SUBROUTINE FIT_L2_MODEL_QNEWT_SQUARE
	

	
	!
!	
!	SUBROUTINE UPDATE_IS_LAND_SQUARE(IS_LAND,GRID_LENGTH,MAX_GRID_LENGTH)
!	!UPDATES THE IS_LAND VECTOR WITH VALUES FROM A FILE WHICH TELL WHETHER
!	!THE ASSOCIATED VALUE IS LAND(1) OR WATER(0)
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,MAX_GRID_LENGTH,INPUT_UNIT=21
!	INTEGER, DIMENSION(2) ::GRID_LENGTH
!	INTEGER, DIMENSION(MAX_GRID_LENGTH) :: INT_VEC
!	LOGICAL, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: IS_LAND
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	
!	WRITE(TEXT_END,'(I0)') MAX_GRID_LENGTH
!	TEXT_FILE="GridCoordSquare"//TRIM(TEXT_END)//"Water.txt"
!	OPEN(INPUT_UNIT,FILE=TEXT_FILE)
!	
!	PRINT*, "GRID_LENGTH", GRID_LENGTH(1),GRID_LENGTH(2)
!	IS_LAND=.FALSE.
!!	DO I=1,GRID_LENGTH(2)
!!		READ(INPUT_UNIT,*) (INT_VEC(J),J=1,GRID_LENGTH(1))
!!		DO J=1,GRID_LENGTH(1)
!!			IF(INT_VEC(J).EQ.1) THEN
!!				IS_LAND(J,I)=.TRUE.
!!			END IF
!!		END DO
!!!		PRINT*, "INT_VEC", INT_VEC(1:10)
!!		PRINT*, "IS_LAND", IS_LAND(1:10,I)
!!!		PRINT*, INT_VEC(1:10)
!!!		PRINT*, IS_LAND(1:10,I)
!!	END DO
!!	IS_LAND=TRANSPOSE(IS_LAND)
!!	GRID_LENGTH IS 50 30
!
!	DO I=1,GRID_LENGTH(2)
!		READ(INPUT_UNIT,*) (INT_VEC(J),J=1,GRID_LENGTH(1))
!		DO J=1,GRID_LENGTH(1)
!			IF(INT_VEC(J).EQ.1) THEN
!				IS_LAND(J,GRID_LENGTH(2)+1-I)=.TRUE.
!			END IF
!		END DO
!!		PRINT*, INT_VEC(1:10)
!!		PRINT*, IS_LAND(1:10,I)
!	END DO
!	
!	!IS_LAND=TRANSPOSE(IS_LAND)
!	
!	CLOSE(INPUT_UNIT)
!	
!	END SUBROUTINE UPDATE_IS_LAND_SQUARE
!	
!	
	
	
!	
!	
!	SUBROUTINE READ_FIT_SQUARE(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO)
!	!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,MAX_GRID_LENGTH,OUTPUT_UNIT=11,TEMP_INT
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE, population, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: ALLELE_FREQUENCY,QNEWT_FREQ !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,LONG_WEIGHT,LAT_WEIGHT,TEMP_REAL
!	INTEGER, DIMENSION(2) :: GRID_LENGTH
!	REAL(KIND=DBLE_PREC), DIMENSION(NUMBER_SNPS) :: MEAN_VARIANCES
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: POPULATION_VARIANCES
!	
!!	PRINT*, "READ AND FIT SQUARE"
!	
!	GRID_LENGTH=0
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	ALLOCATE(POPULATION_VARIANCES(SAMPLE_SITES))
!	POPULATION_VARIANCES=0.
!	
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD_SQUARE(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!	ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
!	ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)),QNEWT_FREQ(GRID_LENGTH(1),GRID_LENGTH(2)))
!!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
!	LONG_WEIGHT=ONE
!	LAT_WEIGHT=ONE
!!	LONG_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))
!!	LAT_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	ALLELE_FREQUENCY=ZERO
!	QNEWT_FREQ=ZERO
!	
!!	!!this short code writes out the grid_coord to file...
!!	WRITE(TEXT_END,'(I0)') MAX_GRID_LENGTH
!!	TEXT_FILE="GridCoordSquare"//TRIM(TEXT_END)//".txt"
!!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!!	CALL WRITE_MATRIX(GRID_COORD,OUTPUT_UNIT)
!!	CLOSE(OUTPUT_UNIT)
!	
!	MEAN_VARIANCES=0
!	
!	DO I=1,NUMBER_SNPS
!!		PRINT*, "READ AND FIT-SNP: ", I
!		CALL INTPR("READ AND FIT SNP",-1,I,1)
!		CALL POOL_GENO_DATA_SQUARE(DATA_MATRIX(:,:,I),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,&
!				GRID_COORD,SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
!!		CALL FIT_L2_MODEL_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!		
!		CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,QNEWT_FREQ,RHO,&
!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!!		PRINT*, "MAX DIFF BETWEEN QNEWT AND MM: ", MAXVAL(ABS(ALLELE_FREQUENCY-QNEWT_FREQ))
!		TEMP_INT=0
!		DO K=1,GRID_LENGTH(2)
!			DO J=1,GRID_LENGTH(1)
!				TEMP_REAL=DATA_ALLELE1(J,K)/(QNEWT_FREQ(J,K)**2)+&
!					(DATA_TOTAL(J,K)-DATA_ALLELE1(J,K))/(1.-QNEWT_FREQ(J,K))**2
!				IF(TEMP_REAL>0) THEN
!					TEMP_INT=TEMP_INT+1
!					POPULATION_VARIANCES(TEMP_INT)=1./(TEMP_REAL+8*RHO*(1.+1./ROOT2))
!				END IF
!!				MEAN_VARIANCES(I)=MEAN_VARIANCES(I)+1./(TEMP_REAL+8*RHO*(1.+1./ROOT2))
!			END DO
!		END DO
!		MEAN_VARIANCES(I)=SUM(POPULATION_VARIANCES)/(MAX(1,SIZE(POPULATION_VARIANCES)))
!!		PRINT*, POPULATION_VARIANCES
!		
!	END DO
!	
!	!
!!	!!this short code writes out the grid_coord to file...
!!	WRITE(TEXT_END,'(I0)') MAX_GRID_LENGTH
!!	TEXT_FILE="MeanVariancesRho10Length"//TRIM(TEXT_END)//".txt"
!!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!!	DO I=1,NUMBER_SNPS
!!		PRINT*, "MEAN VARIANCES: ", I, " ", MEAN_VARIANCES(I)
!!		WRITE(OUTPUT_UNIT,*) MEAN_VARIANCES(I)
!!	END DO
!!	TEMP_REAL=SUM(MEAN_VARIANCES)/(MAX(1,SIZE(MEAN_VARIANCES)))
!!	PRINT*, "MEAN OF MEAN VARIANCES: ", TEMP_REAL
!!	WRITE(OUTPUT_UNIT,*) "MEAN OF MEAN VARIANCES: ", TEMP_REAL
!!	CLOSE(OUTPUT_UNIT)
!	
!	
!	END SUBROUTINE READ_FIT_SQUARE
!	
!	
	
	
	!
!	SUBROUTINE READ_FIT_FIND_SQUARE(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO,AVG_DISTANCE)
!	!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT=12
!	INTEGER :: OUTPUT_UNIT2=13,OUTPUT_UNIT3=14,TEMP_INT,OUTPUT_UNIT4=15
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !DATA_COORD(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: ALLELE_FREQUENCY !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP
!	LOGICAL, ALLOCATABLE, DIMENSION(:,:) :: IS_LAND
!	INTEGER, DIMENSION(2) :: GRID_LENGTH
!	
!	PRINT*, "BEGIN READ_FIT_FIND...RHO: ",RHO
!	
!	GRID_LENGTH=0
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD_SQUARE(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!	ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
!	ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)))
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN))
!	ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!	UNKNOWN_OBS=0
!	UNKNOWN_GRID=1.
!	CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	ALLOCATE(IS_LAND(GRID_LENGTH(1),GRID_LENGTH(2)))
!	CALL UPDATE_IS_LAND_SQUARE(IS_LAND,GRID_LENGTH,MAX_GRID_LENGTH)
!
!	LONG_WEIGHT=ONE
!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	LAT_WEIGHT=ONE
!	
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	TEXT_FILE="AlleleSurfacesSquare"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","DataType","Long","Lat","AlleleFreq"
!	
!	TEXT_FILE="UnknownHeatMapsSquare"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT2,FILE=TEXT_FILE)
!	!TESTING HERE...CHANGE LATER
!	WRITE(OUTPUT_UNIT2,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","Individual","Long","Lat","Log(P)"
!	
!!	TEXT_FILE="UnknownPlot"//TRIM(TEXT_END)//".txt"
!!	OPEN(OUTPUT_UNIT3,FILE=TEXT_FILE)
!!	WRITE(OUTPUT_UNIT3,'(A,3X,A,3X,A,3X,A,3X,A,3X,A)') "Individual","NumberSNPs","Membership","Long","Lat","Log(P)"
!	
!	TEXT_FILE="AvgDistancesSquare"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT3,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT3,'(A,3X,A,3X,A)') "Loci","Method","AverageDistance"
!	
!!	TEXT_FILE="IsLandTest"//TRIM(TEXT_END)//".txt"
!!	OPEN(OUTPUT_UNIT4,FILE=TEXT_FILE)
!!!	DO I=1,MAX_GRID_LENGTH
!!!		WRITE(OUTPUT_UNIT4,*) GRID_COORD(:,I)
!!!	END DO
!!!	DO I=1,GRID_LENGTH(2)
!!!		WRITE(OUTPUT_UNIT4,*) IS_LAND(:,I)
!!!	END DO
!!	WRITE(OUTPUT_UNIT4,'(A,3X,A,3X,A)') "Long","Lat","IsLand"
!!	DO I=1,GRID_LENGTH(1)
!!		DO J=1,GRID_LENGTH(2)
!!			IF(IS_LAND(I,J)) THEN
!!				TEMP_INT=1
!!			ELSE
!!				TEMP_INT=0
!!			END IF
!!			WRITE(OUTPUT_UNIT4,'(F14.4,3X,F14.4,3X,I0)') GRID_COORD(1,I),GRID_COORD(2,J),TEMP_INT
!!		END DO
!!	END DO
!!	CLOSE(OUTPUT_UNIT4)
!
!
!	DO I=1,NUMBER_SNPS
!		ALLELE_FREQUENCY=0.
!		PRINT*, "READ AND FIT-SNP: ", I
!		CALL POOL_GENO_DATA_SQUARE(DATA_MATRIX(:,:,I),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,&
!				SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
!!		CALL FIT_L2_MODEL_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!		CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!		
!		!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!		!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!		CALL UPDATE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!					MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN,IS_LAND)
!							
!		!THIS CALCULATES THE DISTANCE BETWEEN THE TRUE VS. ESTIMATED LOCATIONS
!		IF((MOD(I,10).EQ.0).OR.(I.EQ.NUMBER_SNPS)) THEN
!			AVG_DISTANCE=0.
!			TEMP_DISTANCE=0.
!			DO J=1,NUMBER_UNKNOWN 
!				CALL FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID(:,:,J),TEMP_REAL1,TEMP_REAL2,&
!						MAX_GRID_LENGTH,GRID_COORD,IS_LAND)
!				TEMP_DISTANCE=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),DATA_COORD(:,ADMIXED_MEMBERSHIP(1,J)))
!				AVG_DISTANCE=AVG_DISTANCE+TEMP_DISTANCE
!				!WRITE(OUTPUT_UNIT,*) I,J,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2,UNKNOWN_COORD(:,J)
!			END DO
!			AVG_DISTANCE=AVG_DISTANCE/NUMBER_UNKNOWN
!			PRINT*, "LOCI, AVG_DISTANCE ", I, AVG_DISTANCE, RHO
!!			CALL WRITE_ALLELE_SURFACE_SQUARE(ALLELE_FREQUENCY,GRID_COORD,&
!!					DATA_COORD,DATA_MATRIX(:,:,I),SAMPLE_SITES,I,&
!!					GRID_LENGTH,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!					
!			WRITE(OUTPUT_UNIT3,*) I,"L2Penalty",AVG_DISTANCE
!			
!!			WRITE(OUTPUT_UNIT,*) "POOLED DATA: "
!!			CALL INITIALIZE_ALLELE_FREQUENCY(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,GRID_LENGTH)
!!			CALL WRITE_ALLELE_SURFACE(ALLELE_FREQUENCY,GRID_COORD,&
!!					DATA_COORD,DATA_MATRIX(:,:,I),SAMPLE_SITES,I-1,&
!!					GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!					
!					
!			!WRITES A FEW UNKNOWN HEAT MAPS
!!			CALL WRITE_UNKNOWN_HEATMAP_SQUARE(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!!				DATA_COORD,DATA_MATRIX(:,:,I),ADMIXED_MEMBERSHIP,SAMPLE_SITES,I,&
!!				GRID_LENGTH,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT2)
!			
!			!WRITES ALL OF THE UNKNOWN'S ESTIMATED LOCATIONS
!!			CALL WRITE_UNKNOWN_LOCATIONS(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!!				DATA_COORD,DATA_MATRIX(:,:,I),ADMIXED_MEMBERSHIP,SAMPLE_SITES,I,&
!!				GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT3)
!			
!		END IF
!	END DO
!	
!	CLOSE(OUTPUT_UNIT)
!	CLOSE(OUTPUT_UNIT2)
!	CLOSE(OUTPUT_UNIT3)
!	
!	
!!	CALL WRITE_FIGURE_DATA(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!!			DATA_COORD,DATA_MATRIX(:,:,NUMBER_SNPS),ADMIXED_MEMBERSHIP,SAMPLE_SITES,&
!!			NUMBER_SNPS,GRID_LENGTH,NUMBER_UNKNOWN)
!	
!	DEALLOCATE(DATA_MATRIX,DATA_COORD)
!	DEALLOCATE(UNKNOWN_OBS,UNKNOWN_GRID)
!	DEALLOCATE(ADMIXED_MEMBERSHIP)
!	
!	END SUBROUTINE READ_FIT_FIND_SQUARE
!	
!	
!	
	
	
	
	
	
	
!	
!	
!	
!	
!	
!	SUBROUTINE READ_FIT_FIND_TABLE(DATA_FILE,NUMBER_SNPS,GRID_LENGTH,RHO,AVG_DISTANCE)
!	!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT=12
!	INTEGER :: OUTPUT_UNIT2=13,OUTPUT_UNIT3=14,OUTPUT_UNIT4=15,TEMP_INT
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !DATA_COORD(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: TOTAL_NUMBER,NUMBER_CORRECT,INT_VEC,INT_VEC2
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: TEMP_VECTOR
!	LOGICAL, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: IS_LAND
!	
!	PRINT*, "BEGIN READ_FIT_FIND...RHO: ",RHO
!	
!	CALL UPDATE_IS_LAND(IS_LAND,GRID_LENGTH)
!
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH,GRID_LENGTH,NUMBER_UNKNOWN))
!	ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!	UNKNOWN_OBS=0
!	UNKNOWN_GRID=0.
!	CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
!	LONG_WEIGHT=ONE
!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	LAT_WEIGHT=ONE
!	
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	TEXT_FILE="Table1SNP2K10K"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	!WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A)') "Origin","PercentCorrect","SD"
!	
!	TEXT_FILE="Top20SNPSurfaces"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT2,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT2,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","DataType","Long","Lat","AlleleFreq"
!	
!	TEXT_FILE="LikelihoodSurfaces"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT3,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT3,'(A,3X,A,3X,A,3X,A,3X,A,3X,A)') "NumberSNPs","Individual","Long","Lat","Log(P)"
!	
!	TEXT_FILE="UnknownsPlottedOnMap"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT4,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT4,'(A,3X,A,3X,A,3X,A,3X,A,3X,A)') "Individual","NumberSNPs","Membership","Long","Lat","Log(P)"
!		
!	ALLOCATE(TOTAL_NUMBER(SAMPLE_SITES),NUMBER_CORRECT(SAMPLE_SITES),TEMP_VECTOR(SAMPLE_SITES))
!	ALLOCATE(INT_VEC(10),INT_VEC2(10))
!	INT_VEC=(/16,37,30,23,32,11,33,12,15,19/)
!	INT_VEC2=(/219,200,136,128,125,89,84,71,61,44/)
!	DO I=1,NUMBER_SNPS
!		ALLELE_FREQUENCY=0.
!		PRINT*, "READ AND FIT-SNP: ", I
!		CALL POOL_GENO_DATA(DATA_MATRIX(:,:,I),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,SAMPLE_SITES,GRID_LENGTH)
!!		CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!		CALL FIT_L2_MODEL_QNEWT(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!		
!		!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!		!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!		CALL UPDATE_UNKNOWN_GRID(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!							SAMPLE_SITES,NUMBER_UNKNOWN)
!							
!		!THIS CALCULATES THE DISTANCE BETWEEN THE TRUE VS. ESTIMATED LOCATIONS
!		IF((MOD(I,500).EQ.0).OR.(I.EQ.NUMBER_SNPS)) THEN
!			AVG_DISTANCE=0.
!			TEMP_DISTANCE=0.
!			TOTAL_NUMBER=0
!			NUMBER_CORRECT=0
!			DO J=1,NUMBER_UNKNOWN 
!				CALL FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID(:,:,J),TEMP_REAL1,TEMP_REAL2,&
!						GRID_LENGTH,GRID_COORD,IS_LAND)
!				TEMP_DISTANCE=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),DATA_COORD(:,ADMIXED_MEMBERSHIP(1,J)))
!				AVG_DISTANCE=AVG_DISTANCE+TEMP_DISTANCE
!				DO K=1,SAMPLE_SITES
!					TEMP_VECTOR(K)=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),DATA_COORD(:,K))
!				END DO
!				TEMP_INT=MINLOC(TEMP_VECTOR,1)
!				TOTAL_NUMBER(TEMP_INT)=TOTAL_NUMBER(TEMP_INT)+1
!				IF(TEMP_INT.EQ.ADMIXED_MEMBERSHIP(1,J)) THEN
!					NUMBER_CORRECT(TEMP_INT)=NUMBER_CORRECT(TEMP_INT)+1
!				END IF
!			END DO
!			AVG_DISTANCE=AVG_DISTANCE/NUMBER_UNKNOWN
!			PRINT*, "LOCI, AVG_DISTANCE ", I, AVG_DISTANCE, RHO
!			PRINT*, "POPULATION ", "PERCENT CORRECT", "TOTAL NUMBER", "NUMBER_LOCI: ", I
!			WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A)') "Origin","PercentCorrect","SD"
!			DO J=1,SAMPLE_SITES
!				PRINT*, J, (NUMBER_CORRECT(J)+0.)/TOTAL_NUMBER(J), TOTAL_NUMBER(J)
!				!WRITE(OUTPUT_UNIT,*) J, (NUMBER_CORRECT(J)+0.)/TOTAL_NUMBER(J)
!			END DO
!			DO J=1,SIZE(INT_VEC)
!				TEMP_INT=INT_VEC(J)
!				TEMP_REAL1=(NUMBER_CORRECT(TEMP_INT)+0.)/TOTAL_NUMBER(TEMP_INT)
!				TEMP_REAL2=SQRT(TEMP_REAL1*(1.0-TEMP_REAL1)/(INT_VEC2(J)+0.))
!				WRITE(OUTPUT_UNIT,*) TEMP_INT, TEMP_REAL1, TEMP_REAL2
!				PRINT*, TEMP_INT, TEMP_REAL1, TEMP_REAL2
!			END DO
!			
!		END IF
!		IF(I.LE.20) THEN
!			CALL WRITE_ALLELE_SURFACE(ALLELE_FREQUENCY,GRID_COORD,&
!					DATA_COORD,DATA_MATRIX(:,:,I),SAMPLE_SITES,I,&
!					GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT2)
!		END IF
!		!This writes out the unknown likelihoods
!		IF((MOD(I,100).EQ.0).AND.(I.LE.2000)) THEN
!			CALL WRITE_UNKNOWN_HEATMAP(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!				DATA_COORD,DATA_MATRIX(:,:,I),ADMIXED_MEMBERSHIP,SAMPLE_SITES,I,&
!				GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT3)
!		END IF
!		
!		IF((MOD(I,500).EQ.0).AND.(I.LE.2000)) THEN
!			CALL WRITE_UNKNOWN_LOCATIONS(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!				DATA_COORD,DATA_MATRIX(:,:,I),ADMIXED_MEMBERSHIP,IS_LAND,SAMPLE_SITES,I,&
!				GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT4)
!		END IF
!	END DO
!	
!!	DO I=1,GRID_LENGTH
!!		DO J=1,GRID_LENGTH
!!			IF(IS_LAND(I,J)) THEN
!!				WRITE(OUTPUT_UNIT4,'(I0,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F10.4)') &
!!					I,GRID_LENGTH,ADMIXED_MEMBERSHIP(1,1),GRID_COORD(1,I),GRID_COORD(2,J),PI
!!			END IF
!!		END DO
!!	END DO
!	
!	
!	CLOSE(OUTPUT_UNIT)
!	CLOSE(OUTPUT_UNIT2)
!	CLOSE(OUTPUT_UNIT3)
!	CLOSE(OUTPUT_UNIT4)
!	
!!	CALL WRITE_FIGURE_DATA(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!!			DATA_COORD,DATA_MATRIX(:,:,NUMBER_SNPS),ADMIXED_MEMBERSHIP,SAMPLE_SITES,&
!!			NUMBER_SNPS,GRID_LENGTH,NUMBER_UNKNOWN)
!	
!	DEALLOCATE(DATA_MATRIX,DATA_COORD)
!	DEALLOCATE(UNKNOWN_OBS,UNKNOWN_GRID)
!	DEALLOCATE(ADMIXED_MEMBERSHIP)
!!	PRINT*, (IS_LAND(1:8,J),J=1,8)
!	
!	END SUBROUTINE READ_FIT_FIND_TABLE
!	
!	
!	
!	
!	
!	
!	
!	
!	
!	SUBROUTINE READ_FIT_FIND_TABLE2(DATA_FILE,NUMBER_SNPS,GRID_LENGTH,RHO,AVG_DISTANCE)
!	!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT=12
!	INTEGER :: OUTPUT_UNIT2=13,OUTPUT_UNIT3=14,OUTPUT_UNIT4=15,TEMP_INT,NODES
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !DATA_COORD(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: TOTAL_NUMBER,NUMBER_CORRECT,INT_VEC,INT_VEC2
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: TEMP_VECTOR,TEMP_VECTOR2
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_VECTOR
!	LOGICAL, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: IS_LAND
!	
!	PRINT*, "BEGIN READ_FIT_FIND...RHO: ",RHO
!	
!	CALL UPDATE_IS_LAND(IS_LAND,GRID_LENGTH)
!
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH,GRID_LENGTH,NUMBER_UNKNOWN))
!	ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!	UNKNOWN_OBS=0
!	UNKNOWN_GRID=0.
!	CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
!	LONG_WEIGHT=ONE
!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	LAT_WEIGHT=ONE
!	
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	TEXT_FILE="Table1SNP2K10K"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	!WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A)') "Origin","PercentCorrect","SD"
!	
!	TEXT_FILE="Top50SNPSurfaces"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT2,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT2,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","DataType","Long","Lat","AlleleFreq"
!	
!	TEXT_FILE="LikelihoodSurfacesPost"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT3,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT3,'(A,3X,A,3X,A,3X,A,3X,A,3X,A)') "NumberSNPs","Individual","Long","Lat","Prob"
!	
!	TEXT_FILE="UnknownsPlottedOnMap"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT4,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT4,'(A,3X,A,3X,A,3X,A,3X,A,3X,A)') "Individual","NumberSNPs","Membership","Long","Lat","Log(P)"
!		
!	ALLOCATE(TOTAL_NUMBER(SAMPLE_SITES),NUMBER_CORRECT(SAMPLE_SITES),TEMP_VECTOR(SAMPLE_SITES))
!	ALLOCATE(INT_VEC(10),INT_VEC2(10))
!	INT_VEC=(/16,37,30,23,32,11,33,12,15,19/)
!	INT_VEC2=(/219,200,136,128,125,89,84,71,61,44/)
!	NODES=COUNT(IS_LAND)
!	ALLOCATE(UNKNOWN_VECTOR(NODES,NUMBER_UNKNOWN),TEMP_VECTOR2(NODES)) 
!	UNKNOWN_VECTOR=1.0/NODES
!	DO I=1,NUMBER_SNPS
!		ALLELE_FREQUENCY=0.
!		PRINT*, "READ AND FIT-SNP: ", I
!		CALL POOL_GENO_DATA(DATA_MATRIX(:,:,I),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,SAMPLE_SITES,GRID_LENGTH)
!!		CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!		CALL FIT_L2_MODEL_QNEWT(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!		
!		!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!		!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!!		CALL UPDATE_UNKNOWN_GRID(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!!							SAMPLE_SITES,NUMBER_UNKNOWN)
!		TEMP_VECTOR2=RESHAPE_MATRIX_MASK(ALLELE_FREQUENCY,IS_LAND,GRID_LENGTH,NODES)
!		CALL UPDATE_UNKNOWN_VECTOR(TEMP_VECTOR2,UNKNOWN_OBS(:,I),UNKNOWN_VECTOR,NODES,&
!						SAMPLE_SITES,NUMBER_UNKNOWN)
!		DO J=1,NUMBER_UNKNOWN
!			UNKNOWN_GRID(:,:,J)=RESHAPE_VECTOR_MASK(UNKNOWN_VECTOR(:,J),IS_LAND,GRID_LENGTH,NODES)
!		END DO
!							
!		!THIS CALCULATES THE DISTANCE BETWEEN THE TRUE VS. ESTIMATED LOCATIONS
!!		IF((MOD(I,500).EQ.0).OR.(I.EQ.NUMBER_SNPS)) THEN
!!			AVG_DISTANCE=0.
!!			TEMP_DISTANCE=0.
!!			TOTAL_NUMBER=0
!!			NUMBER_CORRECT=0
!!			DO J=1,NUMBER_UNKNOWN 
!!				CALL FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID(:,:,J),TEMP_REAL1,TEMP_REAL2,&
!!						GRID_LENGTH,GRID_COORD,IS_LAND)
!!				TEMP_DISTANCE=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),DATA_COORD(:,ADMIXED_MEMBERSHIP(1,J)))
!!				AVG_DISTANCE=AVG_DISTANCE+TEMP_DISTANCE
!!				DO K=1,SAMPLE_SITES
!!					TEMP_VECTOR(K)=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),DATA_COORD(:,K))
!!				END DO
!!				TEMP_INT=MINLOC(TEMP_VECTOR,1)
!!				TOTAL_NUMBER(TEMP_INT)=TOTAL_NUMBER(TEMP_INT)+1
!!				IF(TEMP_INT.EQ.ADMIXED_MEMBERSHIP(1,J)) THEN
!!					NUMBER_CORRECT(TEMP_INT)=NUMBER_CORRECT(TEMP_INT)+1
!!				END IF
!!			END DO
!!			AVG_DISTANCE=AVG_DISTANCE/NUMBER_UNKNOWN
!!			PRINT*, "LOCI, AVG_DISTANCE ", I, AVG_DISTANCE, RHO
!!			PRINT*, "POPULATION ", "PERCENT CORRECT", "TOTAL NUMBER", "NUMBER_LOCI: ", I
!!			WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A)') "Origin","PercentCorrect","SD"
!!			DO J=1,SAMPLE_SITES
!!				PRINT*, J, (NUMBER_CORRECT(J)+0.)/TOTAL_NUMBER(J), TOTAL_NUMBER(J)
!!				!WRITE(OUTPUT_UNIT,*) J, (NUMBER_CORRECT(J)+0.)/TOTAL_NUMBER(J)
!!			END DO
!!			DO J=1,SIZE(INT_VEC)
!!				TEMP_INT=INT_VEC(J)
!!				TEMP_REAL1=(NUMBER_CORRECT(TEMP_INT)+0.)/TOTAL_NUMBER(TEMP_INT)
!!				TEMP_REAL2=SQRT(TEMP_REAL1*(1.0-TEMP_REAL1)/(INT_VEC2(J)+0.))
!!				WRITE(OUTPUT_UNIT,*) TEMP_INT, TEMP_REAL1, TEMP_REAL2
!!				PRINT*, TEMP_INT, TEMP_REAL1, TEMP_REAL2
!!			END DO
!!			
!!		END IF
!!		IF(I.LE.500) THEN
!!			CALL WRITE_ALLELE_SURFACE(ALLELE_FREQUENCY,GRID_COORD,&
!!					DATA_COORD,DATA_MATRIX(:,:,I),SAMPLE_SITES,I,&
!!					GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT2)
!!		END IF
!!		!This writes out the unknown likelihoods
!!		IF((MOD(I,50).EQ.0).AND.(I.LE.2000)) THEN
!!			CALL WRITE_UNKNOWN_HEATMAP(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!!				DATA_COORD,DATA_MATRIX(:,:,I),ADMIXED_MEMBERSHIP,SAMPLE_SITES,I,&
!!				GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT3)
!!		END IF
!!		
!!		IF((MOD(I,500).EQ.0).AND.(I.LE.4000)) THEN
!!			CALL WRITE_UNKNOWN_LOCATIONS(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!!				DATA_COORD,DATA_MATRIX(:,:,I),ADMIXED_MEMBERSHIP,IS_LAND,SAMPLE_SITES,I,&
!!				GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT4)
!!		END IF
!	END DO
!	
!!	DO I=1,GRID_LENGTH
!!		DO J=1,GRID_LENGTH
!!			IF(IS_LAND(I,J)) THEN
!!				WRITE(OUTPUT_UNIT4,'(I0,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F10.4)') &
!!					I,GRID_LENGTH,ADMIXED_MEMBERSHIP(1,1),GRID_COORD(1,I),GRID_COORD(2,J),PI
!!			END IF
!!		END DO
!!	END DO
!	
!	
!	CLOSE(OUTPUT_UNIT)
!	CLOSE(OUTPUT_UNIT2)
!	CLOSE(OUTPUT_UNIT3)
!	CLOSE(OUTPUT_UNIT4)
!	
!!	CALL WRITE_FIGURE_DATA(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!!			DATA_COORD,DATA_MATRIX(:,:,NUMBER_SNPS),ADMIXED_MEMBERSHIP,SAMPLE_SITES,&
!!			NUMBER_SNPS,GRID_LENGTH,NUMBER_UNKNOWN)
!	
!	DEALLOCATE(DATA_MATRIX,DATA_COORD)
!	DEALLOCATE(UNKNOWN_OBS,UNKNOWN_GRID)
!	DEALLOCATE(ADMIXED_MEMBERSHIP)
!!	PRINT*, (IS_LAND(1:8,J),J=1,8)
!	
!	END SUBROUTINE READ_FIT_FIND_TABLE2
!	
!	
!	
!	
!	
!	
!	
!	
!	
!	
!	
!	SUBROUTINE READ_FIT_FIND_ADMIXED(DATA_FILE,NUMBER_SNPS,GRID_LENGTH,RHO,AVG_DISTANCE)
!	!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT=12
!	INTEGER :: OUTPUT_UNIT2=13,OUTPUT_UNIT3=14,OUTPUT_UNIT4=15,TEMP_INT,TEMP_INT2
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !DATA_COORD(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT,LAMBDA
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: TOTAL_NUMBER,NUMBER_CORRECT,POPS1_VEC,POPS2_VEC
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: TEMP_VECTOR,GENOME_FRACTIONS,GENOME_FRACTIONS2
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: POP_MATRIX,MLE_POP_MATRIX
!	LOGICAL, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: IS_LAND
!	
!	PRINT*, "BEGIN READ_FIT_FIND_ADMIXED...RHO: ",RHO
!	LAMBDA=100
!	CALL UPDATE_IS_LAND(IS_LAND,GRID_LENGTH)
!
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH,GRID_LENGTH,NUMBER_UNKNOWN))
!	ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!	UNKNOWN_OBS=0
!	UNKNOWN_GRID=0.
!	CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
!	LONG_WEIGHT=ONE
!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	LAT_WEIGHT=ONE
!	
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	TEXT_FILE="AdmixedResults"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Type","Origin1","Origin2","Fraction1","Fraction2"
!	
!	ALLOCATE(POP_MATRIX(NUMBER_SNPS,SAMPLE_SITES),MLE_POP_MATRIX(NUMBER_SNPS,SAMPLE_SITES))
!	ALLOCATE(TOTAL_NUMBER(SAMPLE_SITES),NUMBER_CORRECT(SAMPLE_SITES),TEMP_VECTOR(GRID_LENGTH))
!	ALLOCATE(GENOME_FRACTIONS(SAMPLE_SITES),GENOME_FRACTIONS2(SAMPLE_SITES)) 
!
!	DO I=1,NUMBER_SNPS
!		ALLELE_FREQUENCY=0.
!		PRINT*, "READ AND FIT-SNP: ", I
!		CALL POOL_GENO_DATA(DATA_MATRIX(:,:,I),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,SAMPLE_SITES,GRID_LENGTH)
!!		CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!		CALL FIT_L2_MODEL_QNEWT(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!		
!		!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!		!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!		CALL UPDATE_UNKNOWN_GRID(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!							SAMPLE_SITES,NUMBER_UNKNOWN)
!							
!		DO J=1,SAMPLE_SITES
!			TEMP_INT=MINLOC(ABS(GRID_COORD(1,:)-DATA_COORD(1,J)),1)
!			TEMP_INT2=MINLOC(ABS(GRID_COORD(2,:)-DATA_COORD(2,J)),1)
!			POP_MATRIX(I,J)=ALLELE_FREQUENCY(TEMP_INT,TEMP_INT2)
!!			MLE_POP_MATRIX(I,J)=(DATA_ALLELE1(TEMP_INT,TEMP_INT2)+0.)/(DATA_TOTAL(TEMP_INT,TEMP_INT2)+0.)
!			MLE_POP_MATRIX(I,J)=(DATA_MATRIX(1,J,I)+0.)/(SUM(DATA_MATRIX(:,J,I))+0.)
!		END DO
!!		PRINT*, "J ","POP_MATRIX ", "MLE_POP ", "DIFF "
!!		DO J=1,SAMPLE_SITES
!!			PRINT*, J,POP_MATRIX(I,J),MLE_POP_MATRIX(I,J),ABS(MLE_POP_MATRIX(I,J)-POP_MATRIX(I,J))
!!		END DO
!		
!	END DO
!	
!	ALLOCATE(POPS1_VEC(10),POPS2_VEC(10))
!	POPS1_VEC=(/ 16, 16, 16, 32, 23, 30, 23, 11, 12, 12 /)
!	POPS2_VEC=(/ 37, 23, 30, 37, 37, 37, 30, 16, 16, 23 /)
!	GENOME_FRACTIONS=1./SAMPLE_SITES
!	GENOME_FRACTIONS2=1./SAMPLE_SITES
!	TEMP_REAL1=0.
!	TEMP_REAL2=0.
!	TEMP_INT=NINT(NUMBER_UNKNOWN/10.)
!	DO I=1,NUMBER_UNKNOWN
!		CALL FIT_FRACTIONS_MODEL(GENOME_FRACTIONS,POP_MATRIX,&
!			UNKNOWN_OBS(I,:),LAMBDA,NUMBER_SNPS,SAMPLE_SITES)
!		CALL FIT_FRACTIONS_MODEL_QNEWT(GENOME_FRACTIONS2,POP_MATRIX,&
!			UNKNOWN_OBS(I,:),LAMBDA,NUMBER_SNPS,SAMPLE_SITES)
!		PRINT*, "MODEL VS. QUASI NEWT DIFFERENCES"
!		DO J=1,SAMPLE_SITES
!			PRINT*, GENOME_FRACTIONS(J)-GENOME_FRACTIONS2(J)
!		END DO
!		TEMP_INT2=NINT((I-1.)/TEMP_INT)+1
!!		WRITE(OUTPUT_UNIT,*) "Genography",POPS1_VEC(TEMP_INT2),POPS2_VEC(TEMP_INT2),&
!!			GENOME_FRACTIONS(POPS1_VEC(TEMP_INT2)),GENOME_FRACTIONS(POPS2_VEC(TEMP_INT2))
!!		PRINT*, "Genography",POPS1_VEC(TEMP_INT2),POPS2_VEC(TEMP_INT2),&
!!			GENOME_FRACTIONS(POPS1_VEC(TEMP_INT2)),GENOME_FRACTIONS(POPS2_VEC(TEMP_INT2))
!		WRITE(OUTPUT_UNIT,*) "Genography",ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I),&
!			GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(1,I)),GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(2,I))
!		PRINT*, "Genography",ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I),&
!			GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(1,I)),GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(2,I))
!		TEMP_REAL1=TEMP_REAL1+GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(1,I))/TEMP_INT
!		TEMP_REAL2=TEMP_REAL2+GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(2,I))/TEMP_INT
!		IF(MOD(I,TEMP_INT).EQ.0) THEN
!			WRITE(OUTPUT_UNIT,*) "GenographyAVG",ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I),&
!				TEMP_REAL1,TEMP_REAL2
!			PRINT*, "GenographyAVG",ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I),&
!				TEMP_REAL1,TEMP_REAL2
!			TEMP_REAL1=0.
!			TEMP_REAL2=0.
!		END IF
!	END DO
!	TEMP_REAL1=0.
!	TEMP_REAL2=0.
!	GENOME_FRACTIONS=1./SAMPLE_SITES
!	GENOME_FRACTIONS2=1./SAMPLE_SITES
!	DO I=1,NUMBER_UNKNOWN
!		CALL FIT_FRACTIONS_MODEL(GENOME_FRACTIONS,MLE_POP_MATRIX,&
!			UNKNOWN_OBS(I,:),LAMBDA,NUMBER_SNPS,SAMPLE_SITES)
!		CALL FIT_FRACTIONS_MODEL_QNEWT(GENOME_FRACTIONS2,POP_MATRIX,&
!			UNKNOWN_OBS(I,:),LAMBDA,NUMBER_SNPS,SAMPLE_SITES)
!		PRINT*, "MODEL VS. QUASI NEWT DIFFERENCES"
!		DO J=1,SAMPLE_SITES
!			PRINT*, GENOME_FRACTIONS(J)-GENOME_FRACTIONS2(J)
!		END DO
!		TEMP_INT2=NINT((I-1.)/TEMP_INT)+1
!!		WRITE(OUTPUT_UNIT,*) "Admixture",POPS1_VEC(TEMP_INT2),POPS2_VEC(TEMP_INT2),&
!!			GENOME_FRACTIONS(POPS1_VEC(TEMP_INT2)),GENOME_FRACTIONS(POPS2_VEC(TEMP_INT2))
!!		PRINT*, "Admixture",POPS1_VEC(TEMP_INT2),POPS2_VEC(TEMP_INT2),&
!!			GENOME_FRACTIONS(POPS1_VEC(TEMP_INT2)),GENOME_FRACTIONS(POPS2_VEC(TEMP_INT2))
!		WRITE(OUTPUT_UNIT,*) "Admixture",ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I),&
!			GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(1,I)),GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(2,I))
!		PRINT*, "Admixture",ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I),&
!			GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(1,I)),GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(2,I))
!		TEMP_REAL1=TEMP_REAL1+GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(1,I))/TEMP_INT
!		TEMP_REAL2=TEMP_REAL2+GENOME_FRACTIONS(ADMIXED_MEMBERSHIP(2,I))/TEMP_INT
!		IF(MOD(I,TEMP_INT).EQ.0) THEN
!			WRITE(OUTPUT_UNIT,*) "AdmixtureAVG",ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I),&
!				TEMP_REAL1,TEMP_REAL2
!			PRINT*, "AdmixtureAVG",ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I),&
!				TEMP_REAL1,TEMP_REAL2
!			TEMP_REAL1=0.
!			TEMP_REAL2=0.
!		END IF
!	END DO
!	
!	
!	CLOSE(OUTPUT_UNIT)	
!	
!	DEALLOCATE(DATA_MATRIX,DATA_COORD)
!	DEALLOCATE(UNKNOWN_OBS,UNKNOWN_GRID)
!	DEALLOCATE(ADMIXED_MEMBERSHIP)
!	
!	END SUBROUTINE READ_FIT_FIND_ADMIXED
!	
!	
!	
!	
!	
!	
!	SUBROUTINE READ_FIT_FIND_ADMIXED_INTERP(DATA_FILE,NUMBER_SNPS,GRID_LENGTH,RHO,AVG_DISTANCE)
!	!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT=12
!	INTEGER :: OUTPUT_UNIT2=13,OUTPUT_UNIT3=14,OUTPUT_UNIT4=15
!	INTEGER :: NODES,TEMP_INT,TEMP_INT1,TEMP_INT2
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !DATA_COORD(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT,LAMBDA
!!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: TOTAL_NUMBER,NUMBER_CORRECT,POPS1_VEC,POPS2_VEC
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: TEMP_VECTOR
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: POP_MATRIX,GENOME_FRACTIONS
!	LOGICAL, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: IS_LAND
!	
!	PRINT*, "BEGIN READ_FIT_FIND_ADMIXED_INTERP...RHO: ",RHO
!	LAMBDA=100
!	CALL UPDATE_IS_LAND(IS_LAND,GRID_LENGTH)
!
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH,GRID_LENGTH,NUMBER_UNKNOWN))
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS))
!	ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!	UNKNOWN_OBS=0
!	CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
!	LONG_WEIGHT=ONE
!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	LAT_WEIGHT=ONE
!	
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!!	TEXT_FILE="AdmixedResultsInterp"//TRIM(TEXT_END)//".txt"
!!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Type","Origin1","Origin2","Fraction1","Fraction2"
!	
!	TEXT_FILE="AdmixedSurfaces"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT2,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT2,'(A,3X,A,3X,A,3X,A,3X,A,3X,A,3X,A)') "Individual","DataType","Membership1",&
!				"Membership2","Long","Lat","Fraction"
!	
!!	NODES=GRID_LENGTH*GRID_LENGTH
!	NODES=COUNT(IS_LAND)
!	ALLOCATE(POP_MATRIX(NUMBER_SNPS,NODES))
!	ALLOCATE(TOTAL_NUMBER(SAMPLE_SITES),NUMBER_CORRECT(SAMPLE_SITES))
!	ALLOCATE(GENOME_FRACTIONS(GRID_LENGTH,GRID_LENGTH),TEMP_VECTOR(NODES)) 
!	POP_MATRIX=0.
!
!	DO I=1,NUMBER_SNPS
!		ALLELE_FREQUENCY=0.
!		PRINT*, "READ AND FIT-SNP: ", I
!		CALL POOL_GENO_DATA(DATA_MATRIX(:,:,I),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,SAMPLE_SITES,GRID_LENGTH)
!!		CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!		CALL FIT_L2_MODEL_QNEWT(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!		
!		!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!		!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!!		CALL UPDATE_UNKNOWN_GRID(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!!							SAMPLE_SITES,NUMBER_UNKNOWN)
!							
!!		POP_MATRIX(I,:)=RESHAPE(ALLELE_FREQUENCY,SHAPE(POP_MATRIX(I,:)))
!		POP_MATRIX(I,:)=RESHAPE_MATRIX_MASK(ALLELE_FREQUENCY,IS_LAND,GRID_LENGTH,NODES)
!		
!	END DO
!	
!	ALLOCATE(POPS1_VEC(10),POPS2_VEC(10))
!	POPS1_VEC=(/ 16, 16, 16, 32, 23, 30, 23, 11, 12, 12 /)
!	POPS2_VEC=(/ 37, 23, 30, 37, 37, 37, 30, 16, 16, 23 /)
!	GENOME_FRACTIONS=0.
!	TEMP_REAL1=0.
!	TEMP_REAL2=0.
!	TEMP_INT=NINT(NUMBER_UNKNOWN/10.)
!	DO I=1,NUMBER_UNKNOWN
!!		CALL FIT_FRACTIONS_MODEL(TEMP_VECTOR,POP_MATRIX,&
!!			UNKNOWN_OBS(I,:),LAMBDA,NUMBER_SNPS,NODES)
!
!
!!!THE FOLLOWING CODE INITIALIZES THE ALLELE FREQUENCY VECTOR AT ITS TRUE VALUE FOR TESTING
!!		ALLELE_FREQUENCY=0.02/(NODES-2.0)
!!		TEMP_INT=ADMIXED_MEMBERSHIP(1,I)
!!		TEMP_REAL1=DATA_COORD(1,TEMP_INT)
!!		TEMP_REAL2=DATA_COORD(2,TEMP_INT)
!!		TEMP_INT1=MINLOC(ABS(GRID_COORD(1,:)-TEMP_REAL1),1)
!!		TEMP_INT2=MINLOC(ABS(GRID_COORD(2,:)-TEMP_REAL2),1)
!!		PRINT*, "ADMIXED COORD: ", TEMP_REAL1, GRID_COORD(1,TEMP_INT1)
!!		PRINT*, "ADMIXED COORD: ", TEMP_REAL2, GRID_COORD(2,TEMP_INT2)
!!		ALLELE_FREQUENCY(TEMP_INT1,TEMP_INT2)=0.4900000
!!		
!!		TEMP_INT=ADMIXED_MEMBERSHIP(2,I)
!!		TEMP_REAL1=DATA_COORD(1,TEMP_INT)
!!		TEMP_REAL2=DATA_COORD(2,TEMP_INT)
!!		TEMP_INT1=MINLOC(ABS(GRID_COORD(1,:)-TEMP_REAL1),1)
!!		TEMP_INT2=MINLOC(ABS(GRID_COORD(2,:)-TEMP_REAL2),1)
!!		PRINT*, "ADMIXED COORD2: ", TEMP_REAL1, GRID_COORD(1,TEMP_INT1)
!!		PRINT*, "ADMIXED COORD2: ", TEMP_REAL2, GRID_COORD(2,TEMP_INT2)
!!		ALLELE_FREQUENCY(TEMP_INT1,TEMP_INT2)=0.4900000
!		ALLELE_FREQUENCY=ONE/NODES
!		
!		TEMP_VECTOR=RESHAPE_MATRIX_MASK(ALLELE_FREQUENCY,IS_LAND,GRID_LENGTH,NODES)
!		CALL FIT_FRACTIONS_MODEL_QNEWT(TEMP_VECTOR,POP_MATRIX,&
!			UNKNOWN_OBS(I,:),LAMBDA,NUMBER_SNPS,NODES)
!!		GENOME_FRACTIONS=RESHAPE(TEMP_VECTOR,SHAPE(GENOME_FRACTIONS))
!		GENOME_FRACTIONS=RESHAPE_VECTOR_MASK(TEMP_VECTOR,IS_LAND,GRID_LENGTH,NODES)
!		CALL WRITE_GENOME_FRACTIONS(GENOME_FRACTIONS,GRID_COORD,&
!				ADMIXED_MEMBERSHIP,DATA_COORD,SAMPLE_SITES,I,&
!				GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT2)
!
!!		WRITE(OUTPUT_UNIT,*) "Genography",I,ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I)
!		PRINT*, "Genography",I,ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I)
!	END DO
!	
!!	CLOSE(OUTPUT_UNIT)	
!	CLOSE(OUTPUT_UNIT2)
!	
!	DEALLOCATE(DATA_MATRIX,DATA_COORD)
!	DEALLOCATE(UNKNOWN_OBS)
!	DEALLOCATE(ADMIXED_MEMBERSHIP,POP_MATRIX)
!!	DEALLOCATE(UNKNOWN_GRID)
!	
!	END SUBROUTINE READ_FIT_FIND_ADMIXED_INTERP
!	
!	
!	
!	
!	
!	SUBROUTINE READ_FIT_FIND_ADMIXED_INTERPG(DATA_FILE,NUMBER_SNPS,GRID_LENGTH,RHO,AVG_DISTANCE)
!	!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL
!	!THIS SUBROUTINE USES DATA SIMULATED FROM GRANDPARENTS INSTEAD OF PARENTS
!	
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT=12
!	INTEGER :: OUTPUT_UNIT2=13,OUTPUT_UNIT3=14,OUTPUT_UNIT4=15
!	INTEGER :: NODES,TEMP_INT,TEMP_INT1,TEMP_INT2
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !DATA_COORD(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT,LAMBDA
!!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: TOTAL_NUMBER,NUMBER_CORRECT,POPS1_VEC,POPS2_VEC
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: TEMP_VECTOR
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: POP_MATRIX,GENOME_FRACTIONS
!	LOGICAL, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: IS_LAND
!	
!	PRINT*, "BEGIN READ_FIT_FIND_ADMIXED_INTERP...RHO: ",RHO
!	LAMBDA=100
!	CALL UPDATE_IS_LAND(IS_LAND,GRID_LENGTH)
!
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH,GRID_LENGTH,NUMBER_UNKNOWN))
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS))
!	ALLOCATE(ADMIXED_MEMBERSHIP(4,NUMBER_UNKNOWN))
!	UNKNOWN_OBS=0
!	CALL READ_UNKNOWN_DATAG(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
!	LONG_WEIGHT=ONE
!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	LAT_WEIGHT=ONE
!	
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!!	TEXT_FILE="AdmixedResultsInterp"//TRIM(TEXT_END)//".txt"
!!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Type","Origin1","Origin2","Fraction1","Fraction2"
!	
!	TEXT_FILE="AdmixedSurfacesG"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT2,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT2,'(A,3X,A,3X,A,3X,A,3X,A,3X,A,3X,A,3X,A,3X,A)') "Individual","DataType","Membership1",&
!				"Membership2","Membership3","Membership4","Long","Lat","Fraction"
!	
!!	NODES=GRID_LENGTH*GRID_LENGTH
!	NODES=COUNT(IS_LAND)
!	ALLOCATE(POP_MATRIX(NUMBER_SNPS,NODES))
!	ALLOCATE(TOTAL_NUMBER(SAMPLE_SITES),NUMBER_CORRECT(SAMPLE_SITES))
!	ALLOCATE(GENOME_FRACTIONS(GRID_LENGTH,GRID_LENGTH),TEMP_VECTOR(NODES)) 
!	POP_MATRIX=0.
!
!	DO I=1,NUMBER_SNPS
!		ALLELE_FREQUENCY=0.
!		PRINT*, "READ AND FIT-SNP: ", I
!		CALL POOL_GENO_DATA(DATA_MATRIX(:,:,I),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,SAMPLE_SITES,GRID_LENGTH)
!!		CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!		CALL FIT_L2_MODEL_QNEWT(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!		
!		!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!		!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!!		CALL UPDATE_UNKNOWN_GRID(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!!							SAMPLE_SITES,NUMBER_UNKNOWN)
!							
!!		POP_MATRIX(I,:)=RESHAPE(ALLELE_FREQUENCY,SHAPE(POP_MATRIX(I,:)))
!		POP_MATRIX(I,:)=RESHAPE_MATRIX_MASK(ALLELE_FREQUENCY,IS_LAND,GRID_LENGTH,NODES)
!		
!	END DO
!	
!	ALLOCATE(POPS1_VEC(10),POPS2_VEC(10))
!	POPS1_VEC=(/ 16, 16, 16, 32, 23, 30, 23, 11, 12, 12 /)
!	POPS2_VEC=(/ 37, 23, 30, 37, 37, 37, 30, 16, 16, 23 /)
!	GENOME_FRACTIONS=0.
!	TEMP_REAL1=0.
!	TEMP_REAL2=0.
!	TEMP_INT=NINT(NUMBER_UNKNOWN/10.)
!	DO I=1,NUMBER_UNKNOWN
!!		CALL FIT_FRACTIONS_MODEL(TEMP_VECTOR,POP_MATRIX,&
!!			UNKNOWN_OBS(I,:),LAMBDA,NUMBER_SNPS,NODES)
!
!
!!!THE FOLLOWING CODE INITIALIZES THE ALLELE FREQUENCY VECTOR AT ITS TRUE VALUE FOR TESTING
!!		ALLELE_FREQUENCY=0.02/(NODES-2.0)
!!		TEMP_INT=ADMIXED_MEMBERSHIP(1,I)
!!		TEMP_REAL1=DATA_COORD(1,TEMP_INT)
!!		TEMP_REAL2=DATA_COORD(2,TEMP_INT)
!!		TEMP_INT1=MINLOC(ABS(GRID_COORD(1,:)-TEMP_REAL1),1)
!!		TEMP_INT2=MINLOC(ABS(GRID_COORD(2,:)-TEMP_REAL2),1)
!!		PRINT*, "ADMIXED COORD: ", TEMP_REAL1, GRID_COORD(1,TEMP_INT1)
!!		PRINT*, "ADMIXED COORD: ", TEMP_REAL2, GRID_COORD(2,TEMP_INT2)
!!		ALLELE_FREQUENCY(TEMP_INT1,TEMP_INT2)=0.4900000
!!		
!!		TEMP_INT=ADMIXED_MEMBERSHIP(2,I)
!!		TEMP_REAL1=DATA_COORD(1,TEMP_INT)
!!		TEMP_REAL2=DATA_COORD(2,TEMP_INT)
!!		TEMP_INT1=MINLOC(ABS(GRID_COORD(1,:)-TEMP_REAL1),1)
!!		TEMP_INT2=MINLOC(ABS(GRID_COORD(2,:)-TEMP_REAL2),1)
!!		PRINT*, "ADMIXED COORD2: ", TEMP_REAL1, GRID_COORD(1,TEMP_INT1)
!!		PRINT*, "ADMIXED COORD2: ", TEMP_REAL2, GRID_COORD(2,TEMP_INT2)
!!		ALLELE_FREQUENCY(TEMP_INT1,TEMP_INT2)=0.4900000
!		ALLELE_FREQUENCY=ONE/NODES
!		
!		TEMP_VECTOR=RESHAPE_MATRIX_MASK(ALLELE_FREQUENCY,IS_LAND,GRID_LENGTH,NODES)
!		CALL FIT_FRACTIONS_MODEL_QNEWT(TEMP_VECTOR,POP_MATRIX,&
!			UNKNOWN_OBS(I,:),LAMBDA,NUMBER_SNPS,NODES)
!!		GENOME_FRACTIONS=RESHAPE(TEMP_VECTOR,SHAPE(GENOME_FRACTIONS))
!		GENOME_FRACTIONS=RESHAPE_VECTOR_MASK(TEMP_VECTOR,IS_LAND,GRID_LENGTH,NODES)
!		CALL WRITE_GENOME_FRACTIONSG(GENOME_FRACTIONS,GRID_COORD,&
!				ADMIXED_MEMBERSHIP,DATA_COORD,SAMPLE_SITES,I,&
!				GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT2)
!
!!		WRITE(OUTPUT_UNIT,*) "Genography",I,ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I)
!		PRINT*, "Genography",I,ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I),&
!					ADMIXED_MEMBERSHIP(3,I),ADMIXED_MEMBERSHIP(4,I)
!	END DO
!	
!!	CLOSE(OUTPUT_UNIT)	
!	CLOSE(OUTPUT_UNIT2)
!	
!	DEALLOCATE(DATA_MATRIX,DATA_COORD)
!	DEALLOCATE(UNKNOWN_OBS)
!	DEALLOCATE(ADMIXED_MEMBERSHIP,POP_MATRIX)
!!	DEALLOCATE(UNKNOWN_GRID)
!	
!	END SUBROUTINE READ_FIT_FIND_ADMIXED_INTERPG
!	
!	
!!	
!!	SUBROUTINE READ_FIT_FIND_ADMIXED_INTERP2(DATA_FILE,NUMBER_SNPS,GRID_LENGTH,RHO,AVG_DISTANCE)
!!	!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL
!!	!THIS SUBROUTINE INITIALIZES THE ADMIXTURE MODEL AT THE POSTERIOR PROBABILITY WHICH DOES NOT WORK WELL
!!	
!!	IMPLICIT NONE
!!	
!!	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT=12
!!	INTEGER :: OUTPUT_UNIT2=13,OUTPUT_UNIT3=14,OUTPUT_UNIT4=15
!!	INTEGER :: NODES,TEMP_INT,TEMP_INT1,TEMP_INT2
!!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !DATA_COORD(LONG,LAT)
!!	REAL(KIND=DBLE_PREC), DIMENSION(2,GRID_LENGTH) :: GRID_COORD !LONG,LAT
!!	INTEGER, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: DATA_ALLELE1,DATA_TOTAL
!!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY !LONG,LAT
!!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
!!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT,LAMBDA
!!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_VECTOR
!!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP
!!	INTEGER, ALLOCATABLE, DIMENSION(:) :: TOTAL_NUMBER,NUMBER_CORRECT,POPS1_VEC,POPS2_VEC
!!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: TEMP_VECTOR
!!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: POP_MATRIX,GENOME_FRACTIONS
!!	LOGICAL, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: IS_LAND
!!	
!!	PRINT*, "BEGIN READ_FIT_FIND_ADMIXED_INTERP...RHO: ",RHO
!!	LAMBDA=100
!!	CALL UPDATE_IS_LAND(IS_LAND,GRID_LENGTH)
!!
!!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!!	DATA_MATRIX=0
!!	DATA_COORD=0.
!!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!!	
!!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!!!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH,GRID_LENGTH,NUMBER_UNKNOWN))
!!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS))
!!	ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!!	UNKNOWN_OBS=0
!!	CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!!	
!!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
!!	LONG_WEIGHT=ONE
!!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!!	LAT_WEIGHT=ONE
!!	
!!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!!!	TEXT_FILE="AdmixedResultsInterp"//TRIM(TEXT_END)//".txt"
!!!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!!!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Type","Origin1","Origin2","Fraction1","Fraction2"
!!	
!!	TEXT_FILE="AdmixedSurfacesPostInit"//TRIM(TEXT_END)//".txt"
!!	OPEN(OUTPUT_UNIT2,FILE=TEXT_FILE)
!!	WRITE(OUTPUT_UNIT2,'(A,3X,A,3X,A,3X,A,3X,A,3X,A,3X,A)') "Individual","DataType","Membership1",&
!!				"Membership2","Long","Lat","Fraction"
!!	
!!!	NODES=GRID_LENGTH*GRID_LENGTH
!!	NODES=COUNT(IS_LAND)
!!	ALLOCATE(POP_MATRIX(NUMBER_SNPS,NODES))
!!	ALLOCATE(TOTAL_NUMBER(SAMPLE_SITES),NUMBER_CORRECT(SAMPLE_SITES))
!!	ALLOCATE(GENOME_FRACTIONS(GRID_LENGTH,GRID_LENGTH),TEMP_VECTOR(NODES)) 
!!	ALLOCATE(UNKNOWN_VECTOR(NODES,NUMBER_UNKNOWN))
!!	UNKNOWN_VECTOR=1.0/NODES
!!	POP_MATRIX=0.
!!
!!	DO I=1,NUMBER_SNPS
!!		ALLELE_FREQUENCY=0.
!!		PRINT*, "READ AND FIT-SNP: ", I
!!		CALL POOL_GENO_DATA(DATA_MATRIX(:,:,I),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,SAMPLE_SITES,GRID_LENGTH)
!!!		CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!!		CALL FIT_L2_MODEL_QNEWT(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!!		
!!		!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!!		!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!!!		CALL UPDATE_UNKNOWN_GRID(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!!!							SAMPLE_SITES,NUMBER_UNKNOWN)
!!							
!!!		POP_MATRIX(I,:)=RESHAPE(ALLELE_FREQUENCY,SHAPE(POP_MATRIX(I,:)))
!!		POP_MATRIX(I,:)=RESHAPE_MATRIX_MASK(ALLELE_FREQUENCY,IS_LAND,GRID_LENGTH,NODES)
!!		CALL UPDATE_UNKNOWN_VECTOR(POP_MATRIX(I,:),UNKNOWN_OBS(:,I),UNKNOWN_VECTOR,NODES,&
!!							SAMPLE_SITES,NUMBER_UNKNOWN)		
!!!		PRINT*, UNKNOWN_VECTOR(:,1)
!!	END DO
!!	
!!	ALLOCATE(POPS1_VEC(10),POPS2_VEC(10))
!!	POPS1_VEC=(/ 16, 16, 16, 32, 23, 30, 23, 11, 12, 12 /)
!!	POPS2_VEC=(/ 37, 23, 30, 37, 37, 37, 30, 16, 16, 23 /)
!!	GENOME_FRACTIONS=0.
!!	TEMP_REAL1=0.
!!	TEMP_REAL2=0.
!!	TEMP_INT=NINT(NUMBER_UNKNOWN/10.)
!!	DO I=1,NUMBER_UNKNOWN
!!!		CALL FIT_FRACTIONS_MODEL(TEMP_VECTOR,POP_MATRIX,&
!!!			UNKNOWN_OBS(I,:),LAMBDA,NUMBER_SNPS,NODES)
!!
!!
!!!!THE FOLLOWING CODE INITIALIZES THE ALLELE FREQUENCY VECTOR AT ITS TRUE VALUE FOR TESTING
!!!		ALLELE_FREQUENCY=0.02/(NODES-2.0)
!!!		TEMP_INT=ADMIXED_MEMBERSHIP(1,I)
!!!		TEMP_REAL1=DATA_COORD(1,TEMP_INT)
!!!		TEMP_REAL2=DATA_COORD(2,TEMP_INT)
!!!		TEMP_INT1=MINLOC(ABS(GRID_COORD(1,:)-TEMP_REAL1),1)
!!!		TEMP_INT2=MINLOC(ABS(GRID_COORD(2,:)-TEMP_REAL2),1)
!!!		PRINT*, "ADMIXED COORD: ", TEMP_REAL1, GRID_COORD(1,TEMP_INT1)
!!!		PRINT*, "ADMIXED COORD: ", TEMP_REAL2, GRID_COORD(2,TEMP_INT2)
!!!		ALLELE_FREQUENCY(TEMP_INT1,TEMP_INT2)=0.4900000
!!!		
!!!		TEMP_INT=ADMIXED_MEMBERSHIP(2,I)
!!!		TEMP_REAL1=DATA_COORD(1,TEMP_INT)
!!!		TEMP_REAL2=DATA_COORD(2,TEMP_INT)
!!!		TEMP_INT1=MINLOC(ABS(GRID_COORD(1,:)-TEMP_REAL1),1)
!!!		TEMP_INT2=MINLOC(ABS(GRID_COORD(2,:)-TEMP_REAL2),1)
!!!		PRINT*, "ADMIXED COORD2: ", TEMP_REAL1, GRID_COORD(1,TEMP_INT1)
!!!		PRINT*, "ADMIXED COORD2: ", TEMP_REAL2, GRID_COORD(2,TEMP_INT2)
!!!		ALLELE_FREQUENCY(TEMP_INT1,TEMP_INT2)=0.4900000
!!!		ALLELE_FREQUENCY=ONE/NODES
!!		
!!!		TEMP_VECTOR=RESHAPE_MATRIX_MASK(ALLELE_FREQUENCY,IS_LAND,GRID_LENGTH,NODES)
!!		TEMP_VECTOR=UNKNOWN_VECTOR(:,I)
!!		CALL FIT_FRACTIONS_MODEL_QNEWT(TEMP_VECTOR,POP_MATRIX,&
!!			UNKNOWN_OBS(I,:),LAMBDA,NUMBER_SNPS,NODES)
!!!		GENOME_FRACTIONS=RESHAPE(TEMP_VECTOR,SHAPE(GENOME_FRACTIONS))
!!		GENOME_FRACTIONS=RESHAPE_VECTOR_MASK(TEMP_VECTOR,IS_LAND,GRID_LENGTH,NODES)
!!		CALL WRITE_GENOME_FRACTIONS(GENOME_FRACTIONS,GRID_COORD,&
!!				ADMIXED_MEMBERSHIP,DATA_COORD,SAMPLE_SITES,I,&
!!				GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT2)
!!
!!!		WRITE(OUTPUT_UNIT,*) "Genography",I,ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I)
!!		PRINT*, "Genography",I,ADMIXED_MEMBERSHIP(1,I),ADMIXED_MEMBERSHIP(2,I)
!!	END DO
!!	
!!	!this section writes out the posterior probabilities used for the unknowns.
!!	TEXT_FILE="AdmixedSurfacesPostVals"//TRIM(TEXT_END)//".txt"
!!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A,3X,A,3X,A)') "Individual","DataType","Membership1",&
!!				"Membership2","Long","Lat","Fraction"
!!	DO I=1,NUMBER_UNKNOWN
!!		GENOME_FRACTIONS=RESHAPE_VECTOR_MASK(UNKNOWN_VECTOR(:,I),IS_LAND,GRID_LENGTH,NODES)
!!		CALL WRITE_GENOME_FRACTIONS(GENOME_FRACTIONS,GRID_COORD,&
!!				ADMIXED_MEMBERSHIP,DATA_COORD,SAMPLE_SITES,I,&
!!				GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!!	END DO
!!	CLOSE(OUTPUT_UNIT)	
!!	
!!	
!!	CLOSE(OUTPUT_UNIT2)
!!	
!!	DEALLOCATE(DATA_MATRIX,DATA_COORD)
!!	DEALLOCATE(UNKNOWN_OBS)
!!	DEALLOCATE(ADMIXED_MEMBERSHIP,POP_MATRIX)
!!!	DEALLOCATE(UNKNOWN_GRID)
!!	
!!	END SUBROUTINE READ_FIT_FIND_ADMIXED_INTERP2
!!	
!!	
!!	
!!	
!	
	
	
	!
!	
!
!	SUBROUTINE LEAVE_ONE_POP_OUT_SQUARE(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO,AVG_DISTANCE)
!	!THIS SUBROUTINE ESTIMATES THE RHO WEIGHT IN OUR MODEL USING CROSS VALIDATION
!		
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,MAX_GRID_LENGTH,NUMBER_UNKNOWN,TEMP_INT,OUTPUT_UNIT=12
!	INTEGER :: POP,OUTPUT_UNIT2=13,OUTPUT_UNIT3=14,OUTPUT_UNIT4=15
!	INTEGER, DIMENSION(2) :: GRID_LENGTH
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !DATA_COORD(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: ALLELE_FREQUENCY !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT,SUM_DISTANCE
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP,TEMP_DATA_MATRIX
!	LOGICAL, ALLOCATABLE, DIMENSION(:,:) :: IS_LAND
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: SAMPLE_SIZES
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: POP_AVG_DISTANCE
!	
!	PRINT*, "LEAVE ONE POPULATION OUT SQUARE"
!	GRID_LENGTH=0
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	ALLOCATE(TEMP_DATA_MATRIX(2,SAMPLE_SITES))
!	TEMP_DATA_MATRIX=0
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD_SQUARE(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!	ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
!	ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)))
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN))
!	ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!	UNKNOWN_OBS=0
!	UNKNOWN_GRID=0.
!	CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	ALLOCATE(IS_LAND(GRID_LENGTH(1),GRID_LENGTH(2)))
!	CALL UPDATE_IS_LAND_SQUARE(IS_LAND,GRID_LENGTH,MAX_GRID_LENGTH)
!
!	UNKNOWN_GRID=ONE/(COUNT(IS_LAND)+0.)
!	DO J=1,GRID_LENGTH(2)
!		DO I=1,GRID_LENGTH(1)
!			IF(.NOT. IS_LAND(I,J)) THEN
!				UNKNOWN_GRID(I,J,:)=ZERO
!			END IF
!		END DO
!	END DO
!
!	LONG_WEIGHT=ONE
!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	LAT_WEIGHT=ONE
!	
!	PRINT*, "NUMBER_UNKNOWN(1387): ",NUMBER_UNKNOWN
!	
!	ALLOCATE(SAMPLE_SIZES(SAMPLE_SITES))
!	SAMPLE_SIZES=0
!	DO I=1,NUMBER_UNKNOWN
!		SAMPLE_SIZES(ADMIXED_MEMBERSHIP(1,I))=SAMPLE_SIZES(ADMIXED_MEMBERSHIP(1,I))+1
!	END DO
!	
!	ALLOCATE(POP_AVG_DISTANCE(SAMPLE_SITES))
!	POP_AVG_DISTANCE=0.
!
!
!	
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	TEXT_FILE="AlleleSurfacesSqL1PopOut"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","DataType","Long","Lat","AlleleFreq"
!	
!	TEXT_FILE="UnknownHeatMapsSqL1PopOut"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT2,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT2,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","Individual","Long","Lat","Prob"
!	
!	TEXT_FILE="UnknownsPlottedOnMapSqL1PopOut"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT3,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT3,'(A,3X,A,3X,A,3X,A,3X,A,3X,A)') "Individual","NumberSNPs","Membership","Long","Lat","Prob"
!	
!	TEXT_FILE="AvgDistSqLeave1PopOut"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT4,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT4,'(A,3X,A,3X,A)') "Loci","Method","AverageDistance"
!	
!
!
!	DO I=1,NUMBER_SNPS !NEED TO CHANGE THIS TO 1,NUMBER_SNPS
!		IF(MOD(I,1).EQ.0) THEN
!			PRINT*, "READ AND FIT-SNP: ", I
!		END IF
!		DO POP=1,SAMPLE_SITES
!			ALLELE_FREQUENCY=0.
!			TEMP_DATA_MATRIX=DATA_MATRIX(:,:,I)
!			TEMP_DATA_MATRIX(:,POP)=0
!			CALL POOL_GENO_DATA_SQUARE(TEMP_DATA_MATRIX,DATA_COORD,DATA_ALLELE1,DATA_TOTAL, & 
!						GRID_COORD,SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
!!			PRINT*, "DATA_ALLELE1"
!!			CALL PRINT_INT_MATRIX(DATA_ALLELE1(1:8,1:8))
!!			PRINT*, "DATA_TOTAL"
!!			CALL PRINT_INT_MATRIX(DATA_TOTAL(1:8,1:8))
!!			CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)			
!			CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!!			PRINT*, "ALLELE_FREQUENCY MATRIX"
!!			CALL PRINT_MATRIX(ALLELE_FREQUENCY(1:8,1:8))
!			
!			!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!			!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!!			CALL UPDATE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!!								MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN)
!			DO J=1,NUMBER_UNKNOWN
!				IF(ADMIXED_MEMBERSHIP(1,J).EQ.POP) THEN
!					CALL UPDATE_SINGLE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,&
!							GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN,J,IS_LAND)
!				END IF
!			END DO
!!			PRINT*, "M1"
!		END DO				
!		!THIS CALCULATES THE DISTANCE BETWEEN THE TRUE VS. ESTIMATED LOCATIONS
!		IF((MOD(I,500).EQ.0).OR.(I.EQ.NUMBER_SNPS)) THEN
!			SUM_DISTANCE=0.
!			TEMP_DISTANCE=0.
!			DO J=1,NUMBER_UNKNOWN !should be LINES
!				CALL FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID(:,:,J),TEMP_REAL1,TEMP_REAL2,&
!						MAX_GRID_LENGTH,GRID_COORD,IS_LAND)
!				TEMP_DISTANCE=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),DATA_COORD(:,ADMIXED_MEMBERSHIP(1,J)))
!				SUM_DISTANCE=SUM_DISTANCE+TEMP_DISTANCE
!				!WRITES ALL OF THE UNKNOWN'S ESTIMATED LOCATIONS
!!				"Individual","NumberSNPs","Membership","Long","Lat","Log(P)"
!				WRITE(OUTPUT_UNIT3,'(I0,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F12.6)') J,&
!						I,ADMIXED_MEMBERSHIP(1,J),TEMP_REAL1,TEMP_REAL2,MAXVAL(UNKNOWN_GRID(:,:,J))
!			END DO
!			AVG_DISTANCE=SUM_DISTANCE/(NUMBER_UNKNOWN+0.)
!			PRINT*, "LOCI: ", I, "AVG_DISTANCE: ", AVG_DISTANCE, "RHO: ",RHO, "POP: ",POP
!					
!			WRITE(OUTPUT_UNIT4,'(I0,3X,A,3X,F10.4)') I,"LeaveOnePopOut",AVG_DISTANCE
!
!		END IF
!		
!!		IF((MOD(I,250).EQ.0).OR.(I.EQ.NUMBER_SNPS)) THEN
!!			!WRITES A FEW UNKNOWN HEAT MAPS
!!			CALL WRITE_UNKNOWN_HEATMAP_SQUARE(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!!				DATA_COORD,DATA_MATRIX(:,:,I),ADMIXED_MEMBERSHIP,SAMPLE_SITES,I,&
!!				GRID_LENGTH,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT2)
!!		END IF
!!		
!!		
!!		IF(I.LE.500) THEN
!!			CALL WRITE_ALLELE_SURFACE_SQUARE(ALLELE_FREQUENCY,GRID_COORD,&
!!				DATA_COORD,DATA_MATRIX(:,:,I),SAMPLE_SITES,I,&
!!				GRID_LENGTH,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!!		END IF
!		
!	END DO
!	
!	CLOSE(OUTPUT_UNIT)
!	CLOSE(OUTPUT_UNIT2)
!	CLOSE(OUTPUT_UNIT3)
!	CLOSE(OUTPUT_UNIT4)
!
!	END SUBROUTINE LEAVE_ONE_POP_OUT_SQUARE
!	
!	
!	
!	
!	
!	
!	
!	SUBROUTINE LEAVE_ONE_INDIV_OUT_CROSSVAL_SQUARE(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO,AVG_DISTANCE)
!	!THIS SUBROUTINE ESTIMATES THE RHO WEIGHT IN OUR MODEL USING CROSS VALIDATION
!		
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,L,SAMPLE_SITES,NUMBER_SNPS,MAX_GRID_LENGTH,NUMBER_UNKNOWN,TEMP_INT,ITS,OUTPUT_UNIT=13
!	INTEGER :: POP,NUMBER_SAME
!	INTEGER, DIMENSION(2) :: GRID_LENGTH
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD,RHO_VEC !DATA_COORD(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: ALLELE_FREQUENCY !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT,SUM_DISTANCE
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP,TEMP_DATA_MATRIX
!	LOGICAL, ALLOCATABLE, DIMENSION(:,:) :: IS_LAND
!	LOGICAL, ALLOCATABLE, DIMENSION(:) :: NEED_ALLELE_SURFACE
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: SAMPLE_SIZES
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: POP_AVG_DISTANCE
!	
!	GRID_LENGTH=0
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	ALLOCATE(TEMP_DATA_MATRIX(2,SAMPLE_SITES))
!	TEMP_DATA_MATRIX=0
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD_SQUARE(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!	ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
!	ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)))
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN))
!	ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!	UNKNOWN_OBS=0
!
!	CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	ALLOCATE(IS_LAND(GRID_LENGTH(1),GRID_LENGTH(2)))
!	CALL UPDATE_IS_LAND_SQUARE(IS_LAND,GRID_LENGTH,MAX_GRID_LENGTH)
!	
!	UNKNOWN_GRID=ONE/(COUNT(IS_LAND)+0.)
!	DO J=1,GRID_LENGTH(2)
!		DO I=1,GRID_LENGTH(1)
!			IF(.NOT. IS_LAND(I,J)) THEN
!				UNKNOWN_GRID(I,J,:)=ZERO
!			END IF
!		END DO
!	END DO
!	LONG_WEIGHT=ONE
!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	LAT_WEIGHT=ONE
!	
!	PRINT*, "NUMBER_UNKNOWN(1387): ",NUMBER_UNKNOWN
!	
!	ALLOCATE(SAMPLE_SIZES(SAMPLE_SITES))
!	SAMPLE_SIZES=0
!	DO I=1,NUMBER_UNKNOWN
!		SAMPLE_SIZES(ADMIXED_MEMBERSHIP(1,I))=SAMPLE_SIZES(ADMIXED_MEMBERSHIP(1,I))+1
!	END DO
!	
!	ALLOCATE(POP_AVG_DISTANCE(SAMPLE_SITES))
!	POP_AVG_DISTANCE=0.
!	
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	TEXT_FILE="CrossvalIndivSquare"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A)') "Rho","AvgDist"
!	
!	ALLOCATE(NEED_ALLELE_SURFACE(NUMBER_UNKNOWN))
!	ALLOCATE(RHO_VEC(2,10))
!	RHO_VEC=0.
!	ITS=0
!	NUMBER_SAME=0
!	DO K=1,SIZE(RHO_VEC(2,:))  
!!		IF(MOD(K,2).EQ.1) THEN
!!			ITS=ITS+1
!!			RHO=10.**(ITS-20)
!!		ELSE 
!!			RHO=5.*(10.**(ITS-20))
!!		END IF
!!		RHO=K*(1./6.)
!		RHO=10.**(K-5)
!		UNKNOWN_GRID=ONE/(COUNT(IS_LAND)+0.)
!		DO J=1,GRID_LENGTH(2)
!			DO I=1,GRID_LENGTH(1)
!				IF(.NOT. IS_LAND(I,J)) THEN
!					UNKNOWN_GRID(I,J,:)=ZERO
!				END IF
!			END DO
!		END DO
!!		CALL POOL_GENO_DATA_SQUARE(DATA_MATRIX(:,:,1),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,&
!!				SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
!!		INIT_WEIGHT=0.5
!!		CALL FIND_INITIALIZATION_WEIGHT(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,INIT_WEIGHT)
!	
!		DO I=1,NUMBER_SNPS !NEED TO CHANGE THIS TO 1,NUMBER_SNPS
!			NEED_ALLELE_SURFACE=.TRUE.
!			NUMBER_SAME=0
!			DO J=1,NUMBER_UNKNOWN
!				IF(NEED_ALLELE_SURFACE(J)) THEN
!					NEED_ALLELE_SURFACE(J)=.FALSE.
!					POP=ADMIXED_MEMBERSHIP(1,J)
!					ALLELE_FREQUENCY=0.
!					TEMP_DATA_MATRIX=DATA_MATRIX(:,:,I)
!					
!					!SUBTRACTING OFF THE UNKNOWN INDIVIDUAL'S SNPS
!					TEMP_DATA_MATRIX(1,POP)=TEMP_DATA_MATRIX(1,POP)-UNKNOWN_OBS(J,I)
!					TEMP_DATA_MATRIX(2,POP)=TEMP_DATA_MATRIX(2,POP)-(2-UNKNOWN_OBS(J,I))
!					
!					CALL POOL_GENO_DATA_SQUARE(TEMP_DATA_MATRIX,DATA_COORD,DATA_ALLELE1,DATA_TOTAL, & 
!								GRID_COORD,SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
!		!			PRINT*, "DATA_ALLELE1"
!		!			CALL PRINT_INT_MATRIX(DATA_ALLELE1(1:8,1:8))
!		!			PRINT*, "DATA_TOTAL"
!		!			CALL PRINT_INT_MATRIX(DATA_TOTAL(1:8,1:8))
!		!			CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!		!					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)			
!					CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!							LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!!					PRINT*, "ALLELE_FREQUENCY MATRIX"
!!					CALL PRINT_MATRIX(ALLELE_FREQUENCY(1:8,1:8))
!!					
!!					FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!!					THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!					CALL UPDATE_SINGLE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!										MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN,J,IS_LAND)
!					DO L=J,NUMBER_UNKNOWN
!						IF(NEED_ALLELE_SURFACE(L).AND.(POP.EQ.ADMIXED_MEMBERSHIP(1,L))) THEN
!							IF(UNKNOWN_OBS(J,I).EQ.UNKNOWN_OBS(L,I)) THEN
!								CALL UPDATE_SINGLE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!											MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN,L,IS_LAND)
!								NEED_ALLELE_SURFACE(L)=.FALSE.
!								NUMBER_SAME=NUMBER_SAME+1
!!								PRINT*, "ONE SAME FOUND! ", NUMBER_SAME
!							END IF
!						END IF
!					END DO
!!					PRINT*, "M1"				
!				END IF
!			END DO
!			PRINT*, "NUMBER OF SAME POP/SNP FOUND: ", NUMBER_SAME, " SNP: ",I
!	!					THIS CALCULATES THE DISTANCE BETWEEN THE TRUE VS. ESTIMATED LOCATIONS
!			IF((MOD(I,100).EQ.0).OR.(I.EQ.NUMBER_SNPS)) THEN
!				SUM_DISTANCE=0.
!				TEMP_DISTANCE=0.
!				DO J=1,NUMBER_UNKNOWN !should be LINES
!					CALL FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID(:,:,J),TEMP_REAL1,TEMP_REAL2,&
!							MAX_GRID_LENGTH,GRID_COORD,IS_LAND)
!					TEMP_DISTANCE=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),DATA_COORD(:,ADMIXED_MEMBERSHIP(1,J)))
!					SUM_DISTANCE=SUM_DISTANCE+TEMP_DISTANCE
!					!WRITE(OUTPUT_UNIT,*) I,J,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2,UNKNOWN_COORD(:,J)
!				END DO
!				AVG_DISTANCE=SUM_DISTANCE/(NUMBER_UNKNOWN+0.)
!				PRINT*, "LOCI: ", I, "AVG_DISTANCE: ", AVG_DISTANCE, "RHO: ",RHO, "POP: ",POP
!				IF(I.EQ.NUMBER_SNPS) THEN
!					POP_AVG_DISTANCE(POP)=AVG_DISTANCE
!					RHO_VEC(1,K)=RHO
!					RHO_VEC(2,K)=AVG_DISTANCE
!					WRITE(OUTPUT_UNIT,*), RHO,AVG_DISTANCE
!				END IF
!!						CALL WRITE_ALLELE_SURFACE(ALLELE_FREQUENCY,GRID_COORD,&
!!							DATA_COORD,DATA_MATRIX(:,:,I),SAMPLE_SITES,K,&
!!							GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!			END IF
!		END DO
!!		RHO_VEC(1,K)=RHO
!!		RHO_VEC(2,K)=AVG_DISTANCE
!	END DO
!!	PRINT*, "RHO ESTIMATION RESULTS"
!!	PRINT*, "RHO AVG_DISTANCE"
!!	CALL PRINT_MATRIX(RHO_VEC)
!	
!	TEMP_INT=MINLOC(RHO_VEC(2,:),1)
!	RHO=RHO_VEC(1,TEMP_INT)
!	AVG_DISTANCE=RHO_VEC(2,TEMP_INT)
!	
!	WRITE(OUTPUT_UNIT,*), "RHO TOTAL_AVG_DISTANCE"
!!	CALL WRITE_MATRIX(RHO_VEC,OUTPUT_UNIT)
!	CLOSE(OUTPUT_UNIT)
!	
!	END SUBROUTINE LEAVE_ONE_INDIV_OUT_CROSSVAL_SQUARE
!	
!	
!	
!	
!	
!	
!	
!	SUBROUTINE LEAVE_ONE_INDIV_OUT_SQUARE(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO,AVG_DISTANCE)
!	!THIS SUBROUTINE ESTIMATES THE RHO WEIGHT IN OUR MODEL USING CROSS VALIDATION
!		
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,L,SAMPLE_SITES,NUMBER_SNPS,MAX_GRID_LENGTH,NUMBER_UNKNOWN,TEMP_INT,ITS,OUTPUT_UNIT=12
!	INTEGER :: POP,NUMBER_SAME,OUTPUT_UNIT2=13,OUTPUT_UNIT3=14,OUTPUT_UNIT4=15
!	INTEGER, DIMENSION(2) :: GRID_LENGTH
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !DATA_COORD(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: ALLELE_FREQUENCY !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2,TEMP_REAL
!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT,SUM_DISTANCE
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP,TEMP_DATA_MATRIX
!	LOGICAL, ALLOCATABLE, DIMENSION(:,:) :: IS_LAND
!	LOGICAL, ALLOCATABLE, DIMENSION(:) :: NEED_ALLELE_SURFACE
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: SAMPLE_SIZES
!	
!	
!	GRID_LENGTH=0
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	ALLOCATE(TEMP_DATA_MATRIX(2,SAMPLE_SITES))
!	TEMP_DATA_MATRIX=0
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD_SQUARE(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!	ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
!	ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)))
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN))
!	ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!	UNKNOWN_OBS=0
!
!	CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	ALLOCATE(IS_LAND(GRID_LENGTH(1),GRID_LENGTH(2)))
!	CALL UPDATE_IS_LAND_SQUARE(IS_LAND,GRID_LENGTH,MAX_GRID_LENGTH)
!
!	UNKNOWN_GRID=ONE/(COUNT(IS_LAND)+0.)
!	DO J=1,GRID_LENGTH(2)
!		DO I=1,GRID_LENGTH(1)
!			IF(.NOT. IS_LAND(I,J)) THEN
!				UNKNOWN_GRID(I,J,:)=ZERO
!			END IF
!		END DO
!	END DO
!!	DO I=1,NUMBER_UNKNOWN
!!		TEMP_REAL=SUM(UNKNOWN_GRID(:,:,I))
!!		PRINT*, "TEMP_REAL: ", TEMP_REAL, I
!!	END DO
!	LONG_WEIGHT=ONE
!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	LAT_WEIGHT=ONE
!	
!	PRINT*, "NUMBER_UNKNOWN(1387): ",NUMBER_UNKNOWN
!	
!	ALLOCATE(SAMPLE_SIZES(SAMPLE_SITES))
!	SAMPLE_SIZES=0
!	DO I=1,NUMBER_UNKNOWN
!		SAMPLE_SIZES(ADMIXED_MEMBERSHIP(1,I))=SAMPLE_SIZES(ADMIXED_MEMBERSHIP(1,I))+1
!	END DO
!	
!!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!!	TEXT_FILE="LeaveOneOutSquare"//TRIM(TEXT_END)//".txt"
!!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A)') "Rho","AvgDist"
!	
!	WRITE(TEXT_END,'(I0)') NUMBER_SNPS
!	TEXT_FILE="AlleleSurfacesSquare"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","DataType","Long","Lat","AlleleFreq"
!	
!	TEXT_FILE="UnknownHeatMapsSquare"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT2,FILE=TEXT_FILE)
!	!TESTING HERE...CHANGE LATER
!	WRITE(OUTPUT_UNIT2,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","Individual","Long","Lat","Prob"
!	
!	TEXT_FILE="UnknownsPlottedOnMapSquare"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT3,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT3,'(A,3X,A,3X,A,3X,A,3X,A,3X,A)') "Individual","NumberSNPs","Membership","Long","Lat","Prob"
!	
!	TEXT_FILE="AvgDistancesSquare"//TRIM(TEXT_END)//".txt"
!	OPEN(OUTPUT_UNIT4,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT4,'(A,3X,A,3X,A)') "Loci","Method","AverageDistance"
!	
!	ALLOCATE(NEED_ALLELE_SURFACE(NUMBER_UNKNOWN))
!	ITS=0
!	NUMBER_SAME=0
!	
!!	CALL POOL_GENO_DATA_SQUARE(DATA_MATRIX(:,:,1),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,&
!!				SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
!!	INIT_WEIGHT=0.5
!!	CALL FIND_INITIALIZATION_WEIGHT(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!		LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,INIT_WEIGHT)
!!	
!
!	DO I=1,NUMBER_SNPS !NEED TO CHANGE THIS TO 1,NUMBER_SNPS
!		NEED_ALLELE_SURFACE=.TRUE.
!		NUMBER_SAME=0
!		DO J=1,NUMBER_UNKNOWN
!			IF(NEED_ALLELE_SURFACE(J)) THEN
!				NEED_ALLELE_SURFACE(J)=.FALSE.
!				POP=ADMIXED_MEMBERSHIP(1,J)
!				ALLELE_FREQUENCY=0.
!				TEMP_DATA_MATRIX=DATA_MATRIX(:,:,I)
!				
!				!SUBTRACTING OFF THE UNKNOWN INDIVIDUAL'S SNPS
!				TEMP_DATA_MATRIX(1,POP)=TEMP_DATA_MATRIX(1,POP)-UNKNOWN_OBS(J,I)
!				TEMP_DATA_MATRIX(2,POP)=TEMP_DATA_MATRIX(2,POP)-(2-UNKNOWN_OBS(J,I))
!				
!				CALL POOL_GENO_DATA_SQUARE(TEMP_DATA_MATRIX,DATA_COORD,DATA_ALLELE1,DATA_TOTAL, & 
!							GRID_COORD,SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
!	!			CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!	!					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)			
!				CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!						LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!
!!					FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!!					THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!				CALL UPDATE_SINGLE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!									MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN,J,IS_LAND)
!				DO L=J,NUMBER_UNKNOWN
!					IF(NEED_ALLELE_SURFACE(L).AND.(POP.EQ.ADMIXED_MEMBERSHIP(1,L))) THEN
!						IF(UNKNOWN_OBS(J,I).EQ.UNKNOWN_OBS(L,I)) THEN
!							CALL UPDATE_SINGLE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_OBS(:,I),UNKNOWN_GRID,GRID_LENGTH,&
!										MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN,L,IS_LAND)
!							NEED_ALLELE_SURFACE(L)=.FALSE.
!							NUMBER_SAME=NUMBER_SAME+1
!						END IF
!					END IF
!				END DO				
!			END IF
!		END DO
!		PRINT*, "NUMBER OF SAME POP/SNP FOUND: ", NUMBER_SAME, " SNP: ",I
!!					THIS CALCULATES THE DISTANCE BETWEEN THE TRUE VS. ESTIMATED LOCATIONS
!		IF((MOD(I,50).EQ.0).OR.(I.EQ.NUMBER_SNPS)) THEN
!			SUM_DISTANCE=0.
!			TEMP_DISTANCE=0.
!			DO J=1,NUMBER_UNKNOWN !should be LINES
!				CALL FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID(:,:,J),TEMP_REAL1,TEMP_REAL2,&
!						MAX_GRID_LENGTH,GRID_COORD,IS_LAND)
!				TEMP_DISTANCE=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),DATA_COORD(:,ADMIXED_MEMBERSHIP(1,J)))
!				SUM_DISTANCE=SUM_DISTANCE+TEMP_DISTANCE
!!				"Individual","NumberSNPs","Membership","Long","Lat","Log(P)"
!				WRITE(OUTPUT_UNIT3,'(I0,3X,I0,3X,I0,3X,F10.4,3X,F10.4,3X,F12.6)') J,&
!						I,ADMIXED_MEMBERSHIP(1,J),TEMP_REAL1,TEMP_REAL2,MAXVAL(UNKNOWN_GRID(:,:,J))
!			END DO
!			AVG_DISTANCE=SUM_DISTANCE/(NUMBER_UNKNOWN+0.)
!			PRINT*, "LOCI: ", I, "AVG_DISTANCE: ", AVG_DISTANCE, "RHO: ",RHO, "POP: ",POP
!!			IF(I.EQ.NUMBER_SNPS) THEN
!!				WRITE(OUTPUT_UNIT,*), RHO,AVG_DISTANCE
!!			END IF
!					
!			WRITE(OUTPUT_UNIT4,'(I0,3X,A,3X,F10.4)') I,"L2PenaltySquare",AVG_DISTANCE
!			
!			!WRITES A FEW UNKNOWN HEAT MAPS
!!			CALL WRITE_UNKNOWN_HEATMAP_SQUARE(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!!				DATA_COORD,DATA_MATRIX(:,:,I),ADMIXED_MEMBERSHIP,SAMPLE_SITES,I,&
!!				GRID_LENGTH,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT2)
!			
!			!WRITES ALL OF THE UNKNOWN'S ESTIMATED LOCATIONS
!!			CALL WRITE_UNKNOWN_LOCATIONS_SQUARE(ALLELE_FREQUENCY,UNKNOWN_GRID,GRID_COORD,&
!!				DATA_COORD,DATA_MATRIX(:,:,I),ADMIXED_MEMBERSHIP,IS_LAND,SAMPLE_SITES,I,&
!!				GRID_LENGTH,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT3)
!		END IF
!		
!!		IF(I.LE.500) THEN
!!			CALL WRITE_ALLELE_SURFACE_SQUARE(ALLELE_FREQUENCY,GRID_COORD,&
!!				DATA_COORD,DATA_MATRIX(:,:,I),SAMPLE_SITES,I,&
!!				GRID_LENGTH,MAX_GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!!		END IF
!		
!	END DO
!
!	CLOSE(OUTPUT_UNIT)
!	CLOSE(OUTPUT_UNIT2)
!	CLOSE(OUTPUT_UNIT3)
!	CLOSE(OUTPUT_UNIT4)
!	
!	END SUBROUTINE LEAVE_ONE_INDIV_OUT_SQUARE
!	
!	
!	
	
	
	
	
	
!	
!	
!	SUBROUTINE TEST_GENO_PARS(DATA_FILE,NUMBER_SNPS,GRID_LENGTH,RHO)
!	!THIS SUBROUTINE TESTS THE PROGRAM'S METHODS
!		
!		
!	IMPLICIT NONE
!	
!	INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,GRID_LENGTH,NUMBER_UNKNOWN,TEMP_INT,ITS,OUTPUT_UNIT=13
!	CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: DATA_COORD !DATA_COORD(LONG,LAT)
!	REAL(KIND=DBLE_PREC), DIMENSION(2,GRID_LENGTH) :: GRID_COORD !LONG,LAT
!	INTEGER, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: DATA_ALLELE1,DATA_TOTAL
!	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH,GRID_LENGTH) :: ALLELE_FREQUENCY !LONG,LAT
!	REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
!	REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP
!	REAL(KIND=DBLE_PREC) :: P1,P2,LOGLIK
!	REAL(KIND=DBLE_PREC), DIMENSION(4) :: COEFF
!	REAL(KIND=DBLE_PREC), DIMENSION(3) :: SOLS
!	LOGICAL, DIMENSION(GRID_LENGTH,GRID_LENGTH) :: IS_LAND
!	
!	CALL UPDATE_IS_LAND(IS_LAND,GRID_LENGTH)
!	
!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
!	ALLOCATE(DATA_MATRIX(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!	DATA_MATRIX=0
!	DATA_COORD=0.
!	CALL READ_GENO_DATA(DATA_FILE,DATA_MATRIX,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
!	ALLOCATE(UNKNOWN_OBS(NUMBER_UNKNOWN,NUMBER_SNPS),UNKNOWN_GRID(GRID_LENGTH,GRID_LENGTH,NUMBER_UNKNOWN))
!	ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!	UNKNOWN_OBS=0
!	UNKNOWN_GRID=0.
!	CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)
!	
!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
!	LONG_WEIGHT=ONE
!!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!	LAT_WEIGHT=ONE
!
!	TEXT_FILE="AlleleSurfacesTest.txt"
!	OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,'(A,3X,A,3X,A,3X,A,3X,A)') "Loci","DataType","Long","Lat","AlleleFreq"
!	
!	ITS=0
!	RHO=10.**(-0)
!	UNKNOWN_GRID=0.
!	CALL POOL_GENO_DATA(DATA_MATRIX(:,:,1),DATA_COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,SAMPLE_SITES,GRID_LENGTH)
!!			CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!!					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!	DATA_ALLELE1=0
!	DATA_TOTAL=0
!	DATA_ALLELE1(1,1)=9
!	DATA_TOTAL(1,1)=10
!	DATA_ALLELE1(GRID_LENGTH,GRID_LENGTH)=1
!	DATA_TOTAL(GRID_LENGTH,GRID_LENGTH)=10
!!	DATA_ALLELE1(GRID_LENGTH,1)=1
!!	DATA_TOTAL(GRID_LENGTH,1)=10
!!	DATA_ALLELE1(1,GRID_LENGTH)=0
!!	DATA_TOTAL(1,GRID_LENGTH)=10
!	PRINT*, "DATA_ALLELE1"
!	CALL PRINT_INT_MATRIX(DATA_ALLELE1)
!	PRINT*, "DATA_TOTAL"
!	CALL PRINT_INT_MATRIX(DATA_TOTAL)
!	CALL INITIALIZE_ALLELE_FREQUENCY(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,GRID_LENGTH)
!	PRINT*, "INITIAL ALLELE FREQUENCY"
!	CALL PRINT_MATRIX(ALLELE_FREQUENCY)
!	LOGLIK=CALC_LOGLIK_L2(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!		LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!	PRINT*, "LOGLIK=",LOGLIK
!	
!	DO I=1,10
!		PRINT*, "ITERATION: ", I
!		CALL UPDATE_L2_PARS_ONCE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!						LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!						
!		CALL PRINT_MATRIX(ALLELE_FREQUENCY(1:8,1:8))
!		PRINT*, ALLELE_FREQUENCY(1,1), ALLELE_FREQUENCY(1,2), ALLELE_FREQUENCY(1,3), ALLELE_FREQUENCY(1,4)
!		
!	END DO
!	
!	DO I=1,10000
!		CALL UPDATE_L2_PARS_ONCE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!						LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!	END DO
!	PRINT*, "ITERATION: 10000"
!	CALL PRINT_MATRIX(ALLELE_FREQUENCY(1:8,1:8))
!	PRINT*, ALLELE_FREQUENCY(1,1), ALLELE_FREQUENCY(1,2), ALLELE_FREQUENCY(1,3), ALLELE_FREQUENCY(1,4)
!		
!	PRINT*, "L2 MODEL"
!	CALL FIT_L2_MODEL(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!	
!	PRINT*, "GENO QNEWT"
!	CALL FIT_L2_MODEL_QNEWT(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!	
!	CALL PRINT_MATRIX(ALLELE_FREQUENCY(1:8,1:8))
!	
!	PRINT*, "ONE UPDATE"
!	CALL UPDATE_L2_PARS_ONCE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!						LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH)
!						
!	CALL PRINT_MATRIX(ALLELE_FREQUENCY(1:8,1:8))
!	
!	
!
!	
!	!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
!	!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!!	CALL UPDATE_UNKNOWN_GRID(ALLELE_FREQUENCY,UNKNOWN_OBS(:,1),UNKNOWN_GRID,GRID_LENGTH,&
!!						SAMPLE_SITES,NUMBER_UNKNOWN)
!!			PRINT*, "M1"				
!	!THIS CALCULATES THE DISTANCE BETWEEN THE TRUE VS. ESTIMATED LOCATIONS
!	IF(.FALSE.) THEN
!		AVG_DISTANCE=0.
!		TEMP_DISTANCE=0.
!		DO J=1,NUMBER_UNKNOWN !should be LINES
!			CALL FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID(:,:,J),TEMP_REAL1,TEMP_REAL2,&
!					GRID_LENGTH,GRID_COORD,IS_LAND)
!			TEMP_DISTANCE=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),DATA_COORD(:,ADMIXED_MEMBERSHIP(1,J)))
!			AVG_DISTANCE=AVG_DISTANCE+TEMP_DISTANCE
!			!WRITE(OUTPUT_UNIT,*) I,J,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2,UNKNOWN_COORD(:,J)
!		END DO
!		AVG_DISTANCE=AVG_DISTANCE/NUMBER_UNKNOWN
!		PRINT*, "LOCI, AVG_DISTANCE ",  AVG_DISTANCE, RHO
!		
!		CALL WRITE_ALLELE_SURFACE(ALLELE_FREQUENCY,GRID_COORD,&
!			DATA_COORD,DATA_MATRIX(:,:,1),SAMPLE_SITES,K,&
!			GRID_LENGTH,NUMBER_UNKNOWN,OUTPUT_UNIT)
!	END IF
!
!	CLOSE(OUTPUT_UNIT)
!	
!	P1=0.64
!	P2=0.37
!	PRINT*, "P1: ",P1, " P2: ", P2
!
!	
!	COEFF=(/3.5,7.2,5.6,6.6/)
!	SOLS=0.
!	CALL CUBIC_SOLVER(COEFF,SOLS,K)
!	PRINT*, "NUM_SOLS, SOLS", K, SOLS
!	
!	
!	END SUBROUTINE TEST_GENO_PARS
!	
	
END MODULE ORIGENMOD
!END ORIGEN MODULE ---------------------------------------------------------------------










MODULE MULTINOMIAL

	USE ALLELEFREQUENCIES
	USE QUASINEWTON
	USE CONSTANTS
	USE ORIGENMOD
	
	IMPLICIT NONE
	
	CONTAINS
	

	SUBROUTINE INITIALIZE_MULTI_FREQUENCY(DATA_MULTI,MULTI_FREQUENCY,&
					GRID_LENGTH,MAX_GRID_LENGTH,ALLELES)
					
	IMPLICIT NONE
	
	INTEGER :: H,I,J,K,MAX_GRID_LENGTH,ALLELES
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_MULTI
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: MULTI_FREQUENCY
	
	INTEGER :: NON_ZERO
	INTEGER, DIMENSION(2,GRID_LENGTH(1)*GRID_LENGTH(2)) :: DATA_LOC
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1)*GRID_LENGTH(2),ALLELES) :: DATA_MAT
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1)*GRID_LENGTH(2)) :: WEIGHTS
	
!	PRINT*, "BEGIN INIT"
					
!	AVG_FREQ=(ZERO+SUM(DATA_ALLELE1))/SUM(DATA_TOTAL)
	
	!COUNT THE NUMBER OF NON_ZERO CELLS
	NON_ZERO=0
	DATA_LOC=0
	DATA_MAT=0.
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			IF(SUM(DATA_MULTI(:,I,J))>0) THEN
				NON_ZERO=NON_ZERO+1
				DO H=1,ALLELES
					DATA_MAT(NON_ZERO,H)=(DATA_MULTI(H,I,J)+PSEUDO_COUNT)/&
						(SUM(DATA_MULTI(:,I,J))+PSEUDO_COUNT*ALLELES)
				END DO
				DATA_LOC(1,NON_ZERO)=I
				DATA_LOC(2,NON_ZERO)=J
			END IF
		END DO
	END DO
	
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			IF(SUM(DATA_MULTI(:,I,J))>0) THEN
				DO H=1,ALLELES
					!INITIAL VALUE IS THE MLE FOR THE LOCATION
					MULTI_FREQUENCY(H,I,J)=(DATA_MULTI(H,I,J)+PSEUDO_COUNT)/&
						(SUM(DATA_MULTI(:,I,J))+PSEUDO_COUNT*ALLELES)
				END DO
			ELSE
				!INITIAL VALUE IS THE VALUE OBTAINED FROM INVERSE DISTANCE WEIGHTING
				WEIGHTS=0
				DO K=1,NON_ZERO
					WEIGHTS(K)=ONE/SQRT((I-DATA_LOC(1,K))**2+(J-DATA_LOC(2,K))**2+ZERO)
				END DO
				DO H=1,ALLELES
					MULTI_FREQUENCY(H,I,J)=(DOT_PRODUCT(WEIGHTS,DATA_MAT(:,H))/SUM(WEIGHTS))
				END DO
			END IF
		END DO
	END DO
	
!!!MAKE SURE THAT THE NEW ESTIMATES ARE IN (0,1) AND SUM TO 1
!	DO J=1,GRID_LENGTH(2)
!		DO I=1,GRID_LENGTH(1)
!			DO H=1,ALLELES
!				IF(MULTI_FREQUENCY(H,I,J).LE.0) THEN
!					PRINT*, "INIT FAILED"
!					STOP
!				ELSE IF(MULTI_FREQUENCY(H,I,J).GE.1) THEN
!					PRINT*, "INIT FAILED"
!					STOP
!				END IF
!			END DO
!			IF(ABS(SUM(MULTI_FREQUENCY(:,I,J))-1).GE.1.0E-9) THEN
!				PRINT*, "INIT FAILED"
!				STOP
!			END IF
!		END DO
!	END DO
	
!	PRINT*, "SURFACE INITIALIZED"
	
	END SUBROUTINE INITIALIZE_MULTI_FREQUENCY





!
!	SUBROUTINE UPDATE_MULTI_PARS_ONCE_DO_NOT_USE(DATA_MULTI,MULTI_FREQUENCY,RHO,&
!					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,ALLELES)
!	!THIS SUBROUTINE UPDATES THE ALLELE_FREQUENCY MATRIX ONCE USING MM UPDATES
!	!THIS ONE GIVES DECREASING LOGLIKELIHOODS FOR SOME REASON...
!	
!	IMPLICIT NONE
!	
!	INTEGER :: H,I,J,K,L,MAX_GRID_LENGTH,ALLELES,ITERATIONS
!	INTEGER, DIMENSION(2) :: GRID_LENGTH
!	INTEGER, DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_MULTI
!	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: MULTI_FREQUENCY, TEMP_MULTI_FREQUENCY
!	REAL(KIND=DBLE_PREC) :: RHO,DIAGWEIGHT,LONG_WEIGHT,LAT_WEIGHT,SUMWEIGHTS,TEMP_BETA,BETA1,BETA2,TEMP_SUM
!	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,3,3) :: MINI_FREQUENCY
!	REAL(KIND=DBLE_PREC), DIMENSION(3,3) :: MINI_WEIGHTS
!	REAL(KIND=DBLE_PREC) :: TEST_FUNC,TEMP_SUM2
!	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES) :: BETA_REALS
!	LOGICAL :: DATA_LOGICAL
!	
!!	PRINT*, "BEGIN UPDATE MULTI PARS ONCE"
!	
!	DIAGWEIGHT=ONE/SQRT((ONE/LONG_WEIGHT)**2+(ONE/LAT_WEIGHT)**2)
!	TEMP_MULTI_FREQUENCY=MULTI_FREQUENCY
!		
!	DO J=1,GRID_LENGTH(2)
!!		PRINT*, "BEGIN J: ", J
!		DO I=1,GRID_LENGTH(1)
!			IF(SUM(DATA_MULTI(:,I,J))>0) THEN
!				DATA_LOGICAL=.TRUE.
!			ELSE
!				DATA_LOGICAL=.FALSE.
!			END IF
!			MINI_FREQUENCY=0
!			MINI_WEIGHTS=0
!			DO H=1,ALLELES
!!				MINI_FREQUENCY(H,2,2)=TEMP_MULTI_FREQUENCY(H,I,J)
!				IF(I>1) THEN
!	!				PRINT*, I,J,ALLELE_FREQUENCY(I-1,J)
!					MINI_FREQUENCY(H,1,2)=TEMP_MULTI_FREQUENCY(H,I-1,J)
!					MINI_WEIGHTS(1,2)=LONG_WEIGHT
!					IF(J>1) THEN
!						MINI_FREQUENCY(H,1,1)=TEMP_MULTI_FREQUENCY(H,I-1,J-1)
!						MINI_WEIGHTS(1,1)=DIAGWEIGHT
!					END IF
!					IF(J<GRID_LENGTH(2)) THEN
!						MINI_FREQUENCY(H,1,3)=TEMP_MULTI_FREQUENCY(H,I-1,J+1)
!						MINI_WEIGHTS(1,3)=DIAGWEIGHT
!					END IF
!				END IF
!				IF(I<GRID_LENGTH(1)) THEN
!					MINI_FREQUENCY(H,3,2)=TEMP_MULTI_FREQUENCY(H,I+1,J)
!					MINI_WEIGHTS(3,2)=LONG_WEIGHT
!					IF(J>1) THEN
!						MINI_FREQUENCY(H,3,1)=TEMP_MULTI_FREQUENCY(H,I+1,J-1)
!						MINI_WEIGHTS(3,1)=DIAGWEIGHT
!					END IF
!					IF(J<GRID_LENGTH(2)) THEN
!						MINI_FREQUENCY(H,3,3)=TEMP_MULTI_FREQUENCY(H,I+1,J+1)
!						MINI_WEIGHTS(3,3)=DIAGWEIGHT
!					END IF
!				END IF
!				IF(J>1) THEN
!					MINI_FREQUENCY(H,2,1)=TEMP_MULTI_FREQUENCY(H,I,J-1)
!					MINI_WEIGHTS(2,1)=LAT_WEIGHT
!				END IF
!				IF(J<GRID_LENGTH(2)) THEN
!					MINI_FREQUENCY(H,2,3)=TEMP_MULTI_FREQUENCY(H,I,J+1)
!					MINI_WEIGHTS(2,3)=LAT_WEIGHT
!				END IF
!				
!				IF(IEEE_IS_NAN(SUM(MINI_WEIGHTS)).OR.IEEE_IS_NAN(SUM(MINI_FREQUENCY))) THEN
!					CALL INTPR("SOMETHING WENT WRONG...NAN WEIGHT SUM",-1,1,0)
!					CALL INTPR("RESULTS ARE SPURIOUS",-1,1,0)
!					STOP
!				END IF
!				
!			END DO !H-ALLELES
!			MINI_WEIGHTS=RHO*MINI_WEIGHTS
!			
!			
!			!CALCULATING BETA_I USING BISECTION METHOD
!			SUMWEIGHTS=SUM(MINI_WEIGHTS(:,:))
!			BETA1=-4*SUMWEIGHTS/(ALLELES+ZERO)
!!			PRINT*, "BETA1: ", BETA1
!			BETA2=0.5
!			TEMP_BETA=BETA2
!			ITERATIONS=0
!			TEST_FUNC=1.0
!			
!			!CAN PRECALC MORE NON-BETA ITEMS...
!			!CALC THE THINGS UNDER THE SQRT
!			BETA_REALS=0.
!			IF(DATA_LOGICAL) THEN
!				DO H=1,ALLELES
!					BETA_REALS(H)=8.0*SUMWEIGHTS*(DATA_MULTI(H,I,J)+PSEUDO_COUNT+&
!						TEMP_MULTI_FREQUENCY(H,I,J)*SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:)))
!				END DO
!			ELSE 
!				DO H=1,ALLELES
!					BETA_REALS(H)=8.0*SUMWEIGHTS*(&
!						TEMP_MULTI_FREQUENCY(H,I,J)*SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:)))
!				END DO
!			END IF
!			
!			!LOOKING FOR A NEGATIVE VALUE... A HIGH ENOUGH BETA SHOULD GET A NEGATIVE
!			DO WHILE((TEST_FUNC.GE.1.0E-10).AND.(ITERATIONS.LE.400))
!				IF(SUM(MINI_FREQUENCY)>10) THEN
!					PRINT*, "MINI_WEIGHTS: ", MINI_WEIGHTS
!					PRINT*, "SUMWEIGHTS: ", SUMWEIGHTS
!					PRINT*, "ITERATIONS: ", ITERATIONS
!					PRINT*, "TEMP_BETA: ", TEMP_BETA
!					PRINT*, "DATA_MULTI: ",DATA_MULTI(1:ALLELES,I,J)
!					PRINT*, "SUM(MINI_FREQUENCY): ", SUM(MINI_FREQUENCY)
!					PRINT*, "MINI_FREQUENCY: ", I, J
!					DO H=1,ALLELES
!						DO K=1,3
!							PRINT*, (MINI_FREQUENCY(H,K,L),L=1,3)
!						END DO
!						PRINT*, H
!					END DO
!					PRINT*, MULTI_FREQUENCY(3,:,:) 
!					PRINT*, "SOMETHING IS WRONG..."
!					STOP
!				END IF
!				ITERATIONS=ITERATIONS+1
!				BETA2=TEMP_BETA
!!				TEMP_BETA=BETA2*2
!				
!				TEMP_BETA=BETA1+2**(ITERATIONS-1)
!				TEMP_SUM=0.
!				TEMP_SUM2=0.
!				
!				DO H=1,ALLELES
!					TEMP_SUM=TEMP_SUM+SQRT(TEMP_BETA**2+BETA_REALS(H))
!				END DO
!				!!THIS HAS BEEN REPLACED WITH THE ABOVE
!!				IF(DATA_LOGICAL) THEN
!!					DO H=1,ALLELES
!!						TEMP_SUM=TEMP_SUM+SQRT(TEMP_BETA**2+8.0*SUMWEIGHTS*(DATA_MULTI(H,I,J)+PSEUDO_COUNT+&
!!							TEMP_MULTI_FREQUENCY(H,I,J)*SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:))))
!!						TEMP_SUM2=TEMP_SUM2+SQRT(TEMP_BETA**2+BETA_REALS(H))
!!						PRINT*, "TEMP_SUM DIFFS: ", TEMP_SUM-TEMP_SUM2
!!					END DO
!!				ELSE 
!!					DO H=1,ALLELES
!!						TEMP_SUM=TEMP_SUM+SQRT(TEMP_BETA**2+8.0*SUMWEIGHTS*(&
!!							TEMP_MULTI_FREQUENCY(H,I,J)*SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:))))
!!						TEMP_SUM2=TEMP_SUM2+SQRT(TEMP_BETA**2+BETA_REALS(H))
!!						PRINT*, "TEMP_SUM DIFFS: ", TEMP_SUM-TEMP_SUM2
!!					END DO
!!				END IF
!				!!END OF REPLACEMENT.
!
!!				PRINT*, "TEMP_SUM: ", TEMP_SUM
!				TEST_FUNC=-4.0*SUMWEIGHTS-ALLELES*TEMP_BETA+TEMP_SUM
!!				PRINT*, "TEMP_BETA: ", TEMP_BETA, " TEST_FUNC: ",TEST_FUNC
!			END DO
!!			PRINT*, "ITERATIONS AFTER BETA UPDATE: ", ITERATIONS
!			IF(ITERATIONS.GE.400) THEN
!				PRINT*, "ITERATIONS IN STEP DOUBLEING MAXED... NEGATIVE FUNCTION NOT FOUND"
!				STOP
!			ELSE IF(ITERATIONS.EQ.1) THEN
!				BETA2=TEMP_BETA
!			ELSE
!				BETA1=BETA2
!				BETA2=TEMP_BETA
!			END IF
!			
!			!BEGIN BISECTION METHOD
!			ITERATIONS=0
!			DO WHILE((ABS(TEST_FUNC).GE.1E-12).AND.(ITERATIONS.LE.1000))
!				ITERATIONS=ITERATIONS+1
!				TEMP_SUM=0
!				TEMP_SUM2=0.
!				TEMP_BETA=(BETA1+BETA2)/2.0
!				
!				DO H=1,ALLELES
!					TEMP_SUM=TEMP_SUM+SQRT(TEMP_BETA**2+BETA_REALS(H))
!				END DO
!				!!THIS HAS BEEN REPLACED WITH THE ABOVE
!!				IF(DATA_LOGICAL) THEN
!!					DO H=1,ALLELES
!!						TEMP_SUM=TEMP_SUM+SQRT(TEMP_BETA**2+8.0*SUMWEIGHTS*(DATA_MULTI(H,I,J)+PSEUDO_COUNT+&
!!							TEMP_MULTI_FREQUENCY(H,I,J)*SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:))))
!!						TEMP_SUM2=TEMP_SUM2+SQRT(TEMP_BETA**2+BETA_REALS(H))
!!						PRINT*, "TEMP_SUM DIFFS: ", TEMP_SUM-TEMP_SUM2
!!					END DO
!!				ELSE 
!!					DO H=1,ALLELES
!!						TEMP_SUM=TEMP_SUM+SQRT(TEMP_BETA**2+8.0*SUMWEIGHTS*(&
!!							TEMP_MULTI_FREQUENCY(H,I,J)*SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:))))
!!						TEMP_SUM2=TEMP_SUM2+SQRT(TEMP_BETA**2+BETA_REALS(H))
!!						PRINT*, "TEMP_SUM DIFFS: ", TEMP_SUM-TEMP_SUM2
!!					END DO
!!				END IF
!				!!END REPLACEMENT...
!				
!				
!				TEST_FUNC=-4*SUMWEIGHTS-ALLELES*TEMP_BETA+TEMP_SUM
!!				PRINT*, "BETA1,BETA2: ", BETA1, BETA2
!!				PRINT*, "TEMP_BETA: ", TEMP_BETA, " TEST_FUNC: ",TEST_FUNC
!				IF(TEST_FUNC>0) THEN
!					BETA1=TEMP_BETA
!				ELSE IF(TEST_FUNC<0) THEN
!					BETA2=TEMP_BETA
!				ELSE
!					BETA1=TEMP_BETA
!					BETA2=TEMP_BETA
!				END IF
!			END DO
!!			PRINT*, "TEMP_BETA: ", TEMP_BETA, " TEST_FUNC: ",TEST_FUNC
!			
!			IF(ITERATIONS.GE.1000) THEN
!				PRINT*, "ITERATIONS IN BISECTION METHOD MAXED... ROOT NOT FOUND"
!				STOP
!			END IF
!			
!			!UPDATING MULTI_FREQUENCIES
!			DO H=1,ALLELES
!				MULTI_FREQUENCY(H,I,J)=(-TEMP_BETA+SQRT(TEMP_BETA**2+BETA_REALS(H)))/(4.0*SUMWEIGHTS)
!			END DO
!!			PRINT*, "MULTI_FREQUENCY: ", MULTI_FREQUENCY(:,I,J), SUM(MULTI_FREQUENCY(:,I,J))
!			
!			!!!THIS HAS BEEN REPLACED WITH THE ABOVE
!!			IF(DATA_LOGICAL) THEN
!!				DO H=1,ALLELES
!!					MULTI_FREQUENCY(H,I,J)= (-TEMP_BETA+&
!!						SQRT(TEMP_BETA**2 +8*SUMWEIGHTS*(DATA_MULTI(H,I,J)+PSEUDO_COUNT+TEMP_MULTI_FREQUENCY(H,I,J)*&
!!						SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:)))))/(4.0*SUMWEIGHTS)
!!					TEMP_SUM2=(-TEMP_BETA+SQRT(TEMP_BETA**2+BETA_REALS(H)))/(4.0*SUMWEIGHTS)
!!					PRINT*, "MULTI_FREQ-TEMP_SUM2: ", MULTI_FREQUENCY(H,I,J)-TEMP_SUM2
!!				END DO
!!			ELSE
!!				DO H=1,ALLELES
!!					MULTI_FREQUENCY(H,I,J)= (-TEMP_BETA+&
!!						SQRT(TEMP_BETA**2 +8*SUMWEIGHTS*(TEMP_MULTI_FREQUENCY(H,I,J)*&
!!						SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:)))))/(4.0*SUMWEIGHTS)
!!					TEMP_SUM2=(-TEMP_BETA+SQRT(TEMP_BETA**2+BETA_REALS(H)))/(4.0*SUMWEIGHTS)
!!					PRINT*, "MULTI_FREQ-TEMP_SUM2: ", MULTI_FREQUENCY(H,I,J)-TEMP_SUM2
!!				END DO
!!			END IF
!			!!END REPLACEMENT
!			
!			
!			
!!			PRINT*, "MULTI_FREQUENCY SUMS: ", SUM(MULTI_FREQUENCY)-GRID_LENGTH(1)*GRID_LENGTH(2)
!!			PRINT*, "SUM MINIWEIGHTS*MINIFREQ", SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(1,:,:))
!!			PRINT*, "I: ", I, " J: ", J
!!			PRINT*, "MINIWEIGHTS: "
!!			DO H=1,3
!!				PRINT*, (MINI_WEIGHTS(H,K),K=1,3)
!!			END DO
!!			PRINT*, "MINI_FREQ: "
!!			DO H=1,3
!!				PRINT*, (MINI_FREQUENCY(1,H,K),K=1,3)
!!			END DO
!!			PRINT*, "MINI_WEIGHTS*MINI_FREQUENCY: ", MINI_WEIGHTS(:,:)*MINI_FREQUENCY(1,:,:)
!!			PRINT*, "TEMP_MULTI_FREQ: ", TEMP_MULTI_FREQUENCY(1,:,:)
!			IF(ABS(SUM(MULTI_FREQUENCY(:,I,J))-1)>1.E-5) THEN
!				PRINT*, "DOESN'T SUM TO 1: ", ABS(SUM(MULTI_FREQUENCY(:,I,J))-1), I,J
!			END IF
!		END DO
!	END DO
!	
!	
!	
!	!MAKE SURE THAT THE NEW ESTIMATES ARE IN (0,1) AND SUM TO 1
!!	DO J=1,GRID_LENGTH(2)
!!		DO I=1,GRID_LENGTH(1)
!!			DO H=1,ALLELES
!!				IF(MULTI_FREQUENCY(H,I,J).LE.0) THEN
!!					MULTI_FREQUENCY(H,I,J)=TEMP_MULTI_FREQUENCY(H,I,J)*0.1
!!				ELSE IF(MULTI_FREQUENCY(H,I,J).GE.1) THEN
!!					MULTI_FREQUENCY(H,I,J)=(TEMP_MULTI_FREQUENCY(H,I,J)+9.0)/10.0
!!				END IF
!!			END DO
!!			TEMP_BETA=SUM(MULTI_FREQUENCY(:,I,J))
!!			DO H=1,ALLELES
!!				MULTI_FREQUENCY(H,I,J)=MULTI_FREQUENCY(H,I,J)/TEMP_BETA
!!			END DO
!!		END DO
!!	END DO
!	
!	
!	
!	
!!	PRINT*, "MULTI_FREQUENCY SUMS: ", SUM(MULTI_FREQUENCY)-GRID_LENGTH(1)*GRID_LENGTH(2)
!!	PRINT*, "SUM MINIWEIGHTS*MINIFREQ", SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(1,:,:))
!!	PRINT*, "MINIWEIGHTS: "
!!	DO I=1,3
!!		PRINT*, (MINI_WEIGHTS(I,J),J=1,3)
!!	END DO
!!	PRINT*, "MINI_FREQ: "
!!	DO I=1,3
!!		PRINT*, (MINI_FREQUENCY(1,I,J),J=1,3)
!!	END DO
!!	PRINT*, "MINI_WEIGHTS*MINI_FREQUENCY: ", MINI_WEIGHTS(:,:)*MINI_FREQUENCY(1,:,:)
!!	PRINT*, "TEMP_MULTI_FREQ: ", TEMP_MULTI_FREQUENCY(1,:,:)
!	
!	END SUBROUTINE UPDATE_MULTI_PARS_ONCE_DO_NOT_USE
!
!








	SUBROUTINE UPDATE_MULTI_PARS_ONCE2(DATA_MULTI,MULTI_FREQUENCY,RHO,&
					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,ALLELES)
	!THIS SUBROUTINE UPDATES THE ALLELE_FREQUENCY MATRIX ONCE USING MM UPDATES (A DIFFERENT SET OF UPDATES)
	
	IMPLICIT NONE
	
	INTEGER :: H,I,J,K,L,MAX_GRID_LENGTH,ALLELES,ITERATIONS
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_MULTI
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: MULTI_FREQUENCY, TEMP_MULTI_FREQUENCY
	REAL(KIND=DBLE_PREC) :: RHO,DIAGWEIGHT,LONG_WEIGHT,LAT_WEIGHT,SUMWEIGHTS,TEMP_BETA,BETA1,BETA2,TEMP_SUM
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,3,3) :: MINI_FREQUENCY
	REAL(KIND=DBLE_PREC), DIMENSION(3,3) :: MINI_WEIGHTS
	REAL(KIND=DBLE_PREC) :: TEST_FUNC,TEMP_SUM2
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES) :: BETA_REALS,SUMS
	LOGICAL :: DATA_LOGICAL
	
!	PRINT*, "BEGIN UPDATE MULTI PARS ONCE"
	
	DIAGWEIGHT=ONE/SQRT((ONE/LONG_WEIGHT)**2+(ONE/LAT_WEIGHT)**2)
	TEMP_MULTI_FREQUENCY=MULTI_FREQUENCY
		
	DO J=1,GRID_LENGTH(2)
!		PRINT*, "BEGIN J: ", J
		DO I=1,GRID_LENGTH(1)
			IF(SUM(DATA_MULTI(:,I,J))>0) THEN
				DATA_LOGICAL=.TRUE.
			ELSE
				DATA_LOGICAL=.FALSE.
			END IF
			MINI_FREQUENCY=0
			MINI_WEIGHTS=0
			DO H=1,ALLELES
!				MINI_FREQUENCY(H,2,2)=TEMP_MULTI_FREQUENCY(H,I,J)
				IF(I>1) THEN
	!				PRINT*, I,J,ALLELE_FREQUENCY(I-1,J)
					MINI_FREQUENCY(H,1,2)=TEMP_MULTI_FREQUENCY(H,I-1,J)
					MINI_WEIGHTS(1,2)=LONG_WEIGHT
					IF(J>1) THEN
						MINI_FREQUENCY(H,1,1)=TEMP_MULTI_FREQUENCY(H,I-1,J-1)
						MINI_WEIGHTS(1,1)=DIAGWEIGHT
					END IF
					IF(J<GRID_LENGTH(2)) THEN
						MINI_FREQUENCY(H,1,3)=TEMP_MULTI_FREQUENCY(H,I-1,J+1)
						MINI_WEIGHTS(1,3)=DIAGWEIGHT
					END IF
				END IF
				IF(I<GRID_LENGTH(1)) THEN
					MINI_FREQUENCY(H,3,2)=TEMP_MULTI_FREQUENCY(H,I+1,J)
					MINI_WEIGHTS(3,2)=LONG_WEIGHT
					IF(J>1) THEN
						MINI_FREQUENCY(H,3,1)=TEMP_MULTI_FREQUENCY(H,I+1,J-1)
						MINI_WEIGHTS(3,1)=DIAGWEIGHT
					END IF
					IF(J<GRID_LENGTH(2)) THEN
						MINI_FREQUENCY(H,3,3)=TEMP_MULTI_FREQUENCY(H,I+1,J+1)
						MINI_WEIGHTS(3,3)=DIAGWEIGHT
					END IF
				END IF
				IF(J>1) THEN
					MINI_FREQUENCY(H,2,1)=TEMP_MULTI_FREQUENCY(H,I,J-1)
					MINI_WEIGHTS(2,1)=LAT_WEIGHT
				END IF
				IF(J<GRID_LENGTH(2)) THEN
					MINI_FREQUENCY(H,2,3)=TEMP_MULTI_FREQUENCY(H,I,J+1)
					MINI_WEIGHTS(2,3)=LAT_WEIGHT
				END IF
				
				IF(IEEE_IS_NAN(SUM(MINI_WEIGHTS)).OR.IEEE_IS_NAN(SUM(MINI_FREQUENCY))) THEN
					CALL INTPR("SOMETHING WENT WRONG...NAN WEIGHT SUM",-1,1,0)
					CALL INTPR("RESULTS ARE SPURIOUS",-1,1,0)
!					STOP
				END IF
				
			END DO !H-ALLELES
			MINI_WEIGHTS=RHO*MINI_WEIGHTS
			
			
			!CALCULATING BETA_I USING BISECTION METHOD
			SUMWEIGHTS=SUM(MINI_WEIGHTS(:,:))
			BETA1=-4*SUMWEIGHTS/(ALLELES+ZERO)
!			PRINT*, "BETA1: ", BETA1
			BETA2=0.5
			TEMP_BETA=BETA2
			ITERATIONS=0
			TEST_FUNC=1.0
			
			!CAN PRECALC MORE NON-BETA ITEMS...
			!CALC THE THINGS UNDER THE SQRT
			BETA_REALS=ZERO
			SUMS=ZERO
			IF(DATA_LOGICAL) THEN
				DO H=1,ALLELES
					BETA_REALS(H)=16.0*SUMWEIGHTS*(DATA_MULTI(H,I,J)+PSEUDO_COUNT)
				END DO
			END IF
			DO H=1,ALLELES
				SUMS(H)=TEMP_MULTI_FREQUENCY(H,I,J)*SUMWEIGHTS+SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:))
			END DO
			
			!LOOKING FOR A NEGATIVE VALUE... A HIGH ENOUGH BETA SHOULD GET A NEGATIVE
			DO WHILE((TEST_FUNC.GE.1.0E-10).AND.(ITERATIONS.LE.400))
				IF(SUM(MINI_FREQUENCY)>10) THEN
!					PRINT*, "MINI_WEIGHTS: ", MINI_WEIGHTS
!					PRINT*, "SUMWEIGHTS: ", SUMWEIGHTS
!					PRINT*, "ITERATIONS: ", ITERATIONS
!					PRINT*, "TEMP_BETA: ", TEMP_BETA
!					PRINT*, "DATA_MULTI: ",DATA_MULTI(1:ALLELES,I,J)
!					PRINT*, "SUM(MINI_FREQUENCY): ", SUM(MINI_FREQUENCY)
!					PRINT*, "MINI_FREQUENCY: ", I, J
!					DO H=1,ALLELES
!						DO K=1,3
!							PRINT*, (MINI_FREQUENCY(H,K,L),L=1,3)
!						END DO
!						PRINT*, H
!					END DO
!					PRINT*, MULTI_FREQUENCY(3,:,:) 
!					PRINT*, "SOMETHING IS WRONG..."
!					STOP
				END IF
				ITERATIONS=ITERATIONS+1
				BETA2=TEMP_BETA
!				TEMP_BETA=BETA2*2
				
				TEMP_BETA=BETA1+2**(ITERATIONS-1)
				TEMP_SUM=0.
!				TEMP_SUM2=0.
			
				DO H=1,ALLELES
!					TEMP_SUM2=TEMP_MULTI_FREQUENCY(H,I,J)*SUMWEIGHTS+SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:))
					TEMP_SUM=TEMP_SUM+SQRT((TEMP_BETA-2*SUMS(H))**2+BETA_REALS(H))
				END DO

!				PRINT*, "TEMP_SUM: ", TEMP_SUM
				TEST_FUNC=-4.0*SUMWEIGHTS-ALLELES*TEMP_BETA+TEMP_SUM
!				PRINT*, "TEMP_BETA: ", TEMP_BETA, "  TEST_FUNC: ",TEST_FUNC, "  TEMP_SUM: ", TEMP_SUM
			END DO
!			PRINT*, "ITERATIONS AFTER BETA UPDATE: ", ITERATIONS
			IF((ITERATIONS.GE.400).OR.(IEEE_IS_NAN(TEMP_BETA))) THEN
!				PRINT*, "ITERATIONS IN STEP DOUBLING MAXED OR NAN BETA... ZERO NOT BRACKETED. PROGRAM STOPPED."
!				PRINT*, "ITERATIONS: ", ITERATIONS, "  TEMP_BETA: ", TEMP_BETA, "  TEST_FUNC: ", TEST_FUNC
				CALL INTPR("ITERATIONS IN STEP DOUBLING MAXED OR NAN BETA... ZERO NOT BRACKETED.",-1,1,0)
				CALL INTPR("CONTACT MAINTAINER...",-1,1,0)
				CALL INTPR("RESULTS ARE SPURIOUS",-1,1,0)
!				STOP
			ELSE IF(ITERATIONS.EQ.1) THEN
				BETA2=TEMP_BETA
			ELSE
				BETA1=BETA2
				BETA2=TEMP_BETA
			END IF
			
			!BEGIN BISECTION METHOD
			ITERATIONS=0
			DO WHILE((ABS(TEST_FUNC).GE.1E-7).AND.(ITERATIONS.LE.1000))
				ITERATIONS=ITERATIONS+1
				TEMP_SUM=0
				TEMP_BETA=(BETA1+BETA2)/2.0
				
				DO H=1,ALLELES
!					TEMP_SUM2=TEMP_MULTI_FREQUENCY(H,I,J)*SUMWEIGHTS+SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:))
					TEMP_SUM=TEMP_SUM+SQRT((TEMP_BETA-2*SUMS(H))**2+BETA_REALS(H))
				END DO				
				
				TEST_FUNC=-4*SUMWEIGHTS-ALLELES*TEMP_BETA+TEMP_SUM
!				PRINT*, "BETA1,BETA2: ", BETA1, BETA2
!				PRINT*, "TEMP_BETA: ", TEMP_BETA, " TEST_FUNC: ",TEST_FUNC
				IF(TEST_FUNC>0) THEN
					BETA1=TEMP_BETA
				ELSE IF(TEST_FUNC<0) THEN
					BETA2=TEMP_BETA
				ELSE
					BETA1=TEMP_BETA
					BETA2=TEMP_BETA
				END IF
			END DO
!			PRINT*, "FINAL BETA: ", TEMP_BETA, " TEST_FUNC: ", TEST_FUNC
!			PRINT*, "TEMP_BETA: ", TEMP_BETA, " TEST_FUNC: ",TEST_FUNC
			
			IF(ITERATIONS.GE.1000) THEN
				CALL INTPR("ITERATIONS IN BISECTION METHOD MAXED... ",-1,1,0)
				CALL DBLEPR("IS THE FOLLOWING CLOSE TO 0? ",-1,TEST_FUNC,0)
				CALL DBLEPR("RESULTING BETA: ",-1, TEMP_BETA,0)				
!				PRINT*, "ITERATIONS IN BISECTION METHOD MAXED... "
!				PRINT*, "IS THE FOLLOWING CLOSE TO 0? ", TEST_FUNC
!				PRINT*, "RESULTING BETA: ", TEMP_BETA
			END IF
			
			!UPDATING MULTI_FREQUENCIES
			DO H=1,ALLELES
!				TEMP_SUM2=TEMP_MULTI_FREQUENCY(H,I,J)*SUMWEIGHTS+SUM(MINI_WEIGHTS(:,:)*MINI_FREQUENCY(H,:,:))
				MULTI_FREQUENCY(H,I,J)=(-TEMP_BETA+2*SUMS(H)+SQRT((TEMP_BETA-2*SUMS(H))**2+BETA_REALS(H)))/(8.0*SUMWEIGHTS)
			END DO
!			PRINT*, "MULTI_FREQUENCY: ", MULTI_FREQUENCY(:,I,J), SUM(MULTI_FREQUENCY(:,I,J))

			IF(ABS(SUM(MULTI_FREQUENCY(:,I,J))-1)>1.E-5) THEN
				CALL DBLEPR("DOESN'T SUM TO 1: ", -1,ABS(SUM(MULTI_FREQUENCY(:,I,J))-1),0)	
!				PRINT*, "DOESN'T SUM TO 1: ", ABS(SUM(MULTI_FREQUENCY(:,I,J))-1), I,J
			END IF
		END DO
	END DO
	
	!MAKE SURE THAT THE NEW ESTIMATES ARE IN (0,1) AND SUM TO 1
!	DO J=1,GRID_LENGTH(2)
!		DO I=1,GRID_LENGTH(1)
!			DO H=1,ALLELES
!				IF(MULTI_FREQUENCY(H,I,J).LE.0) THEN
!					MULTI_FREQUENCY(H,I,J)=TEMP_MULTI_FREQUENCY(H,I,J)*0.1
!				ELSE IF(MULTI_FREQUENCY(H,I,J).GE.1) THEN
!					MULTI_FREQUENCY(H,I,J)=(TEMP_MULTI_FREQUENCY(H,I,J)+9.0)/10.0
!				END IF
!			END DO
!			TEMP_BETA=SUM(MULTI_FREQUENCY(:,I,J))
!			DO H=1,ALLELES
!				MULTI_FREQUENCY(H,I,J)=MULTI_FREQUENCY(H,I,J)/TEMP_BETA
!			END DO
!		END DO
!	END DO
	
	END SUBROUTINE UPDATE_MULTI_PARS_ONCE2







	FUNCTION CALC_LOGLIK_MULTI(DATA_MULTI,MULTI_FREQUENCY,RHO,&
					GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,LONG_WEIGHT,LAT_WEIGHT)
					
	IMPLICIT NONE
	
	INTEGER :: H,I,J,K,MAX_GRID_LENGTH,ALLELES
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_MULTI
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: MULTI_FREQUENCY
	REAL(KIND=DBLE_PREC) :: CALC_LOGLIK_MULTI, PENALTY,RHO,DIAG_WEIGHT,LONG_WEIGHT,LAT_WEIGHT,PENALTY2

	CALC_LOGLIK_MULTI=0.
	PENALTY=0.
	PENALTY2=0.
	DIAG_WEIGHT=ONE/SQRT((ONE/LONG_WEIGHT)**2+(ONE/LAT_WEIGHT)**2)
	
	!MULTINOMIAL model
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			IF(SUM(DATA_MULTI(:,I,J))>0) THEN
				DO H=1,ALLELES
					IF(MULTI_FREQUENCY(H,I,J)>0 .AND. MULTI_FREQUENCY(H,I,J)<1) THEN
						CALC_LOGLIK_MULTI=CALC_LOGLIK_MULTI+(DATA_MULTI(H,I,J)+PSEUDO_COUNT)*LOG(MULTI_FREQUENCY(H,I,J))
					ELSE
!						PRINT*, "ERROR... MULTI_FREQUENCY IS UNUSUAL...", MULTI_FREQUENCY(H,I,J)
						CALL DBLEPR("ERROR... MULTI_FREQUENCY IS UNUSUAL...", -1,MULTI_FREQUENCY(H,I,J),0)	
					END IF
				END DO
			END IF
		END DO
	END DO
!	PRINT*, "CALC_LOGLIK_MULTI: ", CALC_LOGLIK_MULTI
	
	!penalty
	DO J=1,GRID_LENGTH(2)-1
		DO I=1,GRID_LENGTH(1)
			DO H=1,ALLELES
				!BOTTOM
				PENALTY=PENALTY+LAT_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I,J+1))**2
	!			PRINT*, "PENALTY: ", PENALTY, I,J,ALLELE_FREQUENCY(I,J),ALLELE_FREQUENCY(I,J+1)
				
				!BOTTOM LEFT
				IF(I>1) THEN
					PENALTY=PENALTY+DIAG_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I-1,J+1))**2
				END IF
				
				!RIGHT(PARTIAL) AND BOTTOM RIGHT
				IF(I<GRID_LENGTH(1)) THEN
				!BOTTOM RIGHT
					PENALTY=PENALTY+DIAG_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I+1,J+1))**2
				!RIGHT SIDE !THIS IS INCOMPLETE... MISSING WHOLE BOTTOM ROW
					PENALTY=PENALTY+LONG_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I+1,J))**2
				END IF
			END DO
		END DO
	END DO
	
	!FINISHING RIGHT SIDE PENALTY(BOTTOM ROW)
	DO I=1,GRID_LENGTH(1)-1
		DO H=1,ALLELES
			PENALTY=PENALTY+LONG_WEIGHT*(MULTI_FREQUENCY(H,I,GRID_LENGTH(2))-MULTI_FREQUENCY(H,I+1,GRID_LENGTH(2)))**2
		END DO
	END DO
	
	!!ANOTHER PENALTY FUNCTION
!	DO J=1,GRID_LENGTH(2)
!		DO I=1,GRID_LENGTH(1)
!			IF(I>1) THEN
!				DO H=1,ALLELES
!					PENALTY2=PENALTY2+LONG_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I-1,J))**2
!				END DO
!			END IF
!			IF(J>1) THEN
!				DO H=1,ALLELES
!					PENALTY2=PENALTY2+LAT_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I,J-1))**2
!				END DO
!			END IF
!			IF(I<GRID_LENGTH(1)) THEN 
!				DO H=1,ALLELES
!					PENALTY2=PENALTY2+LONG_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I+1,J))**2
!				END DO
!			END IF
!			IF(J<GRID_LENGTH(2)) THEN
!				DO H=1,ALLELES
!					PENALTY2=PENALTY2+LAT_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I,J+1))**2
!				END DO
!			END IF
!			IF(I>1 .AND. J>1) THEN
!				DO H=1,ALLELES
!					PENALTY2=PENALTY2+DIAG_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I-1,J-1))**2
!				END DO
!			END IF
!			IF(I>1 .AND. J<GRID_LENGTH(2)) THEN
!				DO H=1,ALLELES
!					PENALTY2=PENALTY2+DIAG_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I-1,J+1))**2
!				END DO
!			END IF
!			IF(I<GRID_LENGTH(1) .AND. J>1) THEN
!				DO H=1,ALLELES
!					PENALTY2=PENALTY2+DIAG_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I+1,J-1))**2
!				END DO
!			END IF
!			IF(I<GRID_LENGTH(1) .AND. J<GRID_LENGTH(2)) THEN
!				DO H=1,ALLELES
!					PENALTY2=PENALTY2+DIAG_WEIGHT*(MULTI_FREQUENCY(H,I,J)-MULTI_FREQUENCY(H,I+1,J+1))**2
!				END DO
!			END IF
!		END DO
!	END DO
!	PRINT*, "PENALTY DIFF: ", 2*PENALTY-PENALTY2

!	PRINT*, "PENALTY: ", 2*RHO*PENALTY, "   CALC_LOGLIK_MULTI: ", CALC_LOGLIK_MULTI
	
	
	CALC_LOGLIK_MULTI=CALC_LOGLIK_MULTI-2*RHO*PENALTY
	
	IF(IEEE_IS_NAN(CALC_LOGLIK_MULTI)) THEN
!		PRINT*, "PENALTY ", PENALTY
		CALL INTPR("NAN LOGLIK FOUND",-1,1,0)
		CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
!		STOP
	END IF

	END FUNCTION CALC_LOGLIK_MULTI






	!Checked
	SUBROUTINE FIT_MULTI_MODEL(DATA_MULTI,MULTI_FREQUENCY,RHO,&
					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,INIT_BOOL)
	!THIS SUBROUTINE UPDATES THE ALLELE_FREQUENCY MATRIX ONCE USING MM UPDATES
	
	IMPLICIT NONE
	
	INTEGER :: H,I,J,K,MAX_GRID_LENGTH,ALLELES,ITERATIONS,ERROR_COUNT
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_MULTI
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: MULTI_FREQUENCY,OLD_MULTI_FREQ
	REAL(KIND=DBLE_PREC) :: RHO,LONG_WEIGHT,LAT_WEIGHT,LOGLIK,NEW_LOGLIK,ERROR_MAX
	LOGICAL :: INIT_BOOL,NOT_CONVERGED
	
	
	
!	PRINT*, "BEGIN FIT MULTI MODEL"
	IF(INIT_BOOL) THEN
		CALL INITIALIZE_MULTI_FREQUENCY(DATA_MULTI,MULTI_FREQUENCY,&
					GRID_LENGTH,MAX_GRID_LENGTH,ALLELES)
	END IF
	OLD_MULTI_FREQ=MULTI_FREQUENCY
	
!	PRINT*, "MULTI_FREQUENCY ", MULTI_FREQUENCY(1,1:5,1:5)
	
	LOGLIK=CALC_LOGLIK_MULTI(DATA_MULTI,MULTI_FREQUENCY,RHO,&
					GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,LONG_WEIGHT,LAT_WEIGHT)
!	PRINT*, "LOGLIK: ",  LOGLIK
	
	NOT_CONVERGED=.TRUE.
	ITERATIONS=0
	ERROR_COUNT=0
	ERROR_MAX=1
	DO WHILE(NOT_CONVERGED)
		ITERATIONS=ITERATIONS+1
		CALL UPDATE_MULTI_PARS_ONCE2(DATA_MULTI,MULTI_FREQUENCY,RHO,&
					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,ALLELES)
					
		NEW_LOGLIK=CALC_LOGLIK_MULTI(DATA_MULTI,MULTI_FREQUENCY,RHO,&
					GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,LONG_WEIGHT,LAT_WEIGHT)
!		PRINT*, "NEW_LOGLIK: ", NEW_LOGLIK
		IF(IEEE_IS_NAN(SUM(MULTI_FREQUENCY))) THEN
!			PRINT*, "ALLELE_FREQUENCY IS NAN"
			CALL INTPR("NAN ALLELE FREQUENCY FOUND",-1,1,0)
			CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
!			STOP
		END IF
		
		!MODEL CHECK
		IF(NEW_LOGLIK-LOGLIK<-1.E-4) THEN
!			PRINT*, "DECREASE IN LOGLIK2(IT,DIFF,LOGLIK,NEW_LOGLIK): ", ITERATIONS, (NEW_LOGLIK-LOGLIK), LOGLIK, NEW_LOGLIK
			ERROR_COUNT=ERROR_COUNT+1
			IF(NEW_LOGLIK-LOGLIK<ERROR_MAX) THEN
				ERROR_MAX=NEW_LOGLIK-LOGLIK
			END IF
		END IF
		
		!CHECK CONVERGENCE
!		CALL CHECK_CONVERGENCE(LOGLIK,NEW_LOGLIK,ITERATIONS,NOT_CONVERGED)
		CALL CHECK_CONVERGENCE_MATRIX(OLD_MULTI_FREQ(1,:,:),MULTI_FREQUENCY(1,:,:),ITERATIONS,NOT_CONVERGED)
		
		
!		IF((MOD(ITERATIONS,5000).EQ.0).OR.(.NOT.NOT_CONVERGED)) THEN
!			PRINT*, "ITERATIONS: ", ITERATIONS, "LOGLIK(DIFF,LOGLIK): ", (NEW_LOGLIK-LOGLIK), NEW_LOGLIK
!		END IF
		
!		PRINT*, "ITERATIONS: ", ITERATIONS, "LOGLIK(DIFF,LOGLIK): ", (NEW_LOGLIK-LOGLIK), NEW_LOGLIK
!		IF(ITERATIONS.EQ.1) THEN
!!			PRINT*, MULTI_FREQUENCY(1,1:5,1:5)-OLD_MULTI_FREQ(1,1:5,1:5)
!			PRINT*, "MAXVALdiff: ", MAXVAL(ABS(MULTI_FREQUENCY-OLD_MULTI_FREQ))
!			PRINT*, "MAXLOC: ", MAXLOC(ABS(MULTI_FREQUENCY-OLD_MULTI_FREQ))
!!			PRINT*, MULTI_FREQUENCY(3,5,3), OLD_MULTI_FREQ(3,5,3), DATA_MULTI(3,5,3), SUM(DATA_MULTI(:,5,3))
!!			PRINT*, MULTI_FREQUENCY(1:ALLELES,5,3)
!!			PRINT*, OLD_MULTI_FREQ(1:ALLELES,5,3)
!!			PRINT*, DATA_MULTI(1:ALLELES,5,3)
!		END IF
		
		IF(ITERATIONS>=20000) THEN
			CALL INTPR("REACHED MAXIMUM ITERATIONS FOR MM UPDATE!",-1,1,0)
!			PRINT*, "REACHED MAXIMUM ITERATIONS FOR MM UPDATE!" 
		END IF
		
		LOGLIK=NEW_LOGLIK
		OLD_MULTI_FREQ=MULTI_FREQUENCY
	END DO
	
	IF(ERROR_COUNT>0) THEN
!		PRINT*, "ERROR COUNT: ", ERROR_COUNT, " ERROR MAXIMUM: ", ERROR_MAX
		CALL INTPR("ERROR COUNT: ",-1,ERROR_COUNT,1)
		CALL DBLEPR("ERROR MAX: ",-1,ERROR_MAX,1)
	END IF
	
!	PRINT*, "END FIT MODEL"
	
	END SUBROUTINE FIT_MULTI_MODEL




	SUBROUTINE UPDATE_UNKNOWN_GRIDS_MULTI(DATA_MULTI,MULTI_FREQUENCY,RHO,UNKNOWN_DATA,UNKNOWN_GRIDS,&
					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,NUMBER_UNKNOWN,IS_LAND)
	!THIS SUBROUTINE UPDATES THE ALLELE_FREQUENCY MATRIX ONCE USING MM UPDATES
	
	IMPLICIT NONE
	
	INTEGER :: H,I,J,K,MAX_GRID_LENGTH,ALLELES,ITERATIONS,NUMBER_UNKNOWN
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_MULTI
	REAL(KIND=DBLE_PREC), DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: MULTI_FREQUENCY
	REAL(KIND=DBLE_PREC) :: RHO,LONG_WEIGHT,LAT_WEIGHT

	INTEGER, DIMENSION(NUMBER_UNKNOWN,2) :: UNKNOWN_DATA
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN) :: UNKNOWN_GRIDS !LONG,LAT,NUMBER_UNKNOWN
	REAL(KIND=DBLE_PREC) :: TEMP_REAL,TEMP_REAL0,TEMP_REAL1,TEMP_REAL2,TEMP_REAL3,TEMP_REAL4
	LOGICAL, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: IS_LAND
	
!	TEMP_REAL=SUM(UNKNOWN_GRID(:,:,INDIV))
!	PRINT*, "TEMP_REAL ORIGINAL: ", TEMP_REAL
!	PRINT*, "UNKNOWN_GRIDS"
!	PRINT*, UNKNOWN_GRIDS(1:5,1:5,1)
	TEMP_REAL=0.
	DO J=1,GRID_LENGTH(2)
		DO I=1,GRID_LENGTH(1)
			IF(IS_LAND(I,J)) THEN
				DO K=1,NUMBER_UNKNOWN
					IF(UNKNOWN_DATA(K,1)<=0.OR.UNKNOWN_DATA(K,2)<=0) THEN
						TEMP_REAL=1
					ELSE IF(UNKNOWN_DATA(K,1)==UNKNOWN_DATA(K,2)) THEN
						TEMP_REAL=MULTI_FREQUENCY(UNKNOWN_DATA(K,1),I,J)**2
					ELSE
						TEMP_REAL=2*MULTI_FREQUENCY(UNKNOWN_DATA(K,1),I,J)*MULTI_FREQUENCY(UNKNOWN_DATA(K,2),I,J)
					END IF
					UNKNOWN_GRIDS(I,J,K)=UNKNOWN_GRIDS(I,J,K)*TEMP_REAL
!					DO H=1,ALLELES
!						IF(UNKNOWN_DATA(K,H).EQ.2) THEN
!							TEMP_REAL=MULTI_FREQUENCY(H,I,J)**2
!						ELSE IF(UNKNOWN_DATA(K,H).EQ.1) THEN
!							TEMP_REAL=MULTI_FREQUENCY(H,I,J)*SQRT(2.0)
!						ELSE
!							TEMP_REAL=ONE
!						END IF
!						UNKNOWN_GRIDS(I,J,K)=UNKNOWN_GRIDS(I,J,K)*TEMP_REAL
!					END DO
!					PRINT*, "TEMP_REAL UNKNOWN_GRIDS: ", TEMP_REAL
!					IF(TEMP_REAL.GE.0) THEN
!						UNKNOWN_GRIDS(I,J,K)=UNKNOWN_GRIDS(I,J,K)*TEMP_REAL
!					ELSE
!!						PRINT*, "ERROR IN UPDATE_UNKNOWN_GRID. PROBABILITY LESS THAN 0.", TEMP_REAL
!						CALL INTPR("SOMETHINGS WRONG...ERROR IN UPDATE_UNKNOWN_GRID. PROBABILITY LESS THAN 0.",-1,1,0)
!						CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
!						STOP
!					END IF
				END DO
			END IF
		END DO
	END DO
	
	DO K=1,NUMBER_UNKNOWN
		TEMP_REAL=SUM(UNKNOWN_GRIDS(:,:,K))
	!	PRINT*, "TEMP_REAL: ", TEMP_REAL
		DO J=1,GRID_LENGTH(2)
			DO I=1,GRID_LENGTH(1)
				UNKNOWN_GRIDS(I,J,K)=UNKNOWN_GRIDS(I,J,K)/TEMP_REAL
			END DO
		END DO
	END DO
		
!	IF(ERROR_INT<0) THEN
!		PRINT*, "TEMP_REAL = 0 IN UPDATE_UNKNOWN_GRID...AF: ", TEMP_REAL4
!		PRINT*, "NUMBER OF ERRORS: ", ERROR_INT
!	END IF

	END SUBROUTINE UPDATE_UNKNOWN_GRIDS_MULTI




	
		
	SUBROUTINE POOL_MULTI_DATA(DATA_MULTI,DATA_MAT,COORD,SAMPLE_SITES,&
					GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,GRID_COORD)
	!THIS SUBROUTINE TAKES IN DATA AND POOLS THE INFORMATION INTO A MATRIX(OF MOSTLY ZEROES)
	!FOR ANALYSIS.
	
	IMPLICIT NONE
	
	INTEGER :: H,I,J,K,MAX_GRID_LENGTH,ALLELES,ITERATIONS,SAMPLE_SITES
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_MULTI
		
	INTEGER :: LOC1,LOC2
	INTEGER, DIMENSION(ALLELES,SAMPLE_SITES) :: DATA_MAT !(FIRST IS ALLELE1, SECOND IS ALLELE2)
	!DATA_VECTOR IS A DATA MATRIX GIVING THE NUMBER OF EACH ALLELE FOUND AT EACH SITE....
	REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: COORD !(LONG,LAT)
	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD
	REAL(KIND=DBLE_PREC) :: DIST1, DIST2, TEMP_DIST
	
!	PRINT*, "START POOL DATA"
	
	DATA_MULTI=0
	DO I=1,SAMPLE_SITES
		LOC1=1
		DIST1=ABS(COORD(1,I)-GRID_COORD(1,LOC1))
		LOC2=1
		DIST2=ABS(COORD(2,I)-GRID_COORD(2,LOC2))
		DO J=2,GRID_LENGTH(1)
			TEMP_DIST=ABS(COORD(1,I)-GRID_COORD(1,J))
			IF(TEMP_DIST<DIST1) THEN
				LOC1=J
				DIST1=TEMP_DIST
			END IF
		END DO	
		
		DO J=2,GRID_LENGTH(2)
			TEMP_DIST=ABS(COORD(2,I)-GRID_COORD(2,J))
			IF(TEMP_DIST<DIST2) THEN
				LOC2=J
				DIST2=TEMP_DIST
			END IF
		END DO
		DO H=1,ALLELES
			DATA_MULTI(H,LOC1,LOC2)=DATA_MULTI(H,LOC1,LOC2)+DATA_MAT(H,I)
		END DO
	END DO
		
!	PRINT*, "END POOL DATA"
	
	END SUBROUTINE POOL_MULTI_DATA


END MODULE MULTINOMIAL









	
SUBROUTINE COUNT_NUMBER_SAMPLE_SITES(SAMPLE_SITES,LOCATION_FILE)
	
USE ALLELEFREQUENCIES
USE QUASINEWTON
USE CONSTANTS
USE ORIGENMOD

IMPLICIT NONE

INTEGER :: SAMPLE_SITES,I,J,K,INPUT_UNIT=4,LINES,TEMP_INT,IOERROR
CHARACTER(LEN=800) :: LOCATION_FILE,LINE
CHARACTER(LEN=50), DIMENSION(2000) :: MEMBERS_LIST
CHARACTER(LEN=50) :: MEMBER_NAME

!FINDS THE NUMBER OF LINES IN THE TEXT = NUMBER OF INDIVIDUALS
CALL INPUT_DATA(LOCATION_FILE,LINE,INPUT_UNIT,LINES)
REWIND(INPUT_UNIT)
!skip first line
LINE = " "
READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
SAMPLE_SITES=0
DO J = 1,LINES-1
	READ(INPUT_UNIT,*) TEMP_INT,MEMBER_NAME
	SAMPLE_SITES=SAMPLE_SITES+1
	MEMBERS_LIST(SAMPLE_SITES)=TRIM(MEMBER_NAME)
	!
	!     Every 100 lines sort and purge the list of ITEMS.
	!
	IF (MOD(J,100)==0.OR.J==LINES-1) THEN
		CALL SORT_STRINGS(MEMBERS_LIST(1:SAMPLE_SITES))
		CALL PURGE_STRINGS(MEMBERS_LIST(1:SAMPLE_SITES),SAMPLE_SITES)
	END IF
END DO
CLOSE(INPUT_UNIT)
END SUBROUTINE COUNT_NUMBER_SAMPLE_SITES





SUBROUTINE COUNT_NUMBER_LINES(NUMBER_LINES,MAP_FILE)
	
USE ALLELEFREQUENCIES
USE QUASINEWTON
USE CONSTANTS
USE ORIGENMOD

IMPLICIT NONE

INTEGER :: NUMBER_LINES,I,J,K,INPUT_UNIT=14
CHARACTER(LEN=800) :: MAP_FILE,LINE,TEXT_FILE

!FINDS THE NUMBER OF LINES IN THE TEXT = NUMBER OF INDIVIDUALS

!TEXT_FILE=TRIM(MAP_FILE)//".map"

!CALL INTPR("NUM",-1,NUMBER_SNPS,1)
CALL INPUT_DATA(MAP_FILE,LINE,INPUT_UNIT,NUMBER_LINES)
CLOSE(INPUT_UNIT)

END SUBROUTINE COUNT_NUMBER_LINES





SUBROUTINE FORMAT_PLINK_DATA(DATA_ARRAY,DATA_COORD,DATA_FILE,LOCATION_FILE,SAMPLE_SITES,NUMBER_SNPS)
!SUBROUTINE FORMAT_PLINK_DATA(DATA_ARRAY,COORD,MEMBERS,DATA_FILE,LOCATION_FILE,SAMPLE_SITES,NUMBER_SNPS)

!THIS INPUTS PLINK FORMATTED DATA AND OUTPUTS POOLED DATA BY SAMPLE_SITE USED IN OriGen

USE ALLELEFREQUENCIES
USE QUASINEWTON
USE CONSTANTS
USE ORIGENMOD

IMPLICIT NONE

CHARACTER(LEN=800) :: TEXT_FILE,LINE,TEXT_FILE2,TEXT_END,DATA_FILE,LOCATION_FILE
INTEGER :: I,J,K,INPUT_UNIT=4,LINES,TEMP_INT,ALLELES,TENTH,TEMP_COUNT
INTEGER :: INDIVIDUALS,IOERROR,NUMBER_SNPS,SAMPLE_SITES
INTEGER :: TEMP_MEM,NUMBER_UNKNOWN
REAL(KIND=DBLE_PREC) :: TEMP_REAL1,TEMP_REAL2
INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_ARRAY
INTEGER, ALLOCATABLE, DIMENSION(:) :: MEMBERSHIP !MEMBERSHIP(INDIVIDUALS)
REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: COORD !COORD(LONG=1;LAT=2,POPULATION)
REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES,2) :: DATA_COORD
INTEGER, ALLOCATABLE, DIMENSION(:,:) :: MOD_OBS, UNKNOWN_OBS
INTEGER, DIMENSION(6) :: TEMP_INT_VEC
INTEGER, DIMENSION(SAMPLE_SITES) :: TEMP_VEC
CHARACTER(LEN=50), DIMENSION(SAMPLE_SITES) :: MEMBERS
CHARACTER(LEN=50), DIMENSION(2000) :: MEMBERS_LIST
CHARACTER(LEN=1), ALLOCATABLE, DIMENSION(:,:) :: MAJOR_ALLELES,TEMP_ALLELES
INTEGER, ALLOCATABLE, DIMENSION(:) :: ALT_ALLELE_FOUND
CHARACTER(LEN=1), ALLOCATABLE, DIMENSION(:,:) :: TEMP_CHAR !LOCI,ALLELES
CHARACTER(LEN=50) :: MEMBER_NAME, TEMP_TEXT

!PRINT*,"BEGIN FORMAT PLINK DATA(NUMBER SNPS) ",NUMBER_SNPS
!PRINT*, "DATA_FILE: ", TRIM(DATA_FILE)
!PRINT*, "LOCATION_FILE: ", TRIM(LOCATION_FILE)
!FINDS THE NUMBER OF LINES IN THE TEXT = NUMBER OF INDIVIDUALS
CALL INPUT_DATA(LOCATION_FILE,LINE,INPUT_UNIT,LINES)
REWIND(INPUT_UNIT)
!skip first line
LINE = " "
READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
SAMPLE_SITES=0
DO J = 1,LINES-1
	READ(INPUT_UNIT,*) TEMP_INT,MEMBER_NAME
	SAMPLE_SITES=SAMPLE_SITES+1
	MEMBERS_LIST(SAMPLE_SITES)=TRIM(MEMBER_NAME)
	!
	!     Every 100 lines sort and purge the list of ITEMS.
	!
	IF (MOD(J,100)==0.OR.J==LINES-1) THEN
		CALL SORT_STRINGS(MEMBERS_LIST(1:SAMPLE_SITES))
		CALL PURGE_STRINGS(MEMBERS_LIST(1:SAMPLE_SITES),SAMPLE_SITES)
	END IF
END DO
MEMBERS=MEMBERS_LIST(1:SAMPLE_SITES)
!PRINT*, "POPS: ", SAMPLE_SITES
!PRINT*, (TRIM(MEMBERS_LIST(J)),J=1,SAMPLE_SITES)

!READ IN POPULATION DATA (WHAT NUMBER BELONGS TO WHICH POPULATION)
!INDIVIDUALS IN PC1_PC2_summary.txt are in the same order as .bed file... So will update
!membership using this file.
!ALSO UPDATES THE COORDINATE DATA
INDIVIDUALS=LINES-1
!PRINT*, "INDIVIDUALS(1387)", INDIVIDUALS
ALLOCATE(MEMBERSHIP(INDIVIDUALS))
COORD=1.
REWIND(INPUT_UNIT)
!skip first line
LINE = " "
READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
DO J=1,LINES-1
!	PRINT*, J
	READ(INPUT_UNIT,*) TEMP_INT,MEMBER_NAME,TEMP_TEXT,TEMP_REAL1,TEMP_REAL2
	TEMP_INT=BISECT_STRING_LIST(MEMBERS_LIST(1:SAMPLE_SITES),TRIM(MEMBER_NAME))
!	PRINT*, "TEMP_INT: ", TEMP_INT
	MEMBERSHIP(J)=TEMP_INT
!	PRINT*, "MEMBER NAME, MEMBERS_LIST: ", TRIM(MEMBER_NAME), ", ", &
!		TRIM(MEMBERS_LIST(TEMP_INT))
	COORD(1,TEMP_INT)=TEMP_REAL1
	COORD(2,TEMP_INT)=TEMP_REAL2
END DO
CLOSE(INPUT_UNIT)


!!COUNT THE NUMBER OF LOCI...SHOULD BE 197146 SNPS
!TEXT_FILE=TRIM(DATA_FILE)//".map"
!!FINDS THE NUMBER OF LINES/LOCI IN THE TEXT
!CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LOCI)
!
!!ALLELES=2 SINCE SNP DATA
!!PRINT*, "LOCI(197146): ", LOCI
!
!CLOSE(INPUT_UNIT)

ALLELES=2

!UPDATES THE MAJOR ALLELES VECTOR
ALLOCATE(MAJOR_ALLELES(ALLELES,NUMBER_SNPS),TEMP_CHAR(NUMBER_SNPS,ALLELES))
ALLOCATE(ALT_ALLELE_FOUND(NUMBER_SNPS),TEMP_ALLELES(ALLELES,NUMBER_SNPS))
TEMP_ALLELES(:,:)="X"
ALT_ALLELE_FOUND(:)=0
!TEXT_FILE="POPRES/Imputed.bim"
!!FINDS THE NUMBER OF LINES IN THE TEXT
!CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
!REWIND(INPUT_UNIT)
!PRINT*, "LINES, LOCI(=):", LINES, LOCI
!DO I=1,LINES
!	READ(INPUT_UNIT,*) TEMP_INT,LINE,TEMP_INT,TEMP_INT, &
!				MAJOR_ALLELES(1,I), MAJOR_ALLELES(2,I)
!END DO
!CLOSE(INPUT_UNIT)



!TEXT_FILE=TRIM(DATA_FILE)//".ped"
TEXT_FILE=DATA_FILE
!PRINT*, TEXT_FILE
CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
REWIND(INPUT_UNIT)
DO I=1,LINES
	READ(INPUT_UNIT,*) (TEMP_INT_VEC(J),J=1,6), &
				(TEMP_CHAR(J,1),TEMP_CHAR(J,2),J=1,NUMBER_SNPS)
	TEMP_MEM=MEMBERSHIP(I)
	DO J=1,NUMBER_SNPS
		IF(TEMP_CHAR(J,1).NE.'0') THEN
			IF(ALT_ALLELE_FOUND(J).EQ.0) THEN
				TEMP_ALLELES(1,J)=TEMP_CHAR(J,1)
				ALT_ALLELE_FOUND(J)=ALT_ALLELE_FOUND(J)+1
			ELSE IF(ALT_ALLELE_FOUND(J).EQ.1) THEN
				IF(TEMP_CHAR(J,1).NE.TEMP_ALLELES(1,J)) THEN
					TEMP_ALLELES(2,J)=TEMP_CHAR(J,1)
					ALT_ALLELE_FOUND(J)=ALT_ALLELE_FOUND(J)+1
				END IF
			ELSE IF(ALT_ALLELE_FOUND(J).EQ.2) THEN
				IF((TEMP_CHAR(J,1).NE.TEMP_ALLELES(1,J)).AND.(TEMP_CHAR(J,1).NE.TEMP_ALLELES(2,J))) THEN
!					PRINT*, "SOMETHINGS WRONG... MORE THAN 2 ALLELES"
					CALL INTPR("SOMETHINGS WRONG...MORE THAN 2 ALLELES FOUND",-1,1,0)
					CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
					CALL INTPR("LOCUS:",-1,J,1)
					CALL INTPR("INDIVIDUAL:",-1,I,1)
!					PRINT*, "LOCUS: ",J, " INDIVIDUAL: ",I, " CHAR: ",TEMP_CHAR(J,1)
				END IF
			END IF
		END IF
		IF(TEMP_CHAR(J,2).NE.'0') THEN
			IF(ALT_ALLELE_FOUND(J).EQ.0) THEN
				TEMP_ALLELES(1,J)=TEMP_CHAR(J,2)
				ALT_ALLELE_FOUND(J)=ALT_ALLELE_FOUND(J)+1
			ELSE IF(ALT_ALLELE_FOUND(J).EQ.1) THEN
				IF(TEMP_CHAR(J,2).NE.TEMP_ALLELES(1,J)) THEN
					TEMP_ALLELES(2,J)=TEMP_CHAR(J,2)
					ALT_ALLELE_FOUND(J)=ALT_ALLELE_FOUND(J)+1
				END IF
			ELSE IF(ALT_ALLELE_FOUND(J).EQ.2) THEN
				IF((TEMP_CHAR(J,2).NE.TEMP_ALLELES(1,J)).AND.(TEMP_CHAR(J,2).NE.TEMP_ALLELES(2,J))) THEN
!					PRINT*, "SOMETHINGS WRONG... MORE THAN 2 ALLELES"
!					PRINT*, "LOCUS: ",J, " INDIVIDUAL: ",I, " CHAR: ",TEMP_CHAR(J,2)
					CALL INTPR("SOMETHINGS WRONG...MORE THAN 2 ALLELES FOUND",-1,1,0)
					CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
					CALL INTPR("LOCUS:",-1,J,1)
					CALL INTPR("INDIVIDUAL:",-1,I,1)
				END IF
			END IF
		END IF
	END DO
END DO


!INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_ARRAY
!ALLOCATE(MAJOR_ALLELES(ALLELES,NUMBER_SNPS),TEMP_CHAR(NUMBER_SNPS,ALLELES))
!ALLOCATE(ALT_ALLELE_FOUND(NUMBER_SNPS),TEMP_ALLELES(ALLELES,NUMBER_SNPS))
!THIS PART OF THE CODE COUNTS AND BINS THE DATA BY LOCATION
!ALLOCATE(MOD_OBS(SAMPLE_SITES,NUMBER_SNPS*ALLELES))
!MOD_OBS=0
DATA_ARRAY=0
REWIND(INPUT_UNIT)
!CALL INTPR("MEMBERSHIP",-1,MEMBERSHIP,INDIVIDUALS)
DO I=1,LINES!LINES IS INDIVIDUALS
	READ(INPUT_UNIT,*) (TEMP_INT_VEC(J),J=1,6), &
				(TEMP_CHAR(J,1),TEMP_CHAR(J,2),J=1,NUMBER_SNPS)
	TEMP_MEM=MEMBERSHIP(I)
	DO K=1,NUMBER_SNPS
	
		!!MOD_OBS(SAMPLE_SITES,NUMBER_SNPS*ALLELES)
		!SAVING OUR DATA
		IF(TEMP_CHAR(K,1)==TEMP_ALLELES(1,K)) THEN
!			MOD_OBS(TEMP_MEM,K*2-1)=MOD_OBS(TEMP_MEM,K*2-1)+1
			DATA_ARRAY(1,TEMP_MEM,K)=DATA_ARRAY(1,TEMP_MEM,K)+1
		ELSE IF(TEMP_CHAR(K,1)==TEMP_ALLELES(2,K)) THEN
!			MOD_OBS(TEMP_MEM,K*2)=MOD_OBS(TEMP_MEM,K*2)+1
			DATA_ARRAY(2,TEMP_MEM,K)=DATA_ARRAY(2,TEMP_MEM,K)+1
		END IF
		IF(TEMP_CHAR(K,2)==TEMP_ALLELES(1,K)) THEN
!			MOD_OBS(TEMP_MEM,K*2-1)=MOD_OBS(TEMP_MEM,K*2-1)+1
			DATA_ARRAY(1,TEMP_MEM,K)=DATA_ARRAY(1,TEMP_MEM,K)+1
		ELSE IF(TEMP_CHAR(K,2)==TEMP_ALLELES(2,K)) THEN
!			MOD_OBS(TEMP_MEM,K*2)=MOD_OBS(TEMP_MEM,K*2)+1
			DATA_ARRAY(2,TEMP_MEM,K)=DATA_ARRAY(2,TEMP_MEM,K)+1
		END IF
		
		
	END DO
	IF(MOD(I,100).EQ.1) THEN
!		PRINT*, "INDIVIDUAL: ",I
		CALL INTPR("INDIVIDUAL",-1,I,1)
	END IF
END DO
CLOSE(INPUT_UNIT)

!SORT THE MAJOR/MINOR ALLELES
MAJOR_ALLELES=TEMP_ALLELES
DO I=1,NUMBER_SNPS
	!MAJOR SNPS IS DECIDED BASED ON MOST OCCURING IN THE SAMPLE
	!SO IF THE MOST OCCURING IS THE SECOND ONE WE SWITCH...
	IF(SUM(DATA_ARRAY(1,:,I)).LT.SUM(DATA_ARRAY(2,:,I))) THEN
		MAJOR_ALLELES(1,I)=TEMP_ALLELES(2,I)
		MAJOR_ALLELES(2,I)=TEMP_ALLELES(1,I)
		TEMP_VEC=DATA_ARRAY(1,:,I)
		DATA_ARRAY(1,:,I)=DATA_ARRAY(2,:,I)
		DATA_ARRAY(2,:,I)=TEMP_VEC
	END IF
END DO

DATA_COORD=TRANSPOSE(COORD)

END SUBROUTINE FORMAT_PLINK_DATA



!SUBROUTINE RTRIALS(ARRAY1,ARRAY_LENGTH)
!
!IMPLICIT NONE
!
!INTEGER :: I,J,K
!INTEGER, DIMENSION(3) :: ARRAY_LENGTH
!INTEGER, DIMENSION(ARRAY_LENGTH(1),ARRAY_LENGTH(2),ARRAY_LENGTH(3)) :: ARRAY1
!
!ARRAY1(2,2,2)=1
!
!END SUBROUTINE RTRIALS



SUBROUTINE CALC_ALL_RANKINGS(DATA_ARRAY,SAMPLE_SITES,NUMBER_SNPS,RANKINGS,TEST_STATS)
! THIS SUBROUTINE CALCULATES THE LIKELIHOOD RATIO STATISTIC ALONG WITH ROSENBERG'S 
!INFORMATIVENESS FOR ASSIGNMENT FOR ALL SNPS IN THE DATA
	
USE ALLELEFREQUENCIES
USE QUASINEWTON
USE CONSTANTS
USE ORIGENMOD

IMPLICIT NONE

INTEGER :: SAMPLE_SITES,NUMBER_SNPS,I,J,K
INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_ARRAY
INTEGER, DIMENSION(NUMBER_SNPS) :: RANKINGS
REAL(KIND=DBLE_PREC), DIMENSION(2,NUMBER_SNPS) :: TEST_STATS,TEMP_STATS

RANKINGS=0
TEST_STATS=0.

DO I=1,NUMBER_SNPS
	TEST_STATS(1,I)=LIKELIHOOD_RATIO(DATA_ARRAY(:,:,I),2,SAMPLE_SITES)
	TEST_STATS(2,I)=INFORMATIVENESS_FOR_ASSIGNMENT(DATA_ARRAY(:,:,I),2,SAMPLE_SITES)
!	PRINT*, I, TEST_STATS(1,I)
END DO

TEMP_STATS=TEST_STATS
CALL RANK_REALS(TEST_STATS(1,:),NUMBER_SNPS,RANKINGS)
DO I=1,NUMBER_SNPS
	TEST_STATS(2,I)=TEMP_STATS(2,RANKINGS(I))
!	PRINT*, RANKINGS(I), TEST_STATS(1,I)
END DO

END SUBROUTINE CALC_ALL_RANKINGS





SUBROUTINE UPDATE_GRID_COORD_SQUARE2(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!THIS IS AN EXACT COPY OF UPDATE_GRID_COORD_SQUARE
!THIS SUBROUTINE UPDATES THE GRID COORDINATES MAKING EACH SITE A SQUARE
!IT ALSO UPDATES THE GRID_LENGTH(2) VECTOR FOR THE DIFFERENT SIZES

USE ALLELEFREQUENCIES
USE QUASINEWTON
USE CONSTANTS
USE ORIGENMOD

IMPLICIT NONE

INTEGER :: I,J,K,SAMPLE_SITES,MAX_GRID_LENGTH
INTEGER, DIMENSION(2) :: GRID_LENGTH
REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES,2) :: DATA_COORD !COORD(LONG=1;LAT=2,POPULATION)
REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
REAL(KIND=DBLE_PREC) :: STEP,MINLONG,MAXLONG,MINLAT,MAXLAT
REAL(KIND=DBLE_PREC) :: MAXSIZE,LONGDIST,LATDIST

!ESTABLISHING THE CORNERS OF THE BOX
MINLONG=MINVAL(DATA_COORD(:,1))-1
!overriding value to inclde all of one country
!MINLONG=-10.0
!PRINT*, "Min Longitude set at -10"
MAXLONG=MAXVAL(DATA_COORD(:,1))+1
MINLAT=MINVAL(DATA_COORD(:,2))-1
MAXLAT=MAXVAL(DATA_COORD(:,2))+1
!MAXLAT=63.0
!PRINT*, "Max Latitude set at 63"
MAXSIZE=MAX(MAXLAT-MINLAT,MAXLONG-MINLONG)

!ESTABLISHING STEP LENGTHS
LONGDIST=(MAXLONG-MINLONG)
LATDIST=(MAXLAT-MINLAT)

IF(LONGDIST.GE.LATDIST) THEN
	GRID_LENGTH(1)=MAX_GRID_LENGTH
	STEP=LONGDIST/(GRID_LENGTH(1)-1.)
	GRID_LENGTH(2)=1
	DO I=1,MAX_GRID_LENGTH
		IF(MINLAT+I*STEP<MAXLAT) THEN
			GRID_LENGTH(2)=GRID_LENGTH(2)+1
		ELSE
			EXIT
		END IF
	END DO
ELSE
	GRID_LENGTH(2)=MAX_GRID_LENGTH
	STEP=LATDIST/(GRID_LENGTH(2)-1.)
	GRID_LENGTH(1)=1
	DO I=1,MAX_GRID_LENGTH
		IF(MINLONG+I*STEP<MAXLONG) THEN
			GRID_LENGTH(1)=GRID_LENGTH(1)+1
		ELSE
			EXIT
		END IF
	END DO
END IF

GRID_COORD=0.
DO I=1,GRID_LENGTH(1)
	GRID_COORD(1,I)=MINLONG+(I-1)*STEP
END DO

DO I=1,GRID_LENGTH(2)
	GRID_COORD(2,I)=MINLAT+(I-1)*STEP
END DO

END SUBROUTINE UPDATE_GRID_COORD_SQUARE2




SUBROUTINE FITORIGENMODEL(ALLELE_FREQUENCY_SURFACES,DATA_ARRAY,NUMBER_SNPS,GRID_LENGTH,RHO,&
		SAMPLE_SITES,MAX_GRID_LENGTH,DATA_COORD,GRID_COORD)
	!(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO)
!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL

USE ALLELEFREQUENCIES
USE QUASINEWTON
USE CONSTANTS
USE ORIGENMOD

IMPLICIT NONE

INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,MAX_GRID_LENGTH
INTEGER, DIMENSION(2) :: GRID_LENGTH
INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_ARRAY !(FIRST IS ALLELE, population, 3RD IS LOCUS)
REAL(KIND=DBLE_PREC), DIMENSION(NUMBER_SNPS,GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY_SURFACES
REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES,2) :: DATA_COORD !(LONG,LAT)
REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
INTEGER, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_ALLELE1,DATA_TOTAL
!REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: ALLELE_FREQUENCY,QNEWT_FREQ !LONG,LAT
REAL(KIND=DBLE_PREC) :: RHO,LONG_WEIGHT,LAT_WEIGHT,TEMP_REAL


!PRINT*, "FITORIGENMODEL"

!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)

!ALLOCATE(DATA_ARRAY(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!DATA_ARRAY=0
!DATA_COORD=0.

!CALL READ_GENO_DATA(DATA_FILE,DATA_ARRAY,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)

!CALL UPDATE_GRID_COORD_SQUARE(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
!ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)),QNEWT_FREQ(GRID_LENGTH(1),GRID_LENGTH(2)))
!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
LONG_WEIGHT=ONE
LAT_WEIGHT=ONE
!	LONG_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))
!	LAT_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!ALLELE_FREQUENCY_SURFACES=ZERO

DO I=1,NUMBER_SNPS
!	PRINT*, "READ AND FIT-SNP: ", I
	CALL INTPR("CURRENT SNP",-1,I,1)
	CALL POOL_GENO_DATA_SQUARE(DATA_ARRAY(:,:,I),TRANSPOSE(DATA_COORD),DATA_ALLELE1,DATA_TOTAL,&
			GRID_COORD,SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
!	CALL FIT_L2_MODEL_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY_SURFACES(I,:,:),RHO,&
!			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,.TRUE.)
	
	CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY_SURFACES(I,:,:),RHO,&
			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)		
		
END DO

END SUBROUTINE FITORIGENMODEL
	






SUBROUTINE FORMAT_UNKNOWN_PLINK_DATA(DATA_ARRAY,DATA_COORD,DATA_FILE,LOCATION_FILE,SAMPLE_SITES,NUMBER_SNPS,&
			UNKNOWN_PED_FILE,NUMBER_UNKNOWN,UNKNOWN_DATA,MEMBERSHIP,NUMBER_KNOWN)
!SUBROUTINE FORMAT_PLINK_DATA(DATA_ARRAY,COORD,MEMBERS,DATA_FILE,LOCATION_FILE,SAMPLE_SITES,NUMBER_SNPS)

!THIS INPUTS PLINK FORMATTED DATA AND OUTPUTS POOLED DATA BY SAMPLE_SITE USED IN OriGen

USE ALLELEFREQUENCIES
USE QUASINEWTON
USE CONSTANTS
USE ORIGENMOD

IMPLICIT NONE

CHARACTER(LEN=800) :: TEXT_FILE,LINE,TEXT_FILE2,TEXT_END,DATA_FILE,LOCATION_FILE,UNKNOWN_PED_FILE
INTEGER :: I,J,K,INPUT_UNIT=4,LINES,TEMP_INT,ALLELES,TENTH,TEMP_COUNT,OUTPUT_UNIT=21
INTEGER :: INDIVIDUALS,IOERROR,NUMBER_SNPS,SAMPLE_SITES
INTEGER :: TEMP_MEM,NUMBER_UNKNOWN,NUMBER_KNOWN
REAL(KIND=DBLE_PREC) :: TEMP_REAL1,TEMP_REAL2
INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_ARRAY
INTEGER, DIMENSION(NUMBER_UNKNOWN,NUMBER_SNPS) :: UNKNOWN_DATA
INTEGER, DIMENSION(NUMBER_KNOWN) :: MEMBERSHIP, TOP_SNPS !MEMBERSHIP(INDIVIDUALS)
!INTEGER, DIMENSION(:) :: TRUE_MEMBERSHIP
REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: COORD !COORD(LONG=1;LAT=2,POPULATION)
REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES,2) :: DATA_COORD
INTEGER, DIMENSION(6) :: TEMP_INT_VEC
INTEGER, DIMENSION(SAMPLE_SITES) :: TEMP_VEC
CHARACTER(LEN=50), DIMENSION(SAMPLE_SITES) :: MEMBERS
CHARACTER(LEN=50), DIMENSION(2000) :: MEMBERS_LIST
CHARACTER(LEN=1), ALLOCATABLE, DIMENSION(:,:) :: MAJOR_ALLELES,TEMP_ALLELES
INTEGER, ALLOCATABLE, DIMENSION(:) :: ALT_ALLELE_FOUND
CHARACTER(LEN=1), ALLOCATABLE, DIMENSION(:,:) :: TEMP_CHAR !LOCI,ALLELES
CHARACTER(LEN=50) :: MEMBER_NAME, TEMP_TEXT

!PRINT*,"WEED_DATA_OURS_ONLY_FULL BEGINNING... SNPS: ",NUMBER_SNPS
!FINDS THE NUMBER OF LINES IN THE TEXT = NUMBER OF INDIVIDUALS
CALL INPUT_DATA(LOCATION_FILE,LINE,INPUT_UNIT,LINES)
REWIND(INPUT_UNIT)
!skip first line
LINE = " "
READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
SAMPLE_SITES=0
DO J = 1,LINES-1
	READ(INPUT_UNIT,*) TEMP_INT,MEMBER_NAME
	SAMPLE_SITES=SAMPLE_SITES+1
	MEMBERS_LIST(SAMPLE_SITES)=TRIM(MEMBER_NAME)
	!
	!     Every 100 lines sort and purge the list of ITEMS.
	!
	IF (MOD(J,100)==0.OR.J==LINES-1) THEN
		CALL SORT_STRINGS(MEMBERS_LIST(1:SAMPLE_SITES))
		CALL PURGE_STRINGS(MEMBERS_LIST(1:SAMPLE_SITES),SAMPLE_SITES)
	END IF
END DO
MEMBERS=MEMBERS_LIST(1:SAMPLE_SITES)
!PRINT*, "POPS: ", SAMPLE_SITES
!PRINT*, (TRIM(MEMBERS_LIST(J)),J=1,SAMPLE_SITES)


!READ IN POPULATION DATA (WHAT NUMBER BELONGS TO WHICH POPULATION)
!INDIVIDUALS IN PC1_PC2_summary.txt are in the same order as .bed file... So will update
!membership using this file.
!ALSO UPDATES THE COORDINATE DATA
INDIVIDUALS=LINES-1
!PRINT*, "INDIVIDUALS(1387)", INDIVIDUALS
!ALLOCATE(MEMBERSHIP(INDIVIDUALS))
COORD=1.
REWIND(INPUT_UNIT)
!skip first line
LINE = " "
READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
DO J=1,LINES-1
	READ(INPUT_UNIT,*) TEMP_INT,MEMBER_NAME,TEMP_TEXT,TEMP_REAL1,TEMP_REAL2
	TEMP_INT=BISECT_STRING_LIST(MEMBERS_LIST(1:SAMPLE_SITES),TRIM(MEMBER_NAME))
	!PRINT*, "TEMP_INT: ", TEMP_INT
	MEMBERSHIP(J)=TEMP_INT
	!PRINT*, "MEMBER NAME, MEMBERS_LIST: ", TRIM(MEMBER_NAME), ", ", &
	!		TRIM(MEMBERS_LIST(TEMP_INT))
	COORD(1,TEMP_INT)=TEMP_REAL1
	COORD(2,TEMP_INT)=TEMP_REAL2
END DO
CLOSE(INPUT_UNIT)


!!COUNT THE NUMBER OF LOCI...SHOULD BE 197146 SNPS
!TEXT_FILE=TRIM(DATA_FILE)//".map"
!!FINDS THE NUMBER OF LINES/LOCI IN THE TEXT
!CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LOCI)
!
!!ALLELES=2 SINCE SNP DATA
!!PRINT*, "LOCI(197146): ", LOCI
!
!CLOSE(INPUT_UNIT)

ALLELES=2

!UPDATES THE MAJOR ALLELES VECTOR
ALLOCATE(MAJOR_ALLELES(ALLELES,NUMBER_SNPS),TEMP_CHAR(NUMBER_SNPS,ALLELES))
ALLOCATE(ALT_ALLELE_FOUND(NUMBER_SNPS),TEMP_ALLELES(ALLELES,NUMBER_SNPS))
TEMP_ALLELES(:,:)="X"
ALT_ALLELE_FOUND(:)=0
!TEXT_FILE="POPRES/Imputed.bim"
!!FINDS THE NUMBER OF LINES IN THE TEXT
!CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
!REWIND(INPUT_UNIT)
!PRINT*, "LINES, LOCI(=):", LINES, LOCI
!DO I=1,LINES
!	READ(INPUT_UNIT,*) TEMP_INT,LINE,TEMP_INT,TEMP_INT, &
!				MAJOR_ALLELES(1,I), MAJOR_ALLELES(2,I)
!END DO
!CLOSE(INPUT_UNIT)

!TEXT_FILE="TEMP.txt"
!OPEN(OUTPUT_UNIT,FILE=TEXT_FILE)


!TEXT_FILE=TRIM(DATA_FILE)//".ped"
TEXT_FILE=DATA_FILE
!PRINT*, TEXT_FILE
CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
REWIND(INPUT_UNIT)
DO I=1,LINES
	READ(INPUT_UNIT,*) (TEMP_INT_VEC(J),J=1,6), &
				(TEMP_CHAR(J,1),TEMP_CHAR(J,2),J=1,NUMBER_SNPS)
	TEMP_MEM=MEMBERSHIP(I)
	DO J=1,NUMBER_SNPS
		IF(TEMP_CHAR(J,1).NE.'0') THEN
			IF(ALT_ALLELE_FOUND(J).EQ.0) THEN
				TEMP_ALLELES(1,J)=TEMP_CHAR(J,1)
				ALT_ALLELE_FOUND(J)=ALT_ALLELE_FOUND(J)+1
			ELSE IF(ALT_ALLELE_FOUND(J).EQ.1) THEN
				IF(TEMP_CHAR(J,1).NE.TEMP_ALLELES(1,J)) THEN
					TEMP_ALLELES(2,J)=TEMP_CHAR(J,1)
					ALT_ALLELE_FOUND(J)=ALT_ALLELE_FOUND(J)+1
				END IF
			ELSE IF(ALT_ALLELE_FOUND(J).EQ.2) THEN
				IF((TEMP_CHAR(J,1).NE.TEMP_ALLELES(1,J)).AND.(TEMP_CHAR(J,1).NE.TEMP_ALLELES(2,J))) THEN
!					PRINT*, "SOMETHINGS WRONG... MORE THAN 2 ALLELES"
!					PRINT*, "LOCUS: ",J, " INDIVIDUAL: ",I, " CHAR: ",TEMP_CHAR(J,1)
					CALL INTPR("SOMETHINGS WRONG...MORE THAN 2 ALLELES FOUND",-1,1,0)
					CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
					CALL INTPR("LOCUS:",-1,J,1)
					CALL INTPR("INDIVIDUAL:",-1,I,1)
				END IF
			END IF
		END IF
		IF(TEMP_CHAR(J,2).NE.'0') THEN
			IF(ALT_ALLELE_FOUND(J).EQ.0) THEN
				TEMP_ALLELES(1,J)=TEMP_CHAR(J,2)
				ALT_ALLELE_FOUND(J)=ALT_ALLELE_FOUND(J)+1
			ELSE IF(ALT_ALLELE_FOUND(J).EQ.1) THEN
				IF(TEMP_CHAR(J,2).NE.TEMP_ALLELES(1,J)) THEN
					TEMP_ALLELES(2,J)=TEMP_CHAR(J,2)
					ALT_ALLELE_FOUND(J)=ALT_ALLELE_FOUND(J)+1
				END IF
			ELSE IF(ALT_ALLELE_FOUND(J).EQ.2) THEN
				IF((TEMP_CHAR(J,2).NE.TEMP_ALLELES(1,J)).AND.(TEMP_CHAR(J,2).NE.TEMP_ALLELES(2,J))) THEN
!					PRINT*, "SOMETHINGS WRONG... MORE THAN 2 ALLELES"
!					PRINT*, "LOCUS: ",J, " INDIVIDUAL: ",I, " CHAR: ",TEMP_CHAR(J,2)
					CALL INTPR("SOMETHINGS WRONG...MORE THAN 2 ALLELES FOUND",-1,1,0)
					CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
					CALL INTPR("LOCUS:",-1,J,1)
					CALL INTPR("INDIVIDUAL:",-1,I,1)
				END IF
			END IF
		END IF
	END DO
END DO
!WRITE(OUTPUT_UNIT,*) "TEMP_ALLELES: ", (I,TEMP_ALLELES(1,I),TEMP_ALLELES(2,I),I=1,NUMBER_SNPS)



!INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_ARRAY
!ALLOCATE(MAJOR_ALLELES(ALLELES,NUMBER_SNPS),TEMP_CHAR(NUMBER_SNPS,ALLELES))
!ALLOCATE(ALT_ALLELE_FOUND(NUMBER_SNPS),TEMP_ALLELES(ALLELES,NUMBER_SNPS))
!THIS PART OF THE CODE COUNTS AND BINS THE DATA BY LOCATION
!ALLOCATE(MOD_OBS(SAMPLE_SITES,NUMBER_SNPS*ALLELES))
!MOD_OBS=0
DATA_ARRAY=0
REWIND(INPUT_UNIT)
!CALL INTPR("MEMBERSHIP",-1,MEMBERSHIP,INDIVIDUALS)
DO I=1,LINES!LINES IS INDIVIDUALS
	READ(INPUT_UNIT,*) (TEMP_INT_VEC(J),J=1,6), &
				(TEMP_CHAR(J,1),TEMP_CHAR(J,2),J=1,NUMBER_SNPS)
	TEMP_MEM=MEMBERSHIP(I)
	DO K=1,NUMBER_SNPS
	
		!!MOD_OBS(SAMPLE_SITES,NUMBER_SNPS*ALLELES)
		!SAVING OUR DATA
		IF(TEMP_CHAR(K,1)==TEMP_ALLELES(1,K)) THEN
!			MOD_OBS(TEMP_MEM,K*2-1)=MOD_OBS(TEMP_MEM,K*2-1)+1
			DATA_ARRAY(1,TEMP_MEM,K)=DATA_ARRAY(1,TEMP_MEM,K)+1
		ELSE IF(TEMP_CHAR(K,1)==TEMP_ALLELES(2,K)) THEN
!			MOD_OBS(TEMP_MEM,K*2)=MOD_OBS(TEMP_MEM,K*2)+1
			DATA_ARRAY(2,TEMP_MEM,K)=DATA_ARRAY(2,TEMP_MEM,K)+1
		END IF
		IF(TEMP_CHAR(K,2)==TEMP_ALLELES(1,K)) THEN
!			MOD_OBS(TEMP_MEM,K*2-1)=MOD_OBS(TEMP_MEM,K*2-1)+1
			DATA_ARRAY(1,TEMP_MEM,K)=DATA_ARRAY(1,TEMP_MEM,K)+1
		ELSE IF(TEMP_CHAR(K,2)==TEMP_ALLELES(2,K)) THEN
!			MOD_OBS(TEMP_MEM,K*2)=MOD_OBS(TEMP_MEM,K*2)+1
			DATA_ARRAY(2,TEMP_MEM,K)=DATA_ARRAY(2,TEMP_MEM,K)+1
		END IF
		
		
	END DO
	IF(MOD(I,100).EQ.1) THEN
!		PRINT*, "INDIVIDUAL: ",I
		CALL INTPR("CURENT INDIVIDUAL",-1,I,1)
	END IF
END DO
CLOSE(INPUT_UNIT)


!SORT THE MAJOR/MINOR ALLELES
MAJOR_ALLELES=TEMP_ALLELES
DO I=1,NUMBER_SNPS
	!MAJOR SNPS IS DECIDED BASED ON MOST OCCURING IN THE SAMPLE
	!SO IF THE MOST OCCURING IS THE SECOND ONE WE SWITCH...
	IF(SUM(DATA_ARRAY(1,:,I)).LT.SUM(DATA_ARRAY(2,:,I))) THEN
		MAJOR_ALLELES(1,I)=TEMP_ALLELES(2,I)
		MAJOR_ALLELES(2,I)=TEMP_ALLELES(1,I)
		TEMP_VEC=DATA_ARRAY(1,:,I)
		DATA_ARRAY(1,:,I)=DATA_ARRAY(2,:,I)
		DATA_ARRAY(2,:,I)=TEMP_VEC
	END IF
END DO


!THIS FILLS IN THE UNKNOWN_DATA
CALL INPUT_DATA(UNKNOWN_PED_FILE,LINE,INPUT_UNIT,LINES)
REWIND(INPUT_UNIT)
UNKNOWN_DATA=0
DO I =1,LINES!LINES SHOULD BE NUMBER_UNKNOWN
	READ(INPUT_UNIT,*) (TEMP_INT_VEC(J),J=1,6), &
				(TEMP_CHAR(J,1),TEMP_CHAR(J,2),J=1,NUMBER_SNPS)
	DO J=1,NUMBER_SNPS
		IF(TEMP_CHAR(J,1).EQ.MAJOR_ALLELES(1,J)) THEN
			UNKNOWN_DATA(I,J)=UNKNOWN_DATA(I,J)+1
		END IF
		IF(TEMP_CHAR(J,2).EQ.MAJOR_ALLELES(1,J)) THEN
			UNKNOWN_DATA(I,J)=UNKNOWN_DATA(I,J)+1
		END IF
	END DO
END DO
CLOSE(INPUT_UNIT)


!WRITE(OUTPUT_UNIT,*) "MAJOR_ALLELES: ", (I,MAJOR_ALLELES(1,I),MAJOR_ALLELES(2,I),I=1,NUMBER_SNPS)
!CLOSE(OUTPUT_UNIT)
DATA_COORD=TRANSPOSE(COORD)

END SUBROUTINE FORMAT_UNKNOWN_PLINK_DATA






SUBROUTINE FITORIGENMODELFINDUNKNOWNS(UNKNOWN_GRIDS,DATA_ARRAY,NUMBER_SNPS,GRID_LENGTH,RHO,&
		SAMPLE_SITES,MAX_GRID_LENGTH,DATA_COORD,NUMBER_UNKNOWN,UNKNOWN_DATA,GRID_COORD)
	!(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO)
!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL

USE ALLELEFREQUENCIES
USE QUASINEWTON
USE CONSTANTS
USE ORIGENMOD

IMPLICIT NONE

INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,MAX_GRID_LENGTH,NUMBER_UNKNOWN,NODES
INTEGER, DIMENSION(2) :: GRID_LENGTH
INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_ARRAY !(FIRST IS ALLELE, population, 3RD IS LOCUS)
!REAL(KIND=DBLE_PREC), DIMENSION(NUMBER_SNPS,GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY_SURFACES
REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES,2) :: DATA_COORD !(LONG,LAT)
REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
INTEGER, DIMENSION(NUMBER_UNKNOWN,NUMBER_SNPS) :: UNKNOWN_DATA
REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN) :: UNKNOWN_GRIDS
INTEGER, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_ALLELE1,DATA_TOTAL
REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY,GENOME_FRACTIONS!,QNEWT_FREQ !LONG,LAT
REAL(KIND=DBLE_PREC) :: RHO,LONG_WEIGHT,LAT_WEIGHT,TEMP_REAL
REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: TEMP_VECTOR
LOGICAL, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: IS_LAND
REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: POP_MATRIX


!PRINT*, "FIT ORIGEN MODEL FIND UNKNOWNS"

!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)

!ALLOCATE(DATA_ARRAY(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
!DATA_ARRAY=0
!DATA_COORD=0.

!CALL READ_GENO_DATA(DATA_FILE,DATA_ARRAY,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)

!CALL UPDATE_GRID_COORD_SQUARE(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
!ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)),QNEWT_FREQ(GRID_LENGTH(1),GRID_LENGTH(2)))
!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
LONG_WEIGHT=ONE
LAT_WEIGHT=ONE
!	LONG_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))
!	LAT_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!ALLELE_FREQUENCY_SURFACES=ZERO

IS_LAND(:,:)=.TRUE.
NODES=COUNT(IS_LAND)
ALLOCATE(POP_MATRIX(NUMBER_SNPS,NODES))
ALLOCATE(TEMP_VECTOR(NODES))
UNKNOWN_GRIDS=1

DO I=1,NUMBER_SNPS
	ALLELE_FREQUENCY=0.
	CALL POOL_GENO_DATA_SQUARE(DATA_ARRAY(:,:,I),TRANSPOSE(DATA_COORD),DATA_ALLELE1,DATA_TOTAL,GRID_COORD,&
			SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
!		CALL FIT_L2_MODEL_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
	CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
	
	!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
	!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
	CALL UPDATE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_DATA(:,I),UNKNOWN_GRIDS,GRID_LENGTH,&
						MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN,IS_LAND)
!	POP_MATRIX(I,:)=RESHAPE_MATRIX_MASK_SQUARE(ALLELE_FREQUENCY,IS_LAND,GRID_LENGTH,NODES)
END DO


END SUBROUTINE FITORIGENMODELFINDUNKNOWNS
	
	
	



	
	


SUBROUTINE LEAVE_ONE_POP_OUT_CROSSVAL_SQUARE(DATA_FILE,LOCATION_FILE,NUMBER_SNPS,MAX_ITS,&
		MAX_GRID_LENGTH,RHO_VEC,RHO,SAMPLE_SITES)
!THIS SUBROUTINE ESTIMATES THE RHO WEIGHT IN OUR MODEL USING CROSS VALIDATION
	
USE ALLELEFREQUENCIES
USE QUASINEWTON
USE CONSTANTS
USE ORIGENMOD

IMPLICIT NONE

INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,MAX_GRID_LENGTH,NUMBER_UNKNOWN,TEMP_INT,ITS,OUTPUT_UNIT=13
INTEGER :: POP, MAX_ITS, PHASE_TWO, LINES
INTEGER, DIMENSION(2) :: GRID_LENGTH
CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END,LOCATION_FILE
!INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_MATRIX !(FIRST IS ALLELE1, SECOND IS ALLELE2, 3RD IS LOCUS)
INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_ARRAY !(FIRST IS ALLELE, population, 3RD IS LOCUS)
REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_ITS) :: RHO_VEC !DATA_COORD(LONG,LAT)
REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
INTEGER, ALLOCATABLE, DIMENSION(:,:) :: DATA_ALLELE1,DATA_TOTAL
REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: ALLELE_FREQUENCY !LONG,LAT
REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT,SUM_DISTANCE
REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_OBS,ADMIXED_MEMBERSHIP,TEMP_DATA_ARRAY
LOGICAL, ALLOCATABLE, DIMENSION(:,:) :: IS_LAND
INTEGER, ALLOCATABLE, DIMENSION(:) :: SAMPLE_SIZES
REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: POP_AVG_DISTANCE
REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: COORD !COORD(LONG=1;LAT=2,POPULATION)
REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES,2) :: DATA_COORD
INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_DATA
INTEGER, ALLOCATABLE, DIMENSION(:) :: MEMBERSHIP

!COUNTS THE NUMBER OF UNKNOWNS
CALL COUNT_NUMBER_LINES(LINES,LOCATION_FILE)
!THIS FILE HAS A HEADER
NUMBER_UNKNOWN=LINES-1
ALLOCATE(UNKNOWN_DATA(NUMBER_UNKNOWN,NUMBER_SNPS),MEMBERSHIP(NUMBER_UNKNOWN))
MEMBERSHIP=0
CALL FORMAT_UNKNOWN_PLINK_DATA(DATA_ARRAY,DATA_COORD,DATA_FILE,LOCATION_FILE,SAMPLE_SITES,NUMBER_SNPS,&
			DATA_FILE,NUMBER_UNKNOWN,UNKNOWN_DATA,MEMBERSHIP,NUMBER_UNKNOWN)

!PRINT*, "MEMBERSHIP 2:", MEMBERSHIP

GRID_LENGTH=0

!ALLOCATE(DATA_ARRAY(2,SAMPLE_SITES,NUMBER_SNPS))
ALLOCATE(TEMP_DATA_ARRAY(2,SAMPLE_SITES))
TEMP_DATA_ARRAY=0
!DATA_ARRAY=0
!COORD=0.

!CALL READ_GENO_DATA(DATA_FILE,DATA_ARRAY,COORD,NUMBER_SNPS,SAMPLE_SITES)

CALL UPDATE_GRID_COORD_SQUARE2(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
COORD=TRANSPOSE(DATA_COORD)
!PRINT*, "HERE"
ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)))


!CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
ALLOCATE(UNKNOWN_GRID(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN))
!ALLOCATE(ADMIXED_MEMBERSHIP(2,NUMBER_UNKNOWN))
!UNKNOWN_OBS=0
UNKNOWN_GRID=0.
!CALL READ_UNKNOWN_DATA(DATA_FILE,UNKNOWN_OBS,ADMIXED_MEMBERSHIP,NUMBER_UNKNOWN,NUMBER_SNPS,SAMPLE_SITES)

ALLOCATE(IS_LAND(GRID_LENGTH(1),GRID_LENGTH(2)))
IS_LAND=.TRUE.
!CALL UPDATE_IS_LAND_SQUARE(IS_LAND,GRID_LENGTH,MAX_GRID_LENGTH)

LONG_WEIGHT=ONE
!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
LAT_WEIGHT=ONE

!PRINT*, "NUMBER_UNKNOWN(1387): ",NUMBER_UNKNOWN
!PRINT*, "SAMPLE_SITES: ", SAMPLE_SITES
CALL INTPR("NUMBER_UNKNOWN:",-1,NUMBER_UNKNOWN,1)
CALL INTPR("SAMPLE_SITES:",-1,SAMPLE_SITES,1)

ALLOCATE(SAMPLE_SIZES(SAMPLE_SITES))
SAMPLE_SIZES=0
DO I=1,NUMBER_UNKNOWN
!	PRINT*, I, MEMBERSHIP(I)
	SAMPLE_SIZES(MEMBERSHIP(I))=SAMPLE_SIZES(MEMBERSHIP(I))+1
END DO
!PRINT*, "SAMPLE_SIZES: ", SAMPLE_SIZES
!PRINT*, "MEMBERSHIP: ", MEMBERSHIP

ALLOCATE(POP_AVG_DISTANCE(SAMPLE_SITES))
POP_AVG_DISTANCE=0.

RHO_VEC=0.
RHO_VEC(1,:)=10.
ITS=0
PHASE_TWO=0
DO K=1,MAX_ITS
!		CALL POOL_GENO_DATA_SQUARE(DATA_ARRAY(:,:,1),COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,&
!				SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
	RHO=RHO_VEC(1,K)
	DO POP=1,SAMPLE_SITES!NEED TO REDUCE THESE AS WELL...
		UNKNOWN_GRID=1.
		DO I=1,NUMBER_SNPS !NEED TO CHANGE THIS TO 1,NUMBER_SNPS
			!reducing the number of snps for compute time SHOULD GO BACK TO MOD(I,10)
			IF((NUMBER_SNPS.LE.5).OR.(MOD(I,3).EQ.0)) THEN
				ALLELE_FREQUENCY=0.
				TEMP_DATA_ARRAY=DATA_ARRAY(:,:,I)
				TEMP_DATA_ARRAY(:,POP)=0
				CALL POOL_GENO_DATA_SQUARE(TEMP_DATA_ARRAY,TRANSPOSE(DATA_COORD),DATA_ALLELE1,DATA_TOTAL, & 
							GRID_COORD,SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)		
				CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
						LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!				PRINT*, "FINISHED QNEWT",K,POP,I
				CALL INTPR("ITS:",-1,K,1)
				CALL INTPR("POP:",-1,POP,1)
				CALL INTPR("SNP:",-1,I,1)
				
				!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
				!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
				CALL UPDATE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_DATA(:,I),UNKNOWN_GRID,GRID_LENGTH,&
									MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN,IS_LAND)
!				PRINT*, "FINISHED UNKNOWN_GRID"
				!THIS CALCULATES THE DISTANCE BETWEEN THE TRUE VS. ESTIMATED LOCATIONS
				IF((MOD(I,100).EQ.0).OR.(I.EQ.NUMBER_SNPS)) THEN
					SUM_DISTANCE=0.
					TEMP_DISTANCE=0.
					DO J=1,NUMBER_UNKNOWN !should be LINES
						IF(MEMBERSHIP(J).EQ.POP) THEN
							CALL FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID(:,:,J),TEMP_REAL1,TEMP_REAL2,&
									MAX_GRID_LENGTH,GRID_COORD,IS_LAND)
							TEMP_DISTANCE=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),COORD(:,MEMBERSHIP(J)))
							SUM_DISTANCE=SUM_DISTANCE+TEMP_DISTANCE
							!WRITE(OUTPUT_UNIT,*) I,J,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2,UNKNOWN_COORD(:,J)
						END IF
					END DO
					AVG_DISTANCE=SUM_DISTANCE/(0.+SAMPLE_SIZES(POP))
					IF(IEEE_IS_NAN(AVG_DISTANCE)) THEN
!						PRINT*, "SUM_DISTANCE: ", SUM_DISTANCE, " SAMPLE_SIZES: ", SAMPLE_SIZES(POP)
						CALL INTPR("SOMETHINGS WRONG...NAN AVERAGE DISTANCE",-1,1,0)
						CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
					END IF
!					PRINT*, "LOCI: ", I, "AVG_DISTANCE: ", AVG_DISTANCE, "RHO: ",RHO, "POP: ",POP
					IF(I.EQ.NUMBER_SNPS) THEN
						POP_AVG_DISTANCE(POP)=AVG_DISTANCE
					END IF
				END IF
			END IF
		END DO
	END DO

	RHO_VEC(2,K)=SUM(SAMPLE_SIZES*POP_AVG_DISTANCE)/(SUM(SAMPLE_SIZES)+0.)
	
	!HERE WE CHOOSE THE NEXT RHO BASED ON THE PREVIOUS RESULTS...
	IF(K.EQ.1) THEN
		RHO_VEC(1,2)=2*RHO_VEC(1,1)
	ELSE IF(K.EQ.2) THEN
		IF(RHO_VEC(2,K).LT.RHO_VEC(2,K-1)) THEN	
			!PROCEEDING IN THE RIGHT DIRECTION ...DOUBLING
			RHO_VEC(1,K+1)=2*RHO_VEC(1,2)
		ELSE
			!GO THE OTHER WAY... HALVING 1
			RHO_VEC(1,K+1)=0.5*RHO_VEC(1,2)
		END IF
	ELSE IF(K.LT.MAX_ITS) THEN
		IF(PHASE_TWO.EQ.0) THEN
			IF(RHO_VEC(2,K).LT.RHO_VEC(2,K-1)) THEN	
				RHO_VEC(1,K+1)=(RHO_VEC(1,K)/RHO_VEC(1,K-1))*RHO_VEC(1,K)
			ELSE
				PHASE_TWO=1
				RHO_VEC(1,K+1)=(RHO_VEC(1,K)+RHO_VEC(1,K-1))/2.
			END IF
		ELSE
			IF(RHO_VEC(2,K-1).LT.RHO_VEC(2,K-2)) THEN
				RHO_VEC(1,K+1)=(RHO_VEC(1,K)+RHO_VEC(1,K-1))/2.
			ELSE
				RHO_VEC(1,K+1)=(RHO_VEC(1,K)+RHO_VEC(1,K-2))/2.
			END IF
		END IF
	END IF
		
END DO
!PRINT*, "RHO ESTIMATION RESULTS"
!PRINT*, "RHO AVG_DISTANCE"

TEMP_INT=MINLOC(RHO_VEC(2,:),1)
RHO=RHO_VEC(1,TEMP_INT)
AVG_DISTANCE=RHO_VEC(2,TEMP_INT)
!PRINT*, "RHO_VEC: ", RHO_VEC
!PRINT*, "RHO: ", RHO

END SUBROUTINE LEAVE_ONE_POP_OUT_CROSSVAL_SQUARE

	
	



SUBROUTINE LEAVE_ONE_POP_OUT_CROSSVAL_SQUARE2(DATA_ARRAY,DATA_COORD,UNKNOWN_DATA,NUMBER_SNPS,SAMPLE_SITES,&
		NUMBER_UNKNOWNS,MAX_ITS,MAX_GRID_LENGTH,RHO_VEC,RHO)
!THIS SUBROUTINE ESTIMATES THE RHO WEIGHT IN OUR MODEL USING CROSS VALIDATION
	
USE ALLELEFREQUENCIES
USE QUASINEWTON
USE CONSTANTS
USE ORIGENMOD

IMPLICIT NONE

INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,MAX_GRID_LENGTH,NUMBER_UNKNOWNS,TEMP_INT,ITS,OUTPUT_UNIT=13
INTEGER :: POP, MAX_ITS, PHASE_TWO, LINES
INTEGER, DIMENSION(2) :: GRID_LENGTH
CHARACTER(LEN=800) :: DATA_FILE,TEXT_FILE,TEXT_END,LOCATION_FILE
INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_ARRAY !(FIRST IS ALLELE, population, 3RD IS LOCUS)
REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_ITS) :: RHO_VEC !DATA_COORD(LONG,LAT)
REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES,2) :: DATA_COORD
INTEGER, DIMENSION(NUMBER_UNKNOWNS,NUMBER_SNPS) :: UNKNOWN_DATA

INTEGER, ALLOCATABLE, DIMENSION(:,:) :: DATA_ALLELE1,DATA_TOTAL
REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: ALLELE_FREQUENCY !LONG,LAT
REAL(KIND=DBLE_PREC) :: RHO,AVG_DISTANCE,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2
REAL(KIND=DBLE_PREC) :: LONG_WEIGHT,LAT_WEIGHT,SUM_DISTANCE
REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRID
INTEGER, ALLOCATABLE, DIMENSION(:,:) :: TEMP_DATA_ARRAY
LOGICAL, ALLOCATABLE, DIMENSION(:,:) :: IS_LAND
INTEGER, ALLOCATABLE, DIMENSION(:) :: SAMPLE_SIZES
REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: POP_AVG_DISTANCE
REAL(KIND=DBLE_PREC), DIMENSION(2,SAMPLE_SITES) :: COORD !COORD(LONG=1;LAT=2,POPULATION)
INTEGER, ALLOCATABLE, DIMENSION(:) :: MEMBERSHIP


GRID_LENGTH=0

!ALLOCATE(DATA_ARRAY(2,SAMPLE_SITES,NUMBER_SNPS))
ALLOCATE(TEMP_DATA_ARRAY(2,SAMPLE_SITES))
TEMP_DATA_ARRAY=0

CALL UPDATE_GRID_COORD_SQUARE2(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
COORD=TRANSPOSE(DATA_COORD)
!PRINT*, "HERE"
ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)))


!CALL COUNT_UNKNOWN_INDIVIDUALS(DATA_FILE,NUMBER_UNKNOWN,NUMBER_SNPS)
ALLOCATE(UNKNOWN_GRID(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWNS))
UNKNOWN_GRID=0.

ALLOCATE(IS_LAND(GRID_LENGTH(1),GRID_LENGTH(2)))
IS_LAND=.TRUE.
!CALL UPDATE_IS_LAND_SQUARE(IS_LAND,GRID_LENGTH,MAX_GRID_LENGTH)

LONG_WEIGHT=ONE
!	LAT_WEIGHT=CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))/&
!				CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
LAT_WEIGHT=ONE

CALL INTPR("NUMBER_UNKNOWN:",-1,NUMBER_UNKNOWNS,1)
CALL INTPR("SAMPLE_SITES:",-1,SAMPLE_SITES,1)

ALLOCATE(SAMPLE_SIZES(SAMPLE_SITES))
SAMPLE_SIZES=0
DO I=1,NUMBER_UNKNOWNS
!	PRINT*, I, MEMBERSHIP(I)
	SAMPLE_SIZES(MEMBERSHIP(I))=SAMPLE_SIZES(MEMBERSHIP(I))+1
END DO
!PRINT*, "SAMPLE_SIZES: ", SAMPLE_SIZES
!PRINT*, "MEMBERSHIP: ", MEMBERSHIP

ALLOCATE(POP_AVG_DISTANCE(SAMPLE_SITES))
POP_AVG_DISTANCE=0.

RHO_VEC=0.
RHO_VEC(1,:)=10.
ITS=0
PHASE_TWO=0
DO K=1,MAX_ITS
!		CALL POOL_GENO_DATA_SQUARE(DATA_ARRAY(:,:,1),COORD,DATA_ALLELE1,DATA_TOTAL,GRID_COORD,&
!				SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
	RHO=RHO_VEC(1,K)
	DO POP=1,SAMPLE_SITES!NEED TO REDUCE THESE AS WELL...
		UNKNOWN_GRID=1.
		DO I=1,NUMBER_SNPS !NEED TO CHANGE THIS TO 1,NUMBER_SNPS
			!reducing the number of snps for compute time SHOULD GO BACK TO MOD(I,10)
			IF((NUMBER_SNPS.LE.5).OR.(MOD(I,3).EQ.0)) THEN
				ALLELE_FREQUENCY=0.
				TEMP_DATA_ARRAY=DATA_ARRAY(:,:,I)
				TEMP_DATA_ARRAY(:,POP)=0
				CALL POOL_GENO_DATA_SQUARE(TEMP_DATA_ARRAY,TRANSPOSE(DATA_COORD),DATA_ALLELE1,DATA_TOTAL, & 
							GRID_COORD,SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)		
				CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
						LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
!				PRINT*, "FINISHED QNEWT",K,POP,I
				CALL INTPR("ITS:",-1,K,1)
				CALL INTPR("POP:",-1,POP,1)
				CALL INTPR("SNP:",-1,I,1)
				
				!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
				!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
				CALL UPDATE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_DATA(:,I),UNKNOWN_GRID,GRID_LENGTH,&
									MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWNS,IS_LAND)
!				PRINT*, "FINISHED UNKNOWN_GRID"
				!THIS CALCULATES THE DISTANCE BETWEEN THE TRUE VS. ESTIMATED LOCATIONS
				IF((MOD(I,100).EQ.0).OR.(I.EQ.NUMBER_SNPS)) THEN
					SUM_DISTANCE=0.
					TEMP_DISTANCE=0.
					DO J=1,NUMBER_UNKNOWNS !should be LINES
						IF(MEMBERSHIP(J).EQ.POP) THEN
							CALL FIND_OPTIMAL_LOCATION_LOCI(UNKNOWN_GRID(:,:,J),TEMP_REAL1,TEMP_REAL2,&
									MAX_GRID_LENGTH,GRID_COORD,IS_LAND)
							TEMP_DISTANCE=CALC_DISTANCE((/TEMP_REAL1,TEMP_REAL2/),COORD(:,MEMBERSHIP(J)))
							SUM_DISTANCE=SUM_DISTANCE+TEMP_DISTANCE
							!WRITE(OUTPUT_UNIT,*) I,J,TEMP_DISTANCE,TEMP_REAL1,TEMP_REAL2,UNKNOWN_COORD(:,J)
						END IF
					END DO
					AVG_DISTANCE=SUM_DISTANCE/(0.+SAMPLE_SIZES(POP))
					IF(IEEE_IS_NAN(AVG_DISTANCE)) THEN
!						PRINT*, "SUM_DISTANCE: ", SUM_DISTANCE, " SAMPLE_SIZES: ", SAMPLE_SIZES(POP)
						CALL INTPR("SOMETHINGS WRONG...NAN AVERAGE DISTANCE",-1,1,0)
						CALL INTPR("RESULTS ARE PROBABLY INCORRECT...PLEASE CONTACT MAINTAINER",-1,1,0)
					END IF
!					PRINT*, "LOCI: ", I, "AVG_DISTANCE: ", AVG_DISTANCE, "RHO: ",RHO, "POP: ",POP
					IF(I.EQ.NUMBER_SNPS) THEN
						POP_AVG_DISTANCE(POP)=AVG_DISTANCE
					END IF
				END IF
			END IF
		END DO
	END DO

	RHO_VEC(2,K)=SUM(SAMPLE_SIZES*POP_AVG_DISTANCE)/(SUM(SAMPLE_SIZES)+0.)
	
	!HERE WE CHOOSE THE NEXT RHO BASED ON THE PREVIOUS RESULTS...
	IF(K.EQ.1) THEN
		RHO_VEC(1,2)=2*RHO_VEC(1,1)
	ELSE IF(K.EQ.2) THEN
		IF(RHO_VEC(2,K).LT.RHO_VEC(2,K-1)) THEN	
			!PROCEEDING IN THE RIGHT DIRECTION ...DOUBLING
			RHO_VEC(1,K+1)=2*RHO_VEC(1,2)
		ELSE
			!GO THE OTHER WAY... HALVING 1
			RHO_VEC(1,K+1)=0.5*RHO_VEC(1,2)
		END IF
	ELSE IF(K.LT.MAX_ITS) THEN
		IF(PHASE_TWO.EQ.0) THEN
			IF(RHO_VEC(2,K).LT.RHO_VEC(2,K-1)) THEN	
				RHO_VEC(1,K+1)=(RHO_VEC(1,K)/RHO_VEC(1,K-1))*RHO_VEC(1,K)
			ELSE
				PHASE_TWO=1
				RHO_VEC(1,K+1)=(RHO_VEC(1,K)+RHO_VEC(1,K-1))/2.
			END IF
		ELSE
			IF(RHO_VEC(2,K-1).LT.RHO_VEC(2,K-2)) THEN
				RHO_VEC(1,K+1)=(RHO_VEC(1,K)+RHO_VEC(1,K-1))/2.
			ELSE
				RHO_VEC(1,K+1)=(RHO_VEC(1,K)+RHO_VEC(1,K-2))/2.
			END IF
		END IF
	END IF
		
END DO
!PRINT*, "RHO ESTIMATION RESULTS"
!PRINT*, "RHO AVG_DISTANCE"

TEMP_INT=MINLOC(RHO_VEC(2,:),1)
RHO=RHO_VEC(1,TEMP_INT)
AVG_DISTANCE=RHO_VEC(2,TEMP_INT)
!PRINT*, "RHO_VEC: ", RHO_VEC
!PRINT*, "RHO: ", RHO

END SUBROUTINE LEAVE_ONE_POP_OUT_CROSSVAL_SQUARE2

	
	







SUBROUTINE FITADMIXEDMODELFINDUNKNOWNS(UNKNOWN_GRIDS,DATA_ARRAY,NUMBER_SNPS,GRID_LENGTH,RHO,&
		SAMPLE_SITES,MAX_GRID_LENGTH,DATA_COORD,GRID_COORD,NUMBER_UNKNOWN,UNKNOWN_DATA,IS_LAND)
	!(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO)
!THIS SUBROUTINE READS IN THE DATA AND FITS THE MODEL

USE CONSTANTS
USE QUASINEWTON
USE ALLELEFREQUENCIES
USE ORIGENMOD

IMPLICIT NONE

INTEGER :: I,J,K,SAMPLE_SITES,NUMBER_SNPS,MAX_GRID_LENGTH,NUMBER_UNKNOWN,NODES
INTEGER, DIMENSION(2) :: GRID_LENGTH
INTEGER, DIMENSION(2,SAMPLE_SITES,NUMBER_SNPS) :: DATA_ARRAY !(FIRST IS ALLELE, population, 3RD IS LOCUS)
!REAL(KIND=DBLE_PREC), DIMENSION(NUMBER_SNPS,GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY_SURFACES
REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES,2) :: DATA_COORD !(LONG,LAT)
REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
INTEGER, DIMENSION(NUMBER_UNKNOWN,NUMBER_SNPS) :: UNKNOWN_DATA
REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN) :: UNKNOWN_GRIDS
INTEGER, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_ALLELE1,DATA_TOTAL
REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: ALLELE_FREQUENCY,GENOME_FRACTIONS!,QNEWT_FREQ !LONG,LAT
REAL(KIND=DBLE_PREC) :: RHO,LONG_WEIGHT,LAT_WEIGHT,TEMP_REAL,LAMBDA
REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:) :: TEMP_VECTOR
LOGICAL, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: IS_LAND
REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: POP_MATRIX


!PRINT*, "FIT ADMIXED MODEL FIND UNKNOWNS"

!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)

!ALLOCATE(DATA_ARRAY(2,SAMPLE_SITES,NUMBER_SNPS),COORD(2,SAMPLE_SITES))
!DATA_ARRAY=0
!COORD=0.

!CALL READ_GENO_DATA(DATA_FILE,DATA_ARRAY,COORD,NUMBER_SNPS,SAMPLE_SITES)

!CALL UPDATE_GRID_COORD_SQUARE(GRID_COORD,COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
!ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)),QNEWT_FREQ(GRID_LENGTH(1),GRID_LENGTH(2)))
!	CALL UPDATE_GRID_COORD(GRID_COORD,COORD,GRID_LENGTH,SAMPLE_SITES)
LONG_WEIGHT=ONE
LAT_WEIGHT=ONE
LAMBDA=100.
!	LONG_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))
!	LAT_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
!ALLELE_FREQUENCY_SURFACES=ZERO

!IS_LAND(:,:)=.TRUE.
NODES=COUNT(IS_LAND)
ALLOCATE(POP_MATRIX(NUMBER_SNPS,NODES))
ALLOCATE(TEMP_VECTOR(NODES))

DO I=1,NUMBER_SNPS
	ALLELE_FREQUENCY=0.
	CALL POOL_GENO_DATA_SQUARE(DATA_ARRAY(:,:,I),TRANSPOSE(DATA_COORD),DATA_ALLELE1,DATA_TOTAL,GRID_COORD,&
			SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
!		CALL FIT_L2_MODEL_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
!				LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
	CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY,RHO,&
			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)
	
	!FILL IN THE UNKNOWN GRID(LISTS PROBABILITIES OF UNKNOWN INDIVIDUALS BELONGING TO EACH LOCATION)
	!THIS TAKES LESS MEMORY THAN STORING ALL OF THE ALLELE FREQUENCIES
!	CALL UPDATE_UNKNOWN_GRID_SQUARE(ALLELE_FREQUENCY,UNKNOWN_DATA(:,I),UNKNOWN_GRIDS,GRID_LENGTH,&
!						MAX_GRID_LENGTH,SAMPLE_SITES,NUMBER_UNKNOWN)
	POP_MATRIX(I,:)=RESHAPE_MATRIX_MASK_SQUARE(ALLELE_FREQUENCY,IS_LAND,GRID_LENGTH,NODES)
	
END DO

DO I=1,NUMBER_UNKNOWN
	ALLELE_FREQUENCY=ONE/NODES	
	TEMP_VECTOR=RESHAPE_MATRIX_MASK_SQUARE(ALLELE_FREQUENCY,IS_LAND,GRID_LENGTH,NODES)
	CALL FIT_FRACTIONS_MODEL_QNEWT(TEMP_VECTOR,POP_MATRIX,&
		UNKNOWN_DATA(I,:),LAMBDA,NUMBER_SNPS,NODES)
!		GENOME_FRACTIONS=RESHAPE(TEMP_VECTOR,SHAPE(GENOME_FRACTIONS))
	GENOME_FRACTIONS=RESHAPE_VECTOR_MASK_SQUARE(TEMP_VECTOR,IS_LAND,GRID_LENGTH,NODES)
	!UNKNOWNS GRIDS IS THE HOLDER FOR THE FRACTIONS MODEL...
	UNKNOWN_GRIDS(:,:,I) = GENOME_FRACTIONS(:,:)
END DO


END SUBROUTINE FITADMIXEDMODELFINDUNKNOWNS




!
!
!	SUBROUTINE GENERATE_MULTINOMIAL_DATA_FILES(NUMBER_SAMPLES,NUMBER_MARKERS,MAX_ALLELES)
!	!THIS SUBROUTINE GENERATES RANDOM DATA FILES WITH COORDINATES SOMEWHERE IN AFRICA
!	!MISSING DATA IS -999, UNKNOWN LOCATION IS -1
!	
!	USE CONSTANTS
!	
!	IMPLICIT NONE
!	
!	CHARACTER(LEN=800) :: TEXT_FILE,TEMP_NAME,TEMP_TEXT
!	INTEGER :: I,J,K,L,OUTPUT_UNIT=23,NUMBER_SAMPLES,MAX_ALLELES,NUMBER_MARKERS,TEMP_INT,TEMP_INT2
!	CHARACTER(LEN=800), DIMENSION(NUMBER_SAMPLES) :: LOCATION_NAMES
!	CHARACTER(LEN=800), DIMENSION(NUMBER_MARKERS) :: MARKER_NAMES
!	INTEGER, DIMENSION(2,NUMBER_SAMPLES,NUMBER_MARKERS) :: SAMPLE_DATA
!	INTEGER, DIMENSION(NUMBER_MARKERS) :: ALLELES_AT_LOCUS
!	INTEGER, DIMENSION(MAX_ALLELES) :: ALLELE_LENGTHS
!	REAL(KIND=DBLE_PREC) :: RAN1,RAN2
!	
!	PRINT*, "GENERATE RANDOM DATA FILES"
!	
!	LOCATION_NAMES=""
!	
!	DO I=1,NUMBER_SAMPLES
!		RAN1=RAND()
!		TEMP_INT=FLOOR(5*RAN1+5)
!		TEMP_NAME=""
!		DO J=1,TEMP_INT
!			RAN2=RAND()
!			TEMP_INT2=FLOOR(25*RAN2+65)
!			TEMP_NAME=TRIM(TEMP_NAME)//ACHAR(TEMP_INT2)
!		END DO
!		LOCATION_NAMES(I)=TRIM(TEMP_NAME)
!	END DO
!	
!	MARKER_NAMES=""
!	DO I=1,NUMBER_MARKERS
!		WRITE(MARKER_NAMES(I),'(A,I0)') "Locus", I
!	END DO
!	
!	!DECIDES THE NUMBER OF ALLELES IN EACH LOCUS
!	ALLELES_AT_LOCUS=0
!	DO I=1,NUMBER_MARKERS
!		RAN1=RAND()
!		ALLELES_AT_LOCUS(I)=FLOOR(RAN1*(MAX_ALLELES-1))+2
!	END DO
!!	PRINT*, "ALLELES_AT_LOCUS: ", ALLELES_AT_LOCUS
!	
!	!CREATES RANDOM DATA
!	DO I=1,NUMBER_MARKERS
!		!CREATES RANDOM ALLELE LENGTHS FROM 5-1004
!		ALLELE_LENGTHS=0
!		DO L=1,ALLELES_AT_LOCUS(I)
!			RAN1=RAND()
!			ALLELE_LENGTHS(L)=5+FLOOR(RAN1*1000)
!		END DO
!!		PRINT*, "ALLELE_LENGTHS: ", ALLELE_LENGTHS
!		DO J=1,NUMBER_SAMPLES
!			DO K=1,2
!				RAN1=RAND()
!				TEMP_INT=1+FLOOR(RAN1*ALLELES_AT_LOCUS(I))
!				SAMPLE_DATA(K,J,I)=ALLELE_LENGTHS(TEMP_INT)
!			END DO
!		END DO
!!		PRINT*, "SAMPLE_DATA: ", SAMPLE_DATA(1:2,:,I)
!	END DO
!	
!	
!	TEXT_FILE="MicrosatTrialData.txt"
!	OPEN(UNIT=OUTPUT_UNIT,FILE=TEXT_FILE)
!	
!	WRITE(OUTPUT_UNIT,'(30(A,5X))') "LocationName", "LocationNumber", (TRIM(MARKER_NAMES(I)),I=1,NUMBER_MARKERS)
!	
!	DO I=1,NUMBER_SAMPLES
!		WRITE(OUTPUT_UNIT,'(A,5X,I0,5X,100(I0,5X))') TRIM(LOCATION_NAMES(I)),I,(SAMPLE_DATA(1,I,J),J=1,NUMBER_MARKERS)
!		WRITE(OUTPUT_UNIT,'(A,5X,I0,5X,100(I0,5X))') TRIM(LOCATION_NAMES(I)),I,(SAMPLE_DATA(2,I,J),J=1,NUMBER_MARKERS)
!	END DO
!	
!	!here we put in an unknown individual with data exactly the same as the first row
!	WRITE(OUTPUT_UNIT,'(A,5X,I0,5X,100(I0,5X))') "Unknown1",-1,(SAMPLE_DATA(1,1,J),J=1,NUMBER_MARKERS)
!	WRITE(OUTPUT_UNIT,'(A,5X,I0,5X,100(I0,5X))') "Unknown1",-1,(SAMPLE_DATA(2,1,J),J=1,NUMBER_MARKERS)	
!	!here we put in another unknown individual with data that is a mix of the first row of person 1 and person 2
!	WRITE(OUTPUT_UNIT,'(A,5X,I0,5X,100(I0,5X))') "Unknown2",-1,(SAMPLE_DATA(1,1,J),J=1,NUMBER_MARKERS)
!	WRITE(OUTPUT_UNIT,'(A,5X,I0,5X,100(I0,5X))') "Unknown2",-1,(SAMPLE_DATA(1,2,J),J=1,NUMBER_MARKERS)	
!
!	CLOSE(OUTPUT_UNIT)
!	
!	
!	TEXT_FILE="LocationTrialData.txt"
!	OPEN(UNIT=OUTPUT_UNIT,FILE=TEXT_FILE)
!	WRITE(OUTPUT_UNIT,'(4(A,5X))') "LocationName", "LocationNumber", "Latitude", "Longitude"
!	DO I=1,NUMBER_SAMPLES
!		RAN1=RAND()
!		RAN2=RAND()
!		WRITE(OUTPUT_UNIT,'(A,5X,I0,5X,F8.4,5X,F8.4)') TRIM(LOCATION_NAMES(I)),I,RAN1*26-14,RAN2*27+13
!	END DO
!	
!	
!	CLOSE(OUTPUT_UNIT)
!	
!	
!	END SUBROUTINE GENERATE_MULTINOMIAL_DATA_FILES
!	
!	
!	
!	
!	
!	
!	
!	SUBROUTINE READ_MULTINOMIAL_DATA_FILES(MAX_ALLELES,NUMBER_SAMPLES,NUMBER_MARKERS,DATA_COORD,DATA_ARRAY)
!	!THIS SUBROUTINE READS IN THE DATA FILES GENERATED BY "GENERATE_DATA_FILES" after being read in and written separately by R
!	!In short, this is for testing only...
!	
!	USE CONSTANTS
!	USE ALLELEFREQUENCIES 
!	
!	IMPLICIT NONE
!	
!	CHARACTER(LEN=800) :: TEXT_FILE,TEMP_NAME,TEMP_TEXT,LINE
!	INTEGER :: I,J,K,L,OUTPUT_UNIT=23,NUMBER_SAMPLES,MAX_ALLELES,NUMBER_MARKERS,TEMP_INT,TEMP_INT2
!	INTEGER :: INPUT_UNIT=24,LINES,COUNTER,IOERROR
!	REAL(KIND=DBLE_PREC), DIMENSION(NUMBER_SAMPLES,2) :: DATA_COORD
!	REAL :: REAL1,REAL2
!	INTEGER, DIMENSION(MAX_ALLELES,NUMBER_SAMPLES,NUMBER_MARKERS) :: DATA_ARRAY
!	INTEGER, DIMENSION(MAX_ALLELES*NUMBER_SAMPLES) :: TEMP_ARRAY
!	
!	TEXT_FILE="ConvertedTrialData.txt"
!	CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!	
!	!skip first line
!	LINE = " "
!	READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
!	
!	DO J = 1,LINES-1
!		READ(INPUT_UNIT,*) TEMP_NAME,(TEMP_ARRAY(I),I=1,MAX_ALLELES*NUMBER_SAMPLES)
!		COUNTER=0
!		DO I=1,MAX_ALLELES
!			DO K=1,NUMBER_SAMPLES
!				COUNTER=COUNTER+1
!				DATA_ARRAY(I,K,J)=TEMP_ARRAY(COUNTER)
!			END DO
!		END DO
!	END DO
!	PRINT*, "DATA_ARRAY(1,,1): ", DATA_ARRAY(1,:,1)
!	
!	CLOSE(INPUT_UNIT)
!	
!	
!	
!	TEXT_FILE="ConvertedTrialCoord.txt"
!	CALL INPUT_DATA(TEXT_FILE,LINE,INPUT_UNIT,LINES)
!	REWIND(INPUT_UNIT)
!	
!	!skip first line
!	LINE = " "
!	READ(INPUT_UNIT,'(800A)',IOSTAT=IOERROR) (LINE(I:I),I=1,LEN(LINE))
!	
!	DO J = 1,LINES-1
!		READ(INPUT_UNIT,*) TEMP_NAME,REAL1,REAL2
!		DATA_COORD(J,1)=REAL1
!		DATA_COORD(J,2)=REAL2
!		PRINT*, "REAL1: ", REAL1, "  REAL2: ",REAL2
!	END DO
!	PRINT*, "DATA_COORD(1,): ", DATA_COORD(1,:)
!	CLOSE(INPUT_UNIT)
!	
!	END SUBROUTINE READ_MULTINOMIAL_DATA_FILES
!

	

	

	SUBROUTINE FITMULTINOMIALMODEL(MULTI_FREQUENCY_SURFACES,DATA_ARRAY,RHO,SAMPLE_SITES,&
			GRID_LENGTH,MAX_GRID_LENGTH,MAX_ALLELES,NUMBER_MARKERS,DATA_COORD,&
			ALLELES_AT_LOCUS,GRID_COORD)
		!THIS SUBROUTINE UPDATES THE ALLELE_FREQUENCY MATRIX ONCE USING MM UPDATES
			
	USE CONSTANTS
	USE MULTINOMIAL
	
	IMPLICIT NONE
	
	INTEGER :: H,I,J,K,MAX_GRID_LENGTH,MAX_ALLELES,ITERATIONS,NUMBER_MARKERS,SAMPLE_SITES,ALLELES
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(MAX_ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_MULTI
	INTEGER, DIMENSION(MAX_ALLELES,SAMPLE_SITES,NUMBER_MARKERS) :: DATA_ARRAY 
	INTEGER, DIMENSION(NUMBER_MARKERS) :: ALLELES_AT_LOCUS
	!DATA_VECTOR IS A DATA MATRIX GIVING THE NUMBER OF EACH ALLELE FOUND AT EACH SITE....
	REAL(KIND=DBLE_PREC), DIMENSION(MAX_ALLELES,GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_MARKERS) :: MULTI_FREQUENCY_SURFACES
	REAL(KIND=DBLE_PREC) :: RHO,LONG_WEIGHT,LAT_WEIGHT,LOGLIK,NEW_LOGLIK
	LOGICAL :: NOT_CONVERGED
	REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES,2) :: DATA_COORD !(LONG,LAT)
	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
	
	
	!PRINT*, "FITORIGENMODEL"
	
	!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
	
	!ALLOCATE(DATA_ARRAY(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
	!DATA_ARRAY=0
	!DATA_COORD=0.
	
	!CALL READ_GENO_DATA(DATA_FILE,DATA_ARRAY,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
	
	!CALL UPDATE_GRID_COORD_SQUARE(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
	!ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
	!ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)),QNEWT_FREQ(GRID_LENGTH(1),GRID_LENGTH(2)))
	!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
	LONG_WEIGHT=ONE
	LAT_WEIGHT=ONE
	MULTI_FREQUENCY_SURFACES=ZERO
	!	LONG_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))
	!	LAT_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
	!ALLELE_FREQUENCY_SURFACES=ZERO
	
	DO I=1,NUMBER_MARKERS
	!	PRINT*, "READ AND FIT-SNP: ", I
		CALL INTPR("CURRENT MARKER",-1,I,1)
		ALLELES=ALLELES_AT_LOCUS(I)
	!	CALL POOL_MULTI_DATA(DATA_ARRAY(:,:,I),TRANSPOSE(DATA_COORD),DATA_ALLELE1,DATA_TOTAL,&
	!			GRID_COORD,SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
		CALL POOL_MULTI_DATA(DATA_MULTI(1:ALLELES,:,:),DATA_ARRAY(1:ALLELES,:,I),TRANSPOSE(DATA_COORD),SAMPLE_SITES,&
						GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,GRID_COORD)
!		PRINT*, "DATA_MULTI"
!		DO J=1,GRID_LENGTH(2)
!			PRINT*, (DATA_MULTI(1,K,J),K=1,GRID_LENGTH(1))
!		END DO
			
	!	CALL FIT_L2_MODEL_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY_SURFACES(I,:,:),RHO,&
	!			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,.TRUE.)
		
	!	CALL FIT_L2_MODEL_QNEWT_SQUARE(DATA_ALLELE1,DATA_TOTAL,ALLELE_FREQUENCY_SURFACES(I,:,:),RHO,&
	!			LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH)	
				
		CALL FIT_MULTI_MODEL(DATA_MULTI(1:ALLELES,:,:),MULTI_FREQUENCY_SURFACES(1:ALLELES,:,:,I),RHO,&
						LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,.TRUE.)	
						
!		CALL INITIALIZE_MULTI_FREQUENCY(DATA_MULTI,MULTI_FREQUENCY_SURFACES(:,:,:,I),&
!					GRID_LENGTH,MAX_GRID_LENGTH,ALLELES)
			
	END DO
	
	END SUBROUTINE FITMULTINOMIALMODEL




	
	
	SUBROUTINE FITMULTINOMIALMODELFIND(UNKNOWN_GRIDS,DATA_ARRAY,RHO,SAMPLE_SITES,&
			GRID_LENGTH,MAX_GRID_LENGTH,MAX_ALLELES,NUMBER_MARKERS,DATA_COORD,&
			GRID_COORD,ALLELES_AT_LOCUS,NUMBER_UNKNOWN,UNKNOWN_DATA)
		!THIS SUBROUTINE UPDATES THE ALLELE_FREQUENCY MATRIX ONCE USING MM UPDATES
	
	USE CONSTANTS
	USE MULTINOMIAL
			
	IMPLICIT NONE
	
	INTEGER :: H,I,J,K,MAX_GRID_LENGTH,ALLELES,ITERATIONS,NUMBER_MARKERS,SAMPLE_SITES
	INTEGER :: NUMBER_UNKNOWN,MAX_ALLELES
	INTEGER, DIMENSION(2) :: GRID_LENGTH
	INTEGER, DIMENSION(MAX_ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: DATA_MULTI
	INTEGER, DIMENSION(MAX_ALLELES,SAMPLE_SITES,NUMBER_MARKERS) :: DATA_ARRAY 
	INTEGER, DIMENSION(NUMBER_UNKNOWN,2,NUMBER_MARKERS) :: UNKNOWN_DATA
	INTEGER, DIMENSION(NUMBER_MARKERS) :: ALLELES_AT_LOCUS
	REAL(KIND=DBLE_PREC), DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN) :: UNKNOWN_GRIDS	!DATA_VECTOR IS A DATA MATRIX GIVING THE NUMBER OF EACH ALLELE FOUND AT EACH SITE....
	REAL(KIND=DBLE_PREC), DIMENSION(MAX_ALLELES,GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_MARKERS) :: MULTI_FREQUENCY_SURFACES
	REAL(KIND=DBLE_PREC), DIMENSION(MAX_ALLELES,GRID_LENGTH(1),GRID_LENGTH(2)) :: MULTI_FREQUENCY
	REAL(KIND=DBLE_PREC) :: RHO,LONG_WEIGHT,LAT_WEIGHT,LOGLIK,NEW_LOGLIK
	LOGICAL :: INIT_BOOL,NOT_CONVERGED
	REAL(KIND=DBLE_PREC), DIMENSION(SAMPLE_SITES,2) :: DATA_COORD !(LONG,LAT)
	REAL(KIND=DBLE_PREC), DIMENSION(2,MAX_GRID_LENGTH) :: GRID_COORD !LONG,LAT
	LOGICAL, DIMENSION(GRID_LENGTH(1),GRID_LENGTH(2)) :: IS_LAND
	
	
	!PRINT*, "FITORIGENMODEL"
	
	!	CALL COUNT_SAMPLE_SITES(DATA_FILE,SAMPLE_SITES,NUMBER_SNPS)
	
	!ALLOCATE(DATA_ARRAY(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(2,SAMPLE_SITES))
	!DATA_ARRAY=0
	!DATA_COORD=0.
	
	!CALL READ_GENO_DATA(DATA_FILE,DATA_ARRAY,DATA_COORD,NUMBER_SNPS,SAMPLE_SITES)
	
	!CALL UPDATE_GRID_COORD_SQUARE(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
	!ALLOCATE(DATA_ALLELE1(GRID_LENGTH(1),GRID_LENGTH(2)),DATA_TOTAL(GRID_LENGTH(1),GRID_LENGTH(2)))
	!ALLOCATE(ALLELE_FREQUENCY(GRID_LENGTH(1),GRID_LENGTH(2)),QNEWT_FREQ(GRID_LENGTH(1),GRID_LENGTH(2)))
	!	CALL UPDATE_GRID_COORD(GRID_COORD,DATA_COORD,GRID_LENGTH,SAMPLE_SITES)
	LONG_WEIGHT=ONE
	LAT_WEIGHT=ONE
	IS_LAND=.TRUE.
	UNKNOWN_GRIDS=ONE/(GRID_LENGTH(1)*GRID_LENGTH(2))
	!	LONG_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,2),GRID_COORD(2,1)/))
	!	LAT_WEIGHT=ONE/CALC_DISTANCE(GRID_COORD(:,1),(/GRID_COORD(1,1),GRID_COORD(2,2)/))
	!ALLELE_FREQUENCY_SURFACES=ZERO
	
	DO I=1,NUMBER_MARKERS
	!	PRINT*, "READ AND FIT-SNP: ", I
		CALL INTPR("CURRENT MARKER",-1,I,1)
		ALLELES=ALLELES_AT_LOCUS(I)
	!	CALL POOL_MULTI_DATA(DATA_ARRAY(:,:,I),TRANSPOSE(DATA_COORD),DATA_ALLELE1,DATA_TOTAL,&
	!			GRID_COORD,SAMPLE_SITES,GRID_LENGTH,MAX_GRID_LENGTH)
		CALL POOL_MULTI_DATA(DATA_MULTI,DATA_ARRAY(:,:,I),TRANSPOSE(DATA_COORD),SAMPLE_SITES,&
						GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,GRID_COORD)
!		PRINT*, "DATA_MULTI"
!		DO J=1,GRID_LENGTH(2)
!			PRINT*, (DATA_MULTI(1,K,J),K=1,GRID_LENGTH(1))
!		END DO
						
		CALL FIT_MULTI_MODEL(DATA_MULTI,MULTI_FREQUENCY,RHO,&
						LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,.TRUE.)	
		
!		CALL INITIALIZE_MULTI_FREQUENCY(DATA_MULTI,MULTI_FREQUENCY,&
!					GRID_LENGTH,MAX_GRID_LENGTH,ALLELES)
						
		CALL UPDATE_UNKNOWN_GRIDS_MULTI(DATA_MULTI,MULTI_FREQUENCY,RHO,UNKNOWN_DATA(:,:,I),UNKNOWN_GRIDS,&
					LONG_WEIGHT,LAT_WEIGHT,GRID_LENGTH,MAX_GRID_LENGTH,ALLELES,NUMBER_UNKNOWN,IS_LAND)				
		
		MULTI_FREQUENCY_SURFACES(:,:,:,I)=MULTI_FREQUENCY
	END DO
	
	END SUBROUTINE FITMULTINOMIALMODELFIND
	
	




!
!PROGRAM ORIGEN
!
!!
!!This program takes observations (allele frequencies with a similar program for isotope data) along with their locations
!!and refines the estimates by taking into account the estimates of all other points weighted by their
!!distance to other points.  Using these estimates we can then estimate the value of the quantity at
!!unobserved locations using an interpolation scheme.  From here we can then find the most likely
!!location of a new sample based solely on the value of the measurement of this quantity.
!
!!There are two (possibly more) inputs for this program.  The first is a matrix of the observed values
!!along with their locations(OBS(Population,ALLELE)) for a single locus.
!!The second is the location coordinates per population (COORD(LAT/LONG,POPULATION))
!!OBS2 is a measurement which we would like to localize (ie find
!!the coordinates that they most likely came from).  A third (optional) input would be locations to
!!try(TRY_COORD).  A fourth (optional) input would  be the names of the Alleles or isotopes.
!!
!!The output of this program is a list of coordinates followed by the probability that the sample (OBS2)
!!came from that location.
!
!	USE CONSTANTS
!	USE QUASINEWTON
!	USE ALLELEFREQUENCIES
!	USE ORIGENMOD
!	USE MULTINOMIAL
!
!	IMPLICIT NONE
!!
!	CHARACTER(LEN=800) :: TEXT_FILE,DATA_FILE,LOCATION_FILE
!	REAL(KIND=DBLE_PREC) :: AVG_DISTANCE,RHO
!	INTEGER :: I,J,K,OUTPUT_UNIT=3,NUMBER_SNPS,MAX_GRID_LENGTH,SAMPLE_SITES
!	INTEGER :: MAX_ITS,NUMBER_UNKNOWN,NUMBER_KNOWN
!	INTEGER, DIMENSION(2) :: GRID_LENGTH
!	LOGICAL, ALLOCATABLE, DIMENSION(:,:) :: IS_LAND
!	CHARACTER(LEN=1), DIMENSION(5) :: TRIAL_CHARS
!	INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: DATA_ARRAY !(FIRST IS ALLELE, population, 3RD IS LOCUS)
!	INTEGER, ALLOCATABLE, DIMENSION(:,:) :: UNKNOWN_DATA
!	INTEGER, ALLOCATABLE, DIMENSION(:) :: RANKINGS,MEMBERSHIP
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: UNKNOWN_GRIDS
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:,:) :: ALLELE_FREQUENCY_SURFACES
!	REAL(KIND=DBLE_PREC), ALLOCATABLE, DIMENSION(:,:) :: GRID_COORD,RHO_VEC,TEST_STATS,DATA_COORD !(LONG,LAT)
!	CHARACTER(LEN=50), ALLOCATABLE, DIMENSION(:) :: MEMBERS
!	
!	DATA_FILE="10SNPs"
!	LOCATION_FILE="PC1PC2ED.txt"
!	NUMBER_SNPS=10
!	SAMPLE_SITES=37
!	MAX_ITS=6
!	MAX_GRID_LENGTH=20
!	GRID_LENGTH=MAX_GRID_LENGTH
!	ALLOCATE(RHO_VEC(2,MAX_ITS))
!	RHO_VEC=0
!	RHO=10.
!	CALL COUNT_NUMBER_LINES(NUMBER_SNPS,TRIM(DATA_FILE)//".map")
!	CALL COUNT_NUMBER_SAMPLE_SITES(SAMPLE_SITES,LOCATION_FILE)
!	CALL COUNT_NUMBER_LINES(NUMBER_KNOWN,TRIM(DATA_FILE)//".ped")
!	NUMBER_UNKNOWN=NUMBER_KNOWN
!	ALLOCATE(DATA_ARRAY(2,SAMPLE_SITES,NUMBER_SNPS),DATA_COORD(SAMPLE_SITES,2))
!	ALLOCATE(MEMBERS(SAMPLE_SITES),RANKINGS(NUMBER_SNPS),TEST_STATS(2,NUMBER_SNPS))
!	ALLOCATE(UNKNOWN_DATA(NUMBER_UNKNOWN,NUMBER_SNPS),MEMBERSHIP(NUMBER_UNKNOWN))
!	ALLOCATE(GRID_COORD(2,MAX_GRID_LENGTH))
!	RANKINGS=0
!	TEST_STATS=0
!	PRINT*, NUMBER_SNPS, SAMPLE_SITES
!	PRINT*, "BEGIN ORIGEN"
!	
!	
!!	CALL UPDATE_GRID_COORD_SQUARE2(GRID_COORD,DATA_COORD,GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!!
!!	ALLOCATE(ALLELE_FREQUENCY_SURFACES(NUMBER_SNPS,GRID_LENGTH(1),GRID_LENGTH(2)))
!!
!!	CALL FORMAT_PLINK_DATA(DATA_ARRAY,DATA_COORD,TRIM(DATA_FILE)//".ped",LOCATION_FILE,SAMPLE_SITES,&
!!		NUMBER_SNPS)
!!
!!	CALL FITORIGENMODEL(ALLELE_FREQUENCY_SURFACES,DATA_ARRAY,NUMBER_SNPS,GRID_LENGTH,RHO,&
!!		SAMPLE_SITES,MAX_GRID_LENGTH,DATA_COORD,GRID_COORD)
!
!
!
!	CALL FORMAT_UNKNOWN_PLINK_DATA(DATA_ARRAY,DATA_COORD,TRIM(DATA_FILE)//".ped",LOCATION_FILE,SAMPLE_SITES,NUMBER_SNPS,&
!			TRIM(DATA_FILE)//".ped",NUMBER_UNKNOWN,UNKNOWN_DATA,MEMBERSHIP,NUMBER_KNOWN)
!	CALL UPDATE_GRID_COORD_SQUARE2(GRID_COORD,TRANSPOSE(DATA_COORD),GRID_LENGTH,MAX_GRID_LENGTH,SAMPLE_SITES)
!	
!	!REDUCING NUMBER UNKNOWN
!	NUMBER_UNKNOWN=2
!	ALLOCATE(UNKNOWN_GRIDS(GRID_LENGTH(1),GRID_LENGTH(2),NUMBER_UNKNOWN))		
!	CALL FITADMIXEDMODELFINDUNKNOWNS(UNKNOWN_GRIDS,DATA_ARRAY,NUMBER_SNPS,GRID_LENGTH,RHO,&
!		SAMPLE_SITES,MAX_GRID_LENGTH,DATA_COORD,GRID_COORD,NUMBER_UNKNOWN,UNKNOWN_DATA(1:NUMBER_UNKNOWN,:))
!	PRINT*, "SUM UNKNOWN: ", SUM(UNKNOWN_GRIDS(:,:,1))
!	PRINT*, UNKNOWN_GRIDS(:,:,1)
!	
!	
!	
!!	PRINT*, "BEGIN RANKINGS"
!!	CALL CALC_ALL_RANKINGS(DATA_ARRAY,SAMPLE_SITES,NUMBER_SNPS,RANKINGS,TEST_STATS)
!!	CALL LEAVE_ONE_POP_OUT_CROSSVAL_SQUARE(TRIM(DATA_FILE)//".ped",LOCATION_FILE,NUMBER_SNPS,MAX_ITS,&
!!		MAX_GRID_LENGTH,RHO_VEC,RHO,SAMPLE_SITES)
!!	PRINT*, "COORD", (COORD(I,:),I=1,SAMPLE_SITES)
!!	PRINT*, "DATA_ARRAY: ", DATA_ARRAY(1,1:3,1:4)
!	PRINT*, "END ORIGEN"
!!	
!!	GRID_LENGTH=50
!!	MAX_GRID_LENGTH=50
!!	RHO=10.0
!!	AVG_DISTANCE=0.
!!!	ALLOCATE(IS_LAND(GRID_LENGTH,GRID_LENGTH))
!!!	CALL UPDATE_IS_LAND(IS_LAND,GRID_LENGTH)
!!	PRINT*, "RHO: ",RHO
!!	
!!	NUMBER_SNPS=3000 !MAX IS 197146
!!!	DATA_FILE="AdmixedData"
!!	DATA_FILE="NormalNewInf"
!!	!WRITE(TEMP_TEXT,'(I0)') NUMBER_SNPS
!!	
!!!	CALL WEED_DATA_OURS_ONLY_FULL2(NUMBER_SNPS,DATA_FILE)
!!!	CALL READ_FIT_SQUARE(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO)
!!!	CALL READ_FIT_FIND_SQUARE(DATA_FILE,NUMBER_SNPS,GRID_LENGTH,RHO,AVG_DISTANCE)
!!
!!!	CALL LEAVE_ONE_POP_OUT_CROSSVAL_SQUARE(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO,AVG_DISTANCE)
!!!	CALL LEAVE_ONE_POP_OUT_SQUARE(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO,AVG_DISTANCE)
!!!	CALL LEAVE_ONE_INDIV_OUT_CROSSVAL_SQUARE(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO,AVG_DISTANCE)
!!!	CALL LEAVE_ONE_INDIV_OUT_SQUARE(DATA_FILE,NUMBER_SNPS,MAX_GRID_LENGTH,RHO,AVG_DISTANCE)
!!
!!
!!!	CALL TEST_GENO_PARS(DATA_FILE,NUMBER_SNPS,GRID_LENGTH,RHO)
!!
!!
!END PROGRAM ORIGEN

