#' @title Plot the response column diagram
#' @description The function `get_response_plot` uses to plot the column diagram of drug response.
#' @param km_data A data frame, including survival status, survival time, and risk score of each sample. The data frame can be generated by the function `get_risk_score`.
#' @param cut_point The threshold uses to classify patients into two subgroups with different OS.
#' @param response Response status of the sample to the drug.
#' @param TRAIN Logical,if set to TRUE,the 'cut_point' is generated by the median of the risk score; Otherwise,'cut_point' can be customized.
#' @importFrom graphics barplot
#' @importFrom stats chisq.test
#' @export
#' @return Comparison of the objective response rate between the high-risk and low-risk groups, plot the bar graph and return the p value.
#' @examples
#' #Load the data.
#' data(km_data,response)
#' #perform the function `get_response_plot`.
#' get_response_plot(km_data,response,cut_point,TRAIN=TRUE)




get_response_plot<-function(km_data,response,cut_point,TRAIN=TRUE){
  newdata<-matrix(ncol = 3,nrow=dim(km_data)[1])
  rownames(newdata)<-rownames(km_data)
  newdata[,1]<-km_data$survival
  newdata[,2]<-km_data$event
  newdata[,3]<-km_data$multiple_score
  colnames(newdata)<-c("survival","event","multiple_score")
  newdata[,3]<-as.numeric(as.matrix(newdata[,3]))
  if(TRAIN){
    cut_point<-quantile(newdata[,3],probs = seq(0,1,1/2),na.rm = F,name=F,type=7,digits = 7)[2]
  }else
    cut_point=cut_point
  high_samp<-response[match(rownames(newdata)[which(newdata[,3]>cut_point)],response$Sample),]
  low_samp<-response[match(rownames(newdata)[which(newdata[,3]<= cut_point)],response$Sample),]
  high_res<-length(which(high_samp$Response%in%c("PR","CR","response")))
  high_nores<-length(which(high_samp$Response%in%c("PD","SD","nonresponse")))
  low_res<-length(which(low_samp$Response%in%c("PR","CR","response")))
  low_nores<-length(which(low_samp$Response%in%c("PD","SD","nonresponse")))
  res_matrix<-matrix(c(high_res/(high_res+high_nores),high_nores/(high_res+high_nores),
                       low_res/(low_res+low_nores),low_nores/(low_res+low_nores)),ncol=2)
  colnames(res_matrix)<-c("high","low")
  rownames(res_matrix)<-c("response","no_response")
  barplot(res_matrix,col=c("pink","lightblue"))
  res_matrix1<-matrix(c(high_res,high_nores,
                        low_res,low_nores),ncol=2)
  chisq.test(res_matrix1)
}

