% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prosper-models.R, R/prosper.ECHCG.R,
%   R/prosper.GALAP.R, R/prosper.LOLRI.R
\name{prosper-models}
\alias{prosper-models}
\alias{param.ECHCG}
\alias{param.GALAP}
\alias{param.LOLRI}
\alias{prosper.ECHCG}
\alias{prosper.GALAP}
\alias{prosper.LOLRI}
\title{Population dynamic models - Examples}
\usage{
prosper.ECHCG(param.weed = PROSPER::param.ECHCG, area = NA, af = NA,
  dom = NA, epis = 0, put = 0.05, sdrate = 0.4, thresh = 20,
  Rmx = 10, rate = 100, duration = NA, repetitions = NA,
  crop_list = "corn", max_vec_length = 1e+07, undersowing = NA)

prosper.GALAP(param.weed = PROSPER::param.GALAP, sens_seeds = 400,
  area = 100, af = c(0.03, 0.08, 0.02), dom = c(0.5, 0.5, 0.5),
  epis = 0, put = 0.05, thresh = 20, Rmx = 10, rate = 100,
  sdrate = 0.4, duration = 15, repetitions = 1,
  crop_list = "wheat", max_vec_length = 1e+07)

prosper.LOLRI(param.weed = PROSPER::param.LOLRI, area = 100,
  af = c(0.005, 0.01, 0.015, 0.02), dom = 0.5, epis = 0,
  put = 0.05, sdrate = 0.4, thresh = 20, Rmx = 10, dc = 150,
  kc = 1/11, kw = 1/33, SSmax = 30000, rate = 100, pen_co = c(1,
  0.5), duration = 10, repetitions = 1, crop_list = c("wheat"),
  max_vec_length = 1e+07)
}
\arguments{
\item{param.weed}{A \code{data.frame} with population dynamic parameters with or without stochasticity. The structure of \code{param.weed} is essential (see details). The easiest way to create the \code{data.frame} is to adopt an example (\link{param.ECHCG}).}

\item{area}{number of area units. \code{positive numeric}.}

\item{af}{initial frequency of resistance alleles in the population. \code{numeric vector} with length of \code{ng} (number of genes) and elements in [0,1].}

\item{dom}{dominance of resistance alleles. \code{numeric vector} with length of \code{n_loci} (number of genes) and elements in [0,1].}

\item{epis}{epistasis value, describing the interaction between resistance alleles. \code{numeric}. \code{epis} = 0: no interaction i.e. additive effects of resistance alleles, \code{epis} < 0: effect of resistance alleles is smaller than additive, \code{epis} > 0: effect of resistance alleles is higher than additive.}

\item{put}{probability of a weed to be untouched by the herbicide. \code{numeric}, \eqn{0 \le \code{put} \le 1}.}

\item{sdrate}{variance of the herbicide rate reaching the weed. \code{positive numeric}, 1 = 1 unit standard deviation.}

\item{thresh}{threshold herbicide rate to kill weeds without resistance. \code{numeric}, \eqn{0 \le \code{thresh} \le 1}.}

\item{Rmx}{maximum resistance value, if all gene loci under consideration are homozygous resistant. \eqn{\code{numeric}, \ge 1}.}

\item{rate}{percentage (\%) of the registered herbicide dose. \code{positive numeric}, can exceed 100 \%.}

\item{duration}{maximum number of simulation loops in the simulation. \code{positive integer}.}

\item{repetitions}{number of repetitions of the simulation. \code{positive integer}}

\item{crop_list}{crop rotation. \code{character vector}, elements must fit to the names in the \code{data.frame} weed.}

\item{max_vec_length}{used internally, a technical term, defining the maximum length of vectors to be used.}

\item{undersowing}{Numerical vector with two values between 0 and 1. See details.}

\item{sens_seeds}{sensitive seeds added every year.}

\item{dc}{crop sowing density, seeds per unit area. \code{numeric}.}

\item{kc}{dimensionless crop competition parameter. \code{numeric}.}

\item{kw}{dimensionless weed competition parameter. \code{numeric}.}

\item{SSmax}{maximum of weed seed production per unit area. \code{positive integer}.}

\item{pen_co}{penalty values for different weed cohorts. \code{numeric vector}, each element in [0,1].}
}
\description{
PROSPER entails full parametrized models, which are described here. 
Population dynamic models adress different purposes the models differ. The functions \code{prosper.*} present different adaptations.

\code{prosper.ECHCG} provides the setting for a simulation of the population dynamic of \emph{Echinochloa crus-galli}.

\code{prosper.GALAP} provides the setting for a simulation of the population dynamic of \emph{Galium aparine}. No selection process is used.

prosper.LOLRI performs a simulation of PROSPER using the setting presented by Renton at al. (2011). To manipulate the parameters see Details.
}
\details{
\code{prosper.ECHCG()} simulates originally the population dynamic of \emph{Echinochloa crus-galli} using the data \code{param.ECHCG}. Different cohorts of weed seedlings are the focus of this model. The focus of this model is the effect of weeds that escape the selection pressure of herbicide treatment. These weeds keep the unselected genetic. Can they buffer the selection process? \emph{E. crus-galli} is able to germinate over a long period after maize planting with decreasing reproductive success (Bagavathiannan, 2013). In the model all germinating individuals are represented by two cohorts; an early, major cohort with with high seed production, and a small late emerging with lower reproduction. Only the first cohort is controlled by a herbicide, which is a typical situation in Germany (Rossberg, 2016). The second cohort escapes the herbicide treatment unaffected. However, the second cohort can be suppressed, for example by an undersown crop. Three scenarios with different degrees of suppression, 0\%, 30\% and 100\%, were simulated (Redwitz, 2016).
The parameter \code{undersowing} describes the probability of surviving a second, not selective pressure on weed seedlings, which germinate after the selective herbicide was applied.
\code{prosper.ECHCG} provides the setting for a simulation of the population dynamic of \emph{Echinochloa crus-galli}.

\code{prosper.GALAP()} simulates originally the population dynamic of \emph{Galium aparine} using the data \code{param.GALAP}. Whether sowing of susceptible weed seeds can restore an 'acceptable' resistance level of a population in the early stages of resistance development, is an extraordinary research question. The patchy occurrence of \emph{Galium aparine} and its large seeds result in highly variable population dynamic parameters. Modeling has to take into account this variability. We used a simple population dynamics model structure (Redwitz et al., 2015). A seedbank in spring provides seeds out of which one cohort is germinating. The weeds are selected by herbicides, produce seeds, which are affected by seed predation and return to the seedbank in autumn. Data of a long term field experiment were used for parametrization (Daedlow, 2015).

\code{prosper.LOLRI()} performs a simulation of \emph{Lolium rigidum} similar to PERTH (Renton et al. 2011) when it is used with \code{param.LOLRI}.
}
\section{Functions}{
\itemize{
\item \code{prosper.ECHCG}: Population dynamic model of \emph{Echinochloa crus-galli}

\item \code{prosper.GALAP}: Population dynamic model of \emph{Galium aparine}

\item \code{prosper.LOLRI}: Population dynamic model of \emph{Lolium multiflorum}
}}

\examples{
mod_echcg <- prosper.ECHCG(param.weed = param.ECHCG, area=100, af=c(0.001),
                             undersowing=0.2,dom=0.5,duration=7,repetitions=1)


\dontrun{
#The model call for Redwitz et al. (2015)
undersowing_prob <- c(1, 0.3, 0) #no undersowing, strong competition, complete dominance
years <- 20
reps  <- 4
####------------------------
simu_collect <- list()
for(simu in 1:3){
simu_collect[[simu]] <- prosper.ECHCG(area          = 100,
                                      param.weed    = param.ECHCG,
                                      thresh        = 20,
                                      duration      = years,
                                      af            = 0.001,     
                                      dom           = 1,      
                                      undersowing   = undersowing_prob[simu],  
                                      repetitions   = reps
                                             )
}
}
\dontrun{   
data(param.GALAP)
mod_galap <- prosper.GALAP(param.weed=param.GALAP, repetitions=2, duration=10) }
\dontrun{
mod_lolri    <- prosper.LOLRI(param.weed=param.LOLRI, area=100, 
                                       duration=15, repetitions=3)}

}
\references{
Redwitz, C von, Pannwitt H, Gerowitt B (2016): About the interplay of sensitive and resistant biotypes in weed populations - simulation exercises for Echinochloa crus-galli in maize crops. Proceedings - 28th German Conference on Weed Biology and Weed Control, Julius-Kuehn-Archiv, 93-99, 452.

Redwitz, C von, Daedlow D, Gerowitt B (2015): Simulation exercises on long-term management of widespread herbicide resistance in a field weed population. Proceedings 17th Symposium of the European Weed Research Society, Montpellier, France, 108.

Renton, M., Diggle, A., Manalil, S. & Powles, S. (2011): Does cutting herbicide rates threaten the sustainability of weed management in cropping systems? Journal of Theoretical Biology, Elsevier BV, 283, 14-27.
}
\seealso{
\code{\link{weed-parameters}}
}
\author{
Christoph von Redwitz, \email{christoph.redwitz@uni-rostock.de}
}
