% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evaluate.e.ofv.fim.R
\name{evaluate.e.ofv.fim}
\alias{evaluate.e.ofv.fim}
\title{Evaluate the expectation of the Fisher Information Matrix (FIM) and the expectation of the OFV(FIM).}
\usage{
evaluate.e.ofv.fim(poped.db, fim.calc.type = NULL,
  bpop = poped.db$parameters$bpop, d = poped.db$parameters$d,
  covd = poped.db$parameters$covd, docc = poped.db$parameters$docc,
  sigma = poped.db$parameters$sigma, model_switch = NULL, ni = NULL,
  xt = NULL, x = NULL, a = NULL, groupsize = poped.db$design$groupsize,
  deriv.type = NULL, bLHS = poped.db$settings$bLHS,
  ofv_calc_type = poped.db$settings$ofv_calc_type,
  ED_samp_size = poped.db$settings$ED_samp_size,
  use_laplace = poped.db$settings$iEDCalculationType, laplace.fim = FALSE,
  ...)
}
\arguments{
\item{poped.db}{A PopED database.}

\item{fim.calc.type}{The method used for calculating the FIM. Potential values:
\itemize{
\item 0 = Full FIM.  No assumption that fixed and random effects are uncorrelated.  See \code{\link{mftot0}}.
\item 1 = Reduced FIM. Assume that there is no correlation in the FIM between the fixed and random effects, and set these elements in 
the FIM to zero. See \code{\link{mftot1}}.
\item 2 = weighted models (placeholder).
\item 3 = Not currently used.
\item 4 = Reduced FIM and computing all derivatives with respect to the standard deviation of the residual unexplained variation (sqrt(SIGMA) in NONMEM). 
This matches what is done in PFIM, and assumes that the standard deviation of the residual unexplained variation is the estimated parameter
(NOTE: NONMEM estimates the variance of the resudual unexplained variation by default). See \code{\link{mftot4}}.
\item 5 = Full FIM parameterized with A,B,C matrices & derivative of variance. See \code{\link{mftot5}}.
\item 6 = Calculate one model switch at a time, good for large matrices. See \code{\link{mftot6}}.
\item 7 = Reduced FIM parameterized with A,B,C matrices & derivative of variance See \code{\link{mftot7}}.
}}

\item{bpop}{Matrix defining the fixed effects, per row (row number = parameter_number) we should have:
\itemize{
\item column 1 the type of the distribution for E-family designs (0 = Fixed, 1 = Normal, 2 = Uniform,
 3 = User Defined Distribution, 4 = lognormal and 5 = truncated normal)
\item column 2  defines the mean.
\item column 3 defines the variance of the distribution (or length of uniform distribution).
}
Can also just supply the parameter values as a vector \code{c()} if no uncertainty around the 
parameter value is to be used.}

\item{d}{Matrix defining the diagnonals of the IIV (same logic as for the fixed efects 
matrix bpop to define uncertainty). One can also just supply the parameter values as a \code{c()}.}

\item{covd}{Column major vector defining the covariances of the IIV variances. 
That is, from your full IIV matrix  \code{covd <-  IIV[lower.tri(IIV)]}.}

\item{docc}{Matrix defining the IOV, the IOV variances and the IOV distribution as for d and bpop.}

\item{sigma}{Matrix defining the variances can covariances of the residual variability terms of the model.
can also just supply the diagnonal parameter values (variances) as a \code{c()}.}

\item{model_switch}{A matrix that is the same size as xt, specifying which model each sample belongs to.}

\item{ni}{A vector of the number of samples in each group.}

\item{xt}{A matrix of sample times.  Each row is a vector of sample times for a group.}

\item{x}{A matrix for the discrete design variables.  Each row is a group.}

\item{a}{A matrix of covariates.  Each row is a group.}

\item{groupsize}{A vector of the numer of individuals in each group.}

\item{deriv.type}{A number indicating the type of derivative to use:
\itemize{
\item 0=Complex difference 
\item 1=Central difference 
\item 20=Analytic derivative (placeholder) 
\item 30=Automatic differentiation (placeholder)
}}

\item{bLHS}{How to sample from distributions in E-family calculations. 0=Random Sampling, 1=LatinHyperCube --}

\item{ofv_calc_type}{OFV calculation type for FIM 
\itemize{ 
\item 1 = "D-optimality". Determinant of the FIM: det(FIM)
\item 2 = "A-optimality".  Inverse of the sum of the expected parameter variances: 
1/trace_matrix(inv(FIM)) 
\item 4 = "lnD-optimality".  Natural logarithm of the determinant of the FIM: log(det(FIM)) 
\item 6 = "Ds-optimality". Ratio of the Determinant of the FIM and the Determinant of the uninteresting
rows and columns of the FIM: det(FIM)/det(FIM_u)
\item 7 = Inverse of the sum of the expected parameter RSE: 1/sum(get_rse(FIM,poped.db,use_percent=FALSE))
}}

\item{ED_samp_size}{Sample size for E-family sampling}

\item{use_laplace}{Should the Laplace method be used in calculating the expectation of the OFV?}

\item{laplace.fim}{Should an E(FIM) be calculated when computing the Laplace approximated E(OFV).  Typically
the FIM does not need to be computed and, if desired,  this calculation
is done usng the standard MC integration technique, so can be slow.}

\item{...}{Other arguments passed to the function.}
}
\value{
A list containing the E(FIM) and E(OFV(FIM)) and the a poped.db updated according  to the function arguments.
}
\description{
Compute the expectation of the FIM and OFV(FIM) given the model, parameters, distributions of parameter uncertainty, design and methods defined in the 
PopED database. Some of the arguments coming from the PopED database can be overwritten;  
by default these arguments are \code{NULL} in the 
function, if they are supplied then they are used instead of the arguments from the PopED database.
}
\examples{
library(PopED)

############# START #################
## Create PopED database
## (warfarin model for optimization
##  with parameter uncertainty)
#####################################

## Warfarin example from software comparison in:
## Nyberg et al., "Methods and software tools for design evaluation 
##   for population pharmacokinetics-pharmacodynamics studies", 
##   Br. J. Clin. Pharm., 2014. 

## Optimization using an additive + proportional reidual error
## to avoid sample times at very low concentrations (time 0 or very late samoples).

## find the parameters that are needed to define from the structural model
ff.PK.1.comp.oral.sd.CL

## -- parameter definition function 
## -- names match parameters in function ff
sfg <- function(x,a,bpop,b,bocc){
  parameters=c(CL=bpop[1]*exp(b[1]),
               V=bpop[2]*exp(b[2]),
               KA=bpop[3]*exp(b[3]),
               Favail=bpop[4],
               DOSE=a[1])
  return(parameters) 
}

# Adding 10\% log-normal Uncertainty to fixed effects (not Favail)
bpop_vals <- c(CL=0.15, V=8, KA=1.0, Favail=1)
bpop_vals_ed_ln <- cbind(ones(length(bpop_vals),1)*4, # log-normal distribution
                         bpop_vals,
                         ones(length(bpop_vals),1)*(bpop_vals*0.1)^2) # 10\% of bpop value
bpop_vals_ed_ln["Favail",]  <- c(0,1,0)
bpop_vals_ed_ln

## -- Define initial design  and design space
poped.db <- create.poped.database(ff_file="ff.PK.1.comp.oral.sd.CL",
                                  fg_file="sfg",
                                  fError_file="feps.add.prop",
                                  bpop=bpop_vals_ed_ln, 
                                  notfixed_bpop=c(1,1,1,0),
                                  d=c(CL=0.07, V=0.02, KA=0.6), 
                                  sigma=c(0.01,0.25),
                                  groupsize=32,
                                  xt=c( 0.5,1,2,6,24,36,72,120),
                                  minxt=0,
                                  maxxt=120,
                                  a=70,
                                  mina=0,
                                  maxa=100)

############# END ###################
## Create PopED database
## (warfarin model for optimization
##  with parameter uncertainty)
#####################################


## ED evaluate (with very few samples)
output <- evaluate.e.ofv.fim(poped.db,ED_samp_size=10)
output$E_ofv

## API evaluate (with very few samples)
output <- evaluate.e.ofv.fim(poped.db,ED_samp_size=10,ofv_calc_type=4)
output$E_ofv

## ED evaluate using Laplace approximation 
tic()
output <- evaluate.e.ofv.fim(poped.db,use_laplace=TRUE)
toc()
output$E_ofv

\dontrun{

  ## ED expected value with more precision. 
  ## Compare time and value to Laplace approximation.
  ## Run a couple of times to see stochasticity of calculation.
  tic()
  e_ofv_mc <- evaluate.e.ofv.fim(poped.db,ED_samp_size=500)
  toc()
  e_ofv_mc$E_ofv
  
  # If you want to get an E(FIM) from the laplace approximation you have to ask for it
  # and it will take more time.
  output <- evaluate.e.ofv.fim(poped.db,use_laplace=TRUE,laplace.fim=TRUE)
  output$E_fim
  
 

}
}
\seealso{
Other E-family: \code{\link{calc_ofv_and_fim}},
  \code{\link{ed_laplace_ofv}}, \code{\link{ed_mftot}}

Other evaluate_FIM: \code{\link{calc_ofv_and_fim}},
  \code{\link{evaluate.fim}}, \code{\link{ofv_fim}}

Other FIM: \code{\link{LinMatrixH}},
  \code{\link{LinMatrixLH}}, \code{\link{LinMatrixL_occ}},
  \code{\link{calc_ofv_and_fim}},
  \code{\link{ed_laplace_ofv}}, \code{\link{ed_mftot}},
  \code{\link{efficiency}}, \code{\link{evaluate.fim}},
  \code{\link{gradf_eps}}, \code{\link{mf3}},
  \code{\link{mf5}}, \code{\link{mf6}}, \code{\link{mf7}},
  \code{\link{mf8}}, \code{\link{mftot0}},
  \code{\link{mftot1}}, \code{\link{mftot2}},
  \code{\link{mftot3}}, \code{\link{mftot4}},
  \code{\link{mftot5}}, \code{\link{mftot6}},
  \code{\link{mftot7}}, \code{\link{mftot}},
  \code{\link{mf}}, \code{\link{ofv_criterion}},
  \code{\link{ofv_fim}}
}

