% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions.R
\name{corrected_meta}
\alias{corrected_meta}
\title{Estimate publication bias-corrected meta-analysis}
\usage{
corrected_meta(yi, vi, eta, clustervar = 1:length(yi), model,
  selection.tails = 1, CI.level = 0.95, small = TRUE)
}
\arguments{
\item{yi}{A vector of point estimates to be meta-analyzed}

\item{vi}{A vector of estimated variances for the point estimates}

\item{eta}{The number of times more likely an affirmative study is to be published than a nonaffirmative study. See Details.}

\item{clustervar}{A character, factor, or numeric vector with the same length as yi. Unique values should indicate
unique clusters of point estimates. By default, assumes all point estimates are independent by default.}

\item{model}{"fixed" for fixed-effects or "robust" for robust random-effects}

\item{selection.tails}{1 (for one-tailed selection, recommended for its conservatism) or 2 (for two-tailed selection)}

\item{CI.level}{Confidence interval level (as proportion) for the corrected point estimate}

\item{small}{Should inference allow for a small meta-analysis? We recommend always using TRUE.}
}
\description{
For a known selection ratio, eta, estimates a publication bias-corrected pooled point
estimate and confidence interval. Model options include fixed-effects, robust independent, and robust
clustered specifications.
}
\details{
The selection ratio, eta, represents the number of times more likely affirmative studies (i.e., those with a "statistically significant" and positive estimate)
are to be published than nonaffirmative studies (i.e., those with a "nonsignificant" or negative estimate).
}
\examples{
# calculate effect sizes from example dataset in metafor
require(metafor)
dat = metafor::escalc(measure="RR", ai=tpos, bi=tneg, ci=cpos, di=cneg, data=dat.bcg)

# first fit fixed-effects model without any bias correction
rma( yi, vi, data = dat, method = "FE" )

# notice this is the same as passing eta = 1 to corrected_meta, which makes sense
corrected_meta( yi = dat$yi,
                vi = dat$vi,
                eta = 1,
                model = "fixed",
                selection.tails = 1,
                CI.level = 0.95,
                small = TRUE )

# assume a known selection ratio of 5
# i.e., affirmative results are 5x more likely to be published
#  than nonaffirmative
corrected_meta( yi = dat$yi,
                vi = dat$vi,
                eta = 5,
                model = "fixed",
                selection.tails = 1,
                CI.level = 0.95,
                small = TRUE )

# same selection ratio, but now account for heterogeneity
# and clustering via robust specification
corrected_meta( yi = dat$yi,
                vi = dat$vi,
                eta = 5,
                clustervar = dat$author,
                model = "robust",
                selection.tails = 1,
                CI.level = 0.95,
                small = TRUE )

##### Make sensitivity plot as in Mathur & VanderWeele (2019) #####
# range of parameters to try (more dense at the very small ones)
eta.list = as.list( c( 200, 150, 100, 50, 40, 30, 20, rev( seq(1,15,1) ) ) )
res.list = lapply( eta.list, function(x) {
                    cat("\\n Working on eta = ", x)
                    return( corrected_meta( yi = dat$yi,
                                            vi = dat$vi,
                                            eta = x,
                                            model = "robust",
                                            clustervar = dat$author ) )
                                        }
                      )

# put results for each eta in a dataframe
res.df = as.data.frame( do.call( "rbind", res.list ) )

require(ggplot2)
ggplot( data = res.df, aes( x = eta, y = est ) ) +

  geom_ribbon( data = res.df, aes( x = eta, ymin = lo, ymax = hi ), fill = "gray" ) +

  geom_line( lwd = 1.2 ) +
  xlab( bquote( eta ) ) +
  ylab( bquote( hat(mu)[eta] ) ) +

  theme_classic()
}
\references{
1. Mathur MB & VanderWeele TJ (2019). Sensitivity analysis for publication bias in meta-analyses. Preprint available at XXX.
}
