% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VecDecomPlot.R
\name{VecDecomPlot}
\alias{VecDecomPlot}
\title{Plotting function for vector decomposition and remainder fields}
\usage{
VecDecomPlot(x.field, y.field, dens, x.bound, y.bound, xlim = "NULL",
  ylim = "NULL", arrow.type = "equal", tail.length = 0.25,
  head.length = 0.25, ...)
}
\arguments{
\item{x.field}{a two-dimensional array containing the x-values for the vector field, generated from \code{\link{VecDecomAll}}, \code{\link{VecDecomVec}}, \code{\link{VecDecomGrad}}, or \code{\link{VecDecomRem}}.}

\item{y.field}{a two-dimensional array containing the y-values for the vector field, generated from \code{\link{VecDecomAll}}, \code{\link{VecDecomVec}}, \code{\link{VecDecomGrad}}, or \code{\link{VecDecomRem}}.}

\item{dens}{two-element vector respectively specifying the number of respective arrows in the x and y directions.}

\item{x.bound}{the x boundaries denoted at c(minimum, maximum) for the quasi-potential simulation.}

\item{y.bound}{the y boundaries denoted at c(minimum, maximum) for the quasi-potential simulation.}

\item{xlim}{numeric vectors of length 2, giving the x coordinate range.}

\item{ylim}{numeric vectors of length 2, giving the y coordinate range.}

\item{arrow.type}{sets the type of line segments plotted. If set to "proportional" the length of the line segments reflects the magnitude of the derivative. If set to "equal" the line segments take equal lengths, simply reflecting the gradient of the derivative(s). Defaults to "equal".}

\item{tail.length}{multiplies the current length of the tail (both proportional and equal arrow.types) by the specified factor.  The argument defaults to 1, which is length of the longest vector within the domain boundaries (i.e., the entire field).}

\item{head.length}{length of the edges of the arrow head (in inches).}

\item{...}{passes arguments to both \code{\link{plot}}.}
}
\description{
This function plots various vector fields
}
\details{
If \code{arrow.type = "proportional"}, a common warning, passed from \code{\link{arrows}}, will appear: "The direction of a zero-length arrow is indeterminate, and hence so is the direction of the arrowheads. To allow for rounding error, arrowheads are omitted (with a warning) on any arrow of length less than 1/1000 inch."  Either increase \code{tail.length} or increase the plot window to avoid this warning.
}
\examples{
# First, system of equations
	equationx <- "1.54*x*(1.0-(x/10.14)) - (y*x*x)/(1.0+x*x)"
	equationy <- "((0.476*x*x*y)/(1+x*x)) - 0.112590*y*y"

# Second, shared parameters for each quasi-potential run
	xbounds <- c(-0.5, 10.0)
	ybounds <- c(-0.5, 10.0)
	xstepnumber <- 100
	ystepnumber <- 100

# Third, first local quasi-potential run
	xinit1 <- 1.40491
	yinit1 <- 2.80808
	storage.eq1 <- QPotential(x.rhs = equationx, x.start = xinit1, 
	x.bound = xbounds, x.num.steps = xstepnumber, y.rhs = equationy, 
	y.start = yinit1, y.bound = ybounds, y.num.steps = ystepnumber)

# Fourth, second local quasi-potential run
	xinit2 <- 4.9040
	yinit2 <- 4.06187
	storage.eq2 <- QPotential(x.rhs = equationx, x.start = xinit2, 
	x.bound = xbounds, x.num.steps = xstepnumber, y.rhs = equationy, 
	y.start = yinit2, y.bound = ybounds, y.num.steps = ystepnumber)

# Fifth, determine global quasi-potential 
	unst.x <- c(0, 4.2008)
	unst.y <- c(0, 4.0039)
	ex1.global <- QPGlobal(local.surfaces = list(storage.eq1, storage.eq2), 
	unstable.eq.x = unst.x, unstable.eq.y = unst.y, x.bound = xbounds, 
	y.bound = ybounds)

# Sixth, decompose the global quasi-potential into the 
# deterministic skeleton, gradient, and remainder vector fields
	VDAll <- VecDecomAll(surface = ex1.global, x.rhs = equationx, y.rhs = equationy, 
	x.bound = xbounds, y.bound = ybounds)

# Seventh, plot all three vector fields
	# The deterministic skeleton vector field
	VecDecomPlot(x.field = VDAll[,,1], y.field = VDAll[,,2], dens = c(25,25), 
	x.bound = xbounds, y.bound = ybounds, tail.length = 0.25, head.length = 0.05)
	# The gradient vector field
	VecDecomPlot(x.field = VDAll[,,3], y.field = VDAll[,,4], dens = c(25,25), 
	x.bound = xbounds, y.bound = ybounds, tail.length = 0.15, head.length = 0.05)
	# The remainder vector field
	VecDecomPlot(x.field = VDAll[,,5], y.field = VDAll[,,6], dens = c(25,25), 
	x.bound = xbounds, y.bound = ybounds, tail.length = 0.15, head.length = 0.05)
}
\keyword{detrministic}
\keyword{field}
\keyword{gradient}
\keyword{plot}
\keyword{plot,}
\keyword{remainder}
\keyword{skeleton}
\keyword{vector}

