% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IM.search2.R
\name{IM.search2}
\alias{IM.search2}
\title{QTL search by IM with Selective Genotyping}
\usage{
IM.search2(
  marker,
  geno,
  y,
  yu = NULL,
  sele.g = "n",
  tL = NULL,
  tR = NULL,
  method = "EM",
  type = "RI",
  D.matrix = NULL,
  ng = 2,
  cM = TRUE,
  speed = 1,
  crit = 10^-5,
  d.eff = FALSE,
  LRT.thre = TRUE,
  simu = 1000,
  alpha = 0.05,
  detect = TRUE,
  QTLdist = 15,
  plot.all = TRUE,
  plot.chr = TRUE,
  console = TRUE
)
}
\arguments{
\item{marker}{matrix. A k*2 matrix contains the marker information,
where the row dimension 'k' represents the number of markers in the
chromosomes. The first column labels the chromosomes where the markers
are located, and the second column labels the positions of markers (in
morgan (M) or centimorgan (cM)). It's important to note that chromosomes
and positions must be sorted in order.}

\item{geno}{matrix. A n*k matrix contains the genotypes of k markers
for n individuals. The marker genotypes of P1 homozygote (MM),
heterozygote (Mm), and P2 homozygote (mm) are coded as 2, 1, and 0,
respectively, with NA indicating missing values.}

\item{y}{vector. A vector that contains the phenotype values of
individuals with genotypes.}

\item{yu}{vector. A vector that contains the phenotype values of
individuals without genotypes.}

\item{sele.g}{character. Determines the type of data being analyzed:
If sele.g="n", it considers the data as complete genotyping data. If
sele.g="f", it treats the data as selective genotyping data and utilizes
the proposed corrected frequency model (Lee 2014) for analysis; If
sele.g="t", it considers the data as selective genotyping data and uses
the truncated model (Lee 2014) for analysis; If sele.g="p", it treats
the data as selective genotyping data and uses the population
frequency-based model (Lee 2014) for analysis. Note that the 'yu'
argument must be provided when sele.g="f" or "p".}

\item{tL}{numeric. The lower truncation point of phenotype value when
sele.g="t". When sele.g="t" and tL=NULL, the 'yu' argument must be
provided. In this case, the function will consider the minimum of 'yu'
as the lower truncation point.}

\item{tR}{numeric. The upper truncation point of phenotype value when
sele.g="t". When sele.g="t" and tR=NULL, the 'yu' argument must be
provided. In this case, the function will consider the maximum of 'yu'
as the upper truncation point.}

\item{method}{character. When method="EM", it indicates that the interval
mapping method by Lander and Botstein (1989) is used in the analysis.
Conversely, when method="REG", it indicates that the approximate regression
interval mapping method by Haley and Knott (1992) is used in the analysis.}

\item{type}{character. The population type of the dataset. Includes
backcross (type="BC"), advanced intercross population (type="AI"), and
recombinant inbred population (type="RI"). The default value is "RI".}

\item{D.matrix}{matrix. The design matrix of the IM model. If D.matrix=NULL,
the design matrix will be constructed using Cockerham’s model: In BC
population, it is a 2*1 matrix with values 0.5 and -0.5 for the additive
effect; In RI or AI population, it is a 3*2 matrix. The first column
consists of 1, 0, and -1 for the additive effect, and the second column
consists of 0.5, -0.5, and 0.5 for the dominant effect.}

\item{ng}{integer. The generation number of the population type. For
instance, in a BC1 population where type="BC", ng=1; in an AI F3
population where type="AI", ng=3.}

\item{cM}{logical. Specify the unit of marker position. If cM=TRUE, it
denotes centimorgan; if cM=FALSE, it denotes morgan.}

\item{speed}{numeric. The walking speed of the QTL search (in cM).}

\item{crit}{numeric. The convergence criterion of EM algorithm.
The E and M steps will iterate until a convergence criterion is met.
It must be a value between 0 and 1.}

\item{d.eff}{logical. Specifies whether the dominant effect will be
considered in the parameter estimation for AI or RI population.}

\item{LRT.thre}{logical or numeric. If set to TRUE, the LRT threshold
will be computed based on the Gaussian stochastic process (Kao and Ho 2012).
Alternatively, users can input a numerical value as the LRT threshold to
evaluate the significance of QTL detection.}

\item{simu}{integer. Determines the number of simulation samples that
will be used to compute the LRT (Likelihood Ratio Test) threshold using
the Gaussian process. It must be a value between 50 and 10^8.}

\item{alpha}{numeric. The type I error rate for the LRT threshold.}

\item{detect}{logical. Determines whether the significant QTL, whose LRT
statistic is larger than the LRT threshold, will be displayed in the
output dataset or not.}

\item{QTLdist}{numeric. The minimum distance (in cM) among different
linked significant QTL.}

\item{plot.all}{logical. When set to TRUE, it directs the function to
output the profile of LRT statistics for the genome in one figure.}

\item{plot.chr}{logical. When set to TRUE, it instructs the function to
output the profile of LRT statistics for the chromosomes.}

\item{console}{logical. Determines whether the process of the algorithm
will be displayed in the R console or not.}
}
\value{
\item{effect}{The estimated effects and LRT statistics of all positions.}
\item{LRT.threshold}{The LRT threshold value computed for the data using the
Gaussian stochastic process (Kuo 2011; Kao and Ho 2012).}
\item{detect.QTL}{The positions, effects and LRT statistics of the detected
QTL significant using the obtained LRT threshold value.}
\item{model}{The model of selective genotyping data in this analyze.}

Graphical outputs including LOD value and effect of each position.
}
\description{
Expectation-maximization algorithm for QTL interval mapping to search
for possible positions of QTL in all chromosomes. It can handle genotype
data which is selective genotyping.
}
\examples{
# load the example data
load(system.file("extdata", "exampledata.RDATA", package = "QTLEMM"))

# make the seletive genotyping data
ys <- y[y > quantile(y)[4] | y < quantile(y)[2]]
yu <- y[y >= quantile(y)[2] & y <= quantile(y)[4]]
geno.s <- geno[y > quantile(y)[4] | y < quantile(y)[2],]

# run and result
result <- IM.search2(marker, geno.s, ys, yu, sele.g = "f", type = "RI", ng = 2,
speed = 7, crit = 10^-3, LRT.thre = 10)
result$detect.QTL
}
\references{
KAO, C.-H. and Z.-B. ZENG 1997 General formulas for obtaining the maximum
likelihood estimates and the asymptotic variance-covariance matrix in QTL
mapping when using the EM algorithm. Biometrics 53, 653-665.

KAO, C.-H., Z.-B. ZENG and R. D. TEASDALE 1999 Multiple interval mapping
for Quantitative Trait Loci. Genetics 152: 1203-1216.

H.-I LEE, H.-A. HO and C.-H. KAO 2014 A new simple method for improving
QTL mapping under selective genotyping. Genetics 198: 1685-1698.

KAO, C.-H. and H.-A. Ho 2012 A score-statistic approach for determining
threshold values in QTL mapping. Frontiers in Bioscience. E4, 2670-2682.
}
\seealso{
\code{\link[QTLEMM]{EM.MIM2}}
\code{\link[QTLEMM]{IM.search}}
\code{\link[QTLEMM]{LRTthre}}
}
