#' @title
#' Check for possible API user privilege issues

#' @description
#' Check for potential user access privilege issues and provide an appropriate
#' warning message. This can occur when metadata forms/field names do not appear
#' in a database export.
#'
#' @return
#' A helpful error message alerting the user to check their API privileges.
#'
#' @importFrom rlang .data
#' @importFrom dplyr filter select
#' @importFrom tidyr pivot_wider
#'
#' @param db_data The REDCap database output generated by
#' \code{REDCapR::redcap_read_oneshot()$data}
#' @param db_metadata The REDCap metadata output generated by \code{REDCapR::redcap_metadata_read()$data}
#'
#' @export
#'
#' @keywords internal

check_user_rights <- function(
    db_data,
    db_metadata
) {
  # Similar to link_arms, use pivot wider to create list elements with top
  # level form_name and vector of associated field_names
  missing_db_metadata <- db_metadata %>%
    filter(!.data$field_name_updated %in% names(db_data)) %>%
    select(.data$field_name_updated, .data$form_name) %>%
    pivot_wider(names_from = .data$form_name,
                values_from = .data$field_name_updated,
                values_fn = list)

  # Supply user with warning message(s) displaying missing form name and
  # associated fields
  for (i in seq_len(ncol(missing_db_metadata))) {
    warning(paste0(
      "Form name {", names(missing_db_metadata)[i], "} detected in metadata, but not found in the database export. This can happen when the user privileges are not set to allow exporting that form via the API. The following variables are affected: ",
      paste(unlist(missing_db_metadata[i][[1]]), collapse = ", ")),
      call. = FALSE)
  }
}

#' @title
#' Check for instruments that have both repeating and non-repeating structure
#'
#' @description
#' Check for potential instruments that are given both repeating and
#' nonrepeating structure. This should never be used with \code{REDCapTidieR}.
#'
#' @return
#' A helpful error message alerting the user to existence of a form containing
#' both repeating and non-repeating structure.
#'
#' @param db_data The REDCap database output generated by
#' \code{REDCapR::redcap_read_oneshot()$data}
#'
#' @importFrom rlang .data
#' @importFrom dplyr slice pull %>% select
#' @importFrom tidyr drop_na
#' @importFrom tidyselect everything
#'
#' @export
#'
#' @keywords internal


check_repeat_and_nonrepeat <- function(db_data) {

  # Loop through dataframe columns then rows to determine if a variable has data
  # and appears alongside redcap_repeat_instance when it is both NA and not NA

  #start at 5 to skip record id, redcap event, instrument, and instance vars
  for (i in 5:length(names(db_data))) {

    nonrepeating <- NA
    repeating <- NA

    for (j in seq_len(nrow(db_data))){

      # Check if variable has data and redcap_repeat_instance is NA
      if (!is.na(db_data[j, i]) && is.na(db_data$redcap_repeat_instance[j])) {
        nonrepeating <- TRUE
      }

      # Check if variable and redcap_repeat_instance both have NA
      if (!is.na(db_data[j, i]) && !is.na(db_data$redcap_repeat_instance[j])) {
        repeating <- TRUE
      }
    }

    # If variable is determined to be both repeating and nonrepeating, error out
    if (isTRUE(nonrepeating) && isTRUE(repeating)) {
      stop(
        paste0("Instrument detected belonging to an instrument that is both repeating and nonrepeating: ",
               db_data %>%
                 select(.data$redcap_repeat_instrument, names(db_data)[i]) %>%
                 drop_na(everything()) %>%
                 slice(1) %>%
                 pull(.data$redcap_repeat_instrument)
        )
      )
    }
  }

}

#' @title
#' Check that a supplied REDCap database is populated

#' @description
#' Check for potential outputs where metadata is present, but \code{nrow} and
#' \code{ncol} equal `0`. This causes \code{multi_choice_to_labels} to fail, but
#' a helpful error message should be provided.
#'
#' @return
#' A helpful error message alerting the user to check their API privileges.
#'
#' @param db_data The REDCap database output generated by
#' \code{REDCapR::redcap_read_oneshot()$data}
#'
#' @export
#'
#' @keywords internal

check_redcap_populated <- function(
    db_data
) {
  if (ncol(db_data) == 0) {
    stop(
      paste0("The REDCap API did not return any data. This can happen when there are no data entered or when the access isn't configured to allow data download through the API."
      )
    )
  }
}
