% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OptSpace.R
\name{OptSpace}
\alias{OptSpace}
\title{OptSpace : an algorithm for matrix reconstruction from a partially revealed set}
\usage{
OptSpace(A, ropt = NA, niter = 50, tol = 1e-06,
  showprogress = TRUE)
}
\arguments{
\item{A}{an \eqn{(n\times m)} matrix whose missing entries should be flaged as NA.}

\item{ropt}{\code{NA} to guess the rank, or a positive integer as a pre-defined rank.}

\item{niter}{maximum number of iterations allowed.}

\item{tol}{stopping criterion for reconstruction in Frobenius norm.}

\item{showprogress}{a logical value; \code{TRUE} to show progress, \code{FALSE} otherwise.}
}
\value{
a named list containing
\describe{
\item{X}{an \eqn{(n \times r)} matrix as left singular vectors.}
\item{S}{an \eqn{(r \times r)} matrix as singular values.}
\item{Y}{an \eqn{(m \times r)} matrix as right singular vectors.}
\item{dist}{a vector containing reconstruction errors at each successive iteration.}
}
}
\description{
Let's assume an ideal matrix \eqn{M} with \eqn{(m\times n)} entries with rank \eqn{r} and
we are given a partially observed matrix \eqn{M\_E} which contains many missing entries.
Matrix reconstruction - or completion - is the task of filling in such entries.
OptSpace is an efficient algorithm that reconstructs \eqn{M} from \eqn{|E|=O(rn)} observed elements
with relative root mean square error (RMSE)
\deqn{RMSE \le C(\alpha)\sqrt{nr/|E|}}
}
\section{Acknowledgements}{

This code is a modified translation of
\href{http://swoh.web.engr.illinois.edu/software/optspace/code.html}{MATLAB package}
maintained by Sewoong Oh (UIUC).
}

\examples{
## Parameter Settings
n = 1000;
m = 100;
r = 3;
tolerance = 1e-7
eps = 10*r*log10(n)

## Generate a matrix with given data
U = matrix(rnorm(n*r),nrow=n)
V = matrix(rnorm(m*r),nrow=m)
Sig = diag(r)
M0 = U\%*\%Sig\%*\%t(V)

## Set some entries to be NA with probability eps/sqrt(m*n)
E = 1 - ceiling(matrix(rnorm(n*m),nrow=n) - eps/sqrt(m*n))
M_E = M0
M_E[(E==0)] = NA

## Create a noisy version
noiselevel = 0.1
M_E_noise  = M_E + matrix(rnorm(n*m),nrow=n)*noiselevel

## Use OptSpace for reconstruction
res1 = OptSpace(M_E,tol=tolerance)
res2 = OptSpace(M_E_noise,tol=tolerance)

## Compute errors for both cases using Frobenius norm
err_clean = norm(res1$X\%*\%res1$S\%*\%t(res1$Y)-M0,'f')/sqrt(m*n)
err_noise = norm(res2$X\%*\%res2$S\%*\%t(res2$Y)-M0,'f')/sqrt(m*n)

## print out the results
m1 = sprintf('RMSE without noise         : \%e',err_clean)
m2 = sprintf('RMSE with noise of \%.2f    : \%e',noiselevel,err_noise)
print(m1)
print(m2)

}
\references{
\insertRef{keshavan_matrix_2010}{ROptSpace}
}
