//=================================================================================================
/*!
//  \file blaze/math/StaticVector.h
//  \brief Header file for the complete StaticVector implementation
//
//  Copyright (C) 2013 Klaus Iglberger - All Rights Reserved
//
//  This file is part of the Blaze library. You can redistribute it and/or modify it under
//  the terms of the New (Revised) BSD License. Redistribution and use in source and binary
//  forms, with or without modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright notice, this list of
//     conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright notice, this list
//     of conditions and the following disclaimer in the documentation and/or other materials
//     provided with the distribution.
//  3. Neither the names of the Blaze development group nor the names of its contributors
//     may be used to endorse or promote products derived from this software without specific
//     prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
//  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
//  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
//  SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
//  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
//  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
//  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//  ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
*/
//=================================================================================================

#ifndef _BLAZE_MATH_STATICVECTOR_H_
#define _BLAZE_MATH_STATICVECTOR_H_


//*************************************************************************************************
// Includes
//*************************************************************************************************

#include <blaze/math/dense/StaticVector.h>
#include <blaze/math/DenseVector.h>
#include <blaze/math/DynamicVector.h>
#include <blaze/math/StaticMatrix.h>
#include <blaze/system/Precision.h>
#include <blaze/util/Random.h>


namespace blaze {

//=================================================================================================
//
//  RAND SPECIALIZATION
//
//=================================================================================================

//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
/*!\brief Specialization of the Rand class template for StaticVector.
// \ingroup random
//
// This specialization of the Rand class creates random instances of StaticVector.
*/
template< typename Type  // Data type of the vector
        , size_t N       // Number of elements
        , bool TF >      // Transpose flag
class Rand< StaticVector<Type,N,TF> >
{
 public:
   //**Generate functions**************************************************************************
   /*!\name Generate functions */
   //@{
   inline const StaticVector<Type,N,TF> generate() const;

   template< typename Arg >
   inline const StaticVector<Type,N,TF> generate( const Arg& min, const Arg& max ) const;
   //@}
   //**********************************************************************************************

   //**Randomize functions*************************************************************************
   /*!\name Randomize functions */
   //@{
   inline void randomize( StaticVector<Type,N,TF>& vector ) const;

   template< typename Arg >
   inline void randomize( StaticVector<Type,N,TF>& vector, const Arg& min, const Arg& max ) const;
   //@}
   //**********************************************************************************************
};
/*! \endcond */
//*************************************************************************************************


//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
/*!\brief Generation of a random StaticVector.
//
// \return The generated random vector.
*/
template< typename Type  // Data type of the vector
        , size_t N       // Number of elements
        , bool TF >      // Transpose flag
inline const StaticVector<Type,N,TF> Rand< StaticVector<Type,N,TF> >::generate() const
{
   StaticVector<Type,N,TF> vector;
   randomize( vector );
   return vector;
}
/*! \endcond */
//*************************************************************************************************


//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
/*!\brief Generation of a random StaticVector.
//
// \param min The smallest possible value for a vector element.
// \param max The largest possible value for a vector element.
// \return The generated random vector.
*/
template< typename Type   // Data type of the vector
        , size_t N        // Number of elements
        , bool TF >       // Transpose flag
template< typename Arg >  // Min/max argument type
inline const StaticVector<Type,N,TF>
   Rand< StaticVector<Type,N,TF> >::generate( const Arg& min, const Arg& max ) const
{
   StaticVector<Type,N,TF> vector;
   randomize( vector, min, max );
   return vector;
}
/*! \endcond */
//*************************************************************************************************


//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
/*!\brief Randomization of a StaticVector.
//
// \param vector The vector to be randomized.
// \return void
*/
template< typename Type  // Data type of the vector
        , size_t N       // Number of elements
        , bool TF >      // Transpose flag
inline void Rand< StaticVector<Type,N,TF> >::randomize( StaticVector<Type,N,TF>& vector ) const
{
   using blaze::randomize;

   for( size_t i=0UL; i<N; ++i ) {
      randomize( vector[i] );
   }
}
/*! \endcond */
//*************************************************************************************************


//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
/*!\brief Randomization of a StaticVector.
//
// \param vector The vector to be randomized.
// \param min The smallest possible value for a vector element.
// \param max The largest possible value for a vector element.
// \return void
*/
template< typename Type   // Data type of the vector
        , size_t N        // Number of elements
        , bool TF >       // Transpose flag
template< typename Arg >  // Min/max argument type
inline void Rand< StaticVector<Type,N,TF> >::randomize( StaticVector<Type,N,TF>& vector,
                                                        const Arg& min, const Arg& max ) const
{
   using blaze::randomize;

   for( size_t i=0UL; i<N; ++i ) {
      randomize( vector[i], min, max );
   }
}
/*! \endcond */
//*************************************************************************************************




//=================================================================================================
//
//  TYPE DEFINITIONS
//
//=================================================================================================

//*************************************************************************************************
/*!\brief 2-dimensional single precision vector.
// \ingroup static_vector_2
*/
typedef StaticVector<float,2UL,false>  Vec2f;
//*************************************************************************************************


//*************************************************************************************************
/*!\brief 2-dimensional double precision vector.
// \ingroup static_vector_2
*/
typedef StaticVector<double,2UL,false>  Vec2d;
//*************************************************************************************************


//*************************************************************************************************
/*!\brief 2-dimensional vector with system-specific precision.
// \ingroup static_vector_2
*/
typedef StaticVector<real_t,2UL,false>  Vec2;
//*************************************************************************************************


//*************************************************************************************************
/*!\brief 3-dimensional single precision vector.
// \ingroup static_vector_3
*/
typedef StaticVector<float,3UL,false>  Vec3f;
//*************************************************************************************************


//*************************************************************************************************
/*!\brief 3-dimensional double precision vector.
// \ingroup static_vector_3
*/
typedef StaticVector<double,3UL,false>  Vec3d;
//*************************************************************************************************


//*************************************************************************************************
/*!\brief 3-dimensional vector with system-specific precision.
// \ingroup static_vector_3
*/
typedef StaticVector<real_t,3UL,false>  Vec3;
//*************************************************************************************************


//*************************************************************************************************
/*!\brief 6-dimensional single precision vector.
// \ingroup static_vector_6
*/
typedef StaticVector<float,6UL,false>  Vec6f;
//*************************************************************************************************


//*************************************************************************************************
/*!\brief 6-dimensional double precision vector.
// \ingroup static_vector_6
*/
typedef StaticVector<double,6UL,false>  Vec6d;
//*************************************************************************************************


//*************************************************************************************************
/*!\brief 6-dimensional vector with system-specific precision.
// \ingroup static_vector_6
*/
typedef StaticVector<real_t,6UL,false>  Vec6;
//*************************************************************************************************

} // namespace blaze

#endif
