#' Roxygenize a package, clean up and build/check the package
#'
#' After the source package is roxygenized, this function will build the
#' package. Optionally it also installs or checks the package, reformats the
#' code in the usage and examples sections. Note \code{\link{rab}} is an alias
#' of \code{\link{roxygen_and_build}}.
#' @param pkg the root directory of the source package
#' @param roxygen.dir the directory for the roxygenized package (by default it
#'   is the same as \code{pkg})
#' @param build whether to build the package
#' @param install whether to install the package
#' @param check whether to check the package
#' @param check.opts options to check the package (e.g. \code{"--no-examples"})
#' @param remove.check whether to remove the directory generated by \command{R
#'   CMD check}
#' @param reformat whether to reformat the usage and examples code; see
#'   \code{\link{reformat_code}}
#' @param ... other arguments passed to \code{\link[roxygen2]{roxygenize}}
#' @return \code{NULL}
#' @author Yihui Xie <\url{http://yihui.name}>
#' @rdname roxygen_and_build
#' @export
#' @examples \dontrun{
#' roxygen_and_build("Rd2roxygen", install = TRUE)
#' ## or simply
#' rab('Rd2roxygen', install = TRUE)
#' }
roxygen_and_build = function(pkg, roxygen.dir = pkg, build = TRUE, install = FALSE,
                             check = FALSE, check.opts = "--as-cran", remove.check = TRUE,
                             reformat = TRUE, ...) {
  roxygenize(pkg, roxygen.dir, ...)
  rd.list = list.files(file.path(roxygen.dir, "man"), ".*\\.Rd$", all.files = TRUE,
                       full.names = TRUE)
  if (reformat) {
    message('Reformatting usage and examples')
    for (f in rd.list) reformat_code(f)
  }
  if (build) system(sprintf("R CMD build %s ", roxygen.dir)) else return()
  pv = read.dcf(file.path(pkg, 'DESCRIPTION'), fields=c('Package', 'Version'))
  res = sprintf('%s_%s.tar.gz', pv[1, 1], pv[1, 2])
  if (install) system(sprintf("R CMD INSTALL %s ", res))
  if (check) {
    if ((system(sprintf("R CMD check %s %s", res, check.opts)) == 0) &&
      remove.check) unlink(sprintf('%s.Rcheck', pv[1, 1]), TRUE)
  }
  invisible(NULL)
}

#' @rdname roxygen_and_build
#' @export
rab = roxygen_and_build


#' Format the code in the usage and examples sections
#'
#' The function \code{\link[formatR]{tidy.source}} in the \pkg{formatR} package
#' is used to polish the Rd files generated by \pkg{roxygen2} in the usage and
#' examples sections.
#' @param path the path of the Rd file
#' @param ... other arguments passed to \code{tidy.source}
#' @return \code{NULL}; as a side effect, the original Rd file will be updated
#' @export
#' @author Yihui Xie <\url{http://yihui.name}>
#' @seealso \code{\link[formatR]{tidy.source}}
#' @note If the usage or examples code is not syntactically correct, it will not
#'   be reformatted and a message will be printed on screen. One possible
#'   situation is the percent symbol \code{\%}, which should be escaped even in
#'   the examples code (cf Writing R Extensions), and this can make the code
#'   syntactically incorrect, e.g. \code{a \%in\% b} should be \code{a
#'   \\\%in\\\% b} but the latter is not valid R code. Anyway, this function
#'   will try to unescape the percent symbols before reformating the code, then
#'   escape them.
#' @examples
#' rd.file = system.file('examples', 'reformat_code_demo.Rd', package = 'Rd2roxygen')
#' file.copy(rd.file, tempdir())
#' fmt.file = file.path(tempdir(), 'reformat_code_demo.Rd')
#'
#' file.show(fmt.file)  ## show the raw Rd
#'
#' reformat_code(fmt.file)
#' file.show(fmt.file)  ## the formatted Rd
reformat_code = function(path, ...) {
  rd = readLines(path)
  tags = sprintf(
    '^\\\\(%s)\\{',
    paste(c("docType", "name", "alias", "title", "format", "source", "usage",
            "arguments", "value", "description", "details", "note", "section",
            "examples", "author", "references", "seealso", "concept", "keyword",
            "subsection"), collapse = "|")
  )
  if (length(idx0 <- grep('^\\\\examples\\{', rd))) {
    # tags after \examples?
    idx1 = grep(tags, rd)
    if (length(idx1) && any(idx1 > idx0))
      idx1 = min(idx1[idx1 > idx0]) - 1 else idx1 = tail(grep('\\}$', rd), 1)
    tmp = rd[idx0:idx1]
    tmp[1] = sub('^\\\\examples\\{', '', tmp[1])
    nn = length(tmp)
    tmp[nn] = sub('\\}$', '', tmp[nn])
    txt = gsub('\\%', '%', tmp, fixed = TRUE) # will escape % later
    txt = sub('^\\\\+dont(run|test|show)', 'tag_name_dont\\1 <- function() ', txt)
    txt = tidy.code(txt, ...)
    if (!inherits(txt, 'try-error')) {
      txt = gsub("(^|[^\\])%", "\\1\\\\%", txt)
      txt = gsub('tag_name_dont(run|test|show) <- function\\(\\) \\{', '\\\\dont\\1{', txt)
      txt[txt == ''] = '\n'
      txt = unlist(strsplit(txt, '\n'))
      # remove the four spaces introduced by disguising \\dontrun as a function
      if (length(idx2 <- grep('\\\\dont(run|test|show)\\{', txt))) {
        for (i in idx2) {
          j = i + 1
          while (txt[j] != '}') {
            txt[j] = sub('^    ', '', txt[j])
            j = j + 1
          }
        }
      }
      txt = paste(txt, rep(c('\n', ''), c(length(txt) - 1, 1)), sep = '', collapse = '')
      txt[1] = paste('\\examples{', txt[1], sep = '')
      nn0 = length(txt)
      txt[nn0] = paste(txt[nn0], '}', sep = '')
      rd[idx0] = paste(txt, collapse = '\n')
      if (idx1 > idx0) rd = rd[-((idx0 + 1):idx1)]
    } else {
      message('(!) failed to reformat examples code in ', path)
      message(paste('   ', tmp, collapse = '\n'))
    }
  }
  if (length(idx0 <- grep('^\\\\usage\\{', rd))) {
    idx1 = grep(tags, rd)
    if (length(idx1) && any(idx1 > idx0))
      idx1 = min(idx1[idx1 > idx0]) - 1 else idx1 = length(rd)
    tmp = rd[idx0:idx1]
    tmp[1] = sub('^\\\\usage\\{', '', tmp[1])
    nn = length(tmp)
    tmp[nn] = sub('\\}$', '', tmp[nn])
    txt = gsub('\\%', '%', tmp, fixed = TRUE) # will escape % later
    txt = gsub('\\\\method\\{([^\\{]+)\\}\\{([^\\{]+)\\}', '`method@\\1@\\2`', txt) # S3
    txt = tidy.code(txt, ...)
    if (!inherits(txt, 'try-error')) {
      txt = gsub('`method@([^@]+)@([^`]+)`', '\\\\method{\\1}{\\2}', txt) # restore S3
      txt = gsub("(^|[^\\])%", "\\1\\\\%", txt)
      if (txt[1] == '') txt = txt[-1]
      if (txt[length(txt)] == '') txt = txt[-length(txt)]
      txt = sprintf('\\usage{\n%s\n}', paste(txt, collapse = '\n'))
      rd[idx0] = txt
      if (idx1 > idx0) rd = rd[-((idx0 + 1):idx1)]
    } else {
      message('(!) failed to reformat usage code in ', path)
      message(paste('   ', tmp, collapse = '\n'))
    }
  }
  # remove trailing spaces
  while (tail(rd, 1) == '') rd = rd[-length(rd)]
  writeLines(rd, path)
  flush.console()
}

tidy.code = function(code, ...) {
  res = try(tidy.source(text = code, output = FALSE, width.cutoff = 80, ...)$text.tidy)
  if (inherits(res, 'try-error')) return(res)
  i = 1
  # R CMD check requires code width to be less than 90
  while (any(nchar(unlist(strsplit(res, '\n'))) >= 90) && i <= 40) {
    res = tidy.source(text = code, output = FALSE, width.cutoff = 90 - i, ...)$text.tidy
    i = i + 1
  }
  if (i > 40) {
    warning('unable to make code width smaller than 90', immediate. = TRUE)
    cat(code, sep = '\n')
    code
  } else res
}
