/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.text;

import java.io.IOException;
import java.math.BigInteger;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;

import javax.xml.bind.JAXBElement;

import org.docx4j.dml.CTRegularTextRun;
import org.docx4j.dml.CTTextCharacterProperties;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.wml.CTBookmark;
import org.docx4j.wml.CTMarkupRange;
import org.docx4j.wml.P;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.Text;
import org.lysis.reporters.formats.TextProperties;
import org.lysis.reporters.html4r.tools.Format;
import org.lysis.reporters.html4r.tools.utils;
import org.lysis.reporters.pptx4r.elements.Utils;
import org.lysis.reporters.tools.HTML4R;


public class Paragraph implements HTML4R{
	private LinkedHashMap<Integer, String> textList;
	private LinkedHashMap<Integer, TextProperties> formatList;
	
	private int index;
	
	private boolean hasBookmark;
	private JAXBElement<CTBookmark> bmStart;
	private JAXBElement<CTMarkupRange> bmEnd;
	private boolean vanilla_format;
	
	public Paragraph( ) {
		textList = new LinkedHashMap<Integer, String>();
		formatList = new LinkedHashMap<Integer, TextProperties>();
		index = 0;
		vanilla_format = false;
	}
	public void setVanilla(){
		vanilla_format = true;
	}
	public void setBookmark(String bookmark, BigInteger bookmarkID){
		hasBookmark = true;
		
		org.docx4j.wml.ObjectFactory factory = Context.getWmlObjectFactory();
		CTBookmark bm = new CTBookmark();
		bm.setId(bookmarkID);
		bm.setName(bookmark);		
		bmStart =  factory.createBodyBookmarkStart(bm);

		CTMarkupRange mr = factory.createCTMarkupRange();
		mr.setId(bookmarkID);
		bmEnd = factory.createBodyBookmarkEnd(mr);
	}
	
	public void addText ( String value, TextProperties tp ) throws IOException{
		index++;
		textList.put(index, value );
		formatList.put(index, tp);
	}
	public void addText ( String value ) throws IOException{
		index++;
		textList.put(index, value );
		formatList.put(index, null);
	}
	
	
	public void addParagraph ( Paragraph value ) throws IOException{
		value = reindex(value, index );
		textList.putAll(value.textList);
		formatList.putAll(value.formatList);
		index = index + value.index;
	}
	
	private static Paragraph reindex(Paragraph value, int add){
		if( value.index < 1 ) return value;
		LinkedHashMap<Integer, String> newTextList = new LinkedHashMap<Integer, String>();
		LinkedHashMap<Integer, TextProperties> newFormatList = new LinkedHashMap<Integer, TextProperties>();
		
		for( int i = 1 ; i <= value.index ; i++ ){
			newTextList.put( i + add , value.textList.get(i));
			newFormatList.put( i + add , value.formatList.get(i));
		}
		value.textList = newTextList;
		value.formatList = newFormatList;
		
		return value;
	}
	
	public void setTextProperties(TextProperties tp){
		for (Iterator<Integer> it1 = formatList.keySet().iterator(); it1.hasNext();) {
			int key = it1.next();
			formatList.put(key, tp);
		}
	}
	


	public String toString(){
		String out = "";
		if( index > 0 )
			for( int i = 1 ; i <= index ; i++ ){
				out += textList.get(i);
			}
		return out;
	}

	@Override
	public String getHTML() {
		String out = "";
		if( index > 0 ){
			for( int i = 1 ; i <= index ; i++ ){
				if( formatList.get(i) != null ){
					TextProperties tp = formatList.get(i);
					String css;
					if( vanilla_format )
						css = Format.getJSString(tp.get_html_pr_vanilla());
					else 
						css = Format.getJSString(tp.get_html_pr());
					out += utils.span_style( textList.get(i), css );
				} else out += utils.span( textList.get(i) );
			}
		}
		return out;
	}

	
	public P get_docx_par(){
		P p = new P();
		fill_docx_par(p);
		return p;
	}
	public P get_docx_par(WordprocessingMLPackage doc, String style){
		P p = doc.getMainDocumentPart().createStyledParagraphOfText(style, "");
		fill_docx_par(p);
		return p;
	}
	
	private void fill_docx_par(P p){
		if( index > 0 ){
			for( int i = 1 ; i <= index ; i++ ){
				if( formatList.get(i) != null ){
					TextProperties tp = formatList.get(i);
					R run = new R();
					Text text = new Text();
					text.setValue( textList.get(i) );
					text.setSpace("preserve");
					run.getContent().add(text);
					RPr rpr = tp.get_docx_pr();
					run.setRPr(rpr);
					p.getContent().add(run);
				} else {
					R run = new R();
					Text text = new Text();
					text.setValue( textList.get(i) );
					text.setSpace("preserve");
					run.getContent().add(text);
					p.getContent().add(run);
				}
				if( hasBookmark ){
					p.getContent().add(0, bmStart);
					p.getContent().add(1, bmEnd);
				} 
			}
		}
	}
	
	public CTTextParagraph get_pptx_par() throws Exception{
		CTTextParagraph textPar = new CTTextParagraph();

		
		if( index > 0 ){
			List<Object> runs = textPar.getEGTextRun();
			for( int i = 1 ; i <= index ; i++ ){
				if( formatList.get(i) != null ){
					TextProperties tp = formatList.get(i);
					CTTextCharacterProperties rpr = tp.get_pptx_pr();
					CTRegularTextRun textRun = Utils.getRun(textList.get(i));
					textRun.setRPr(rpr);
					runs.add(textRun);
				} else {
					CTRegularTextRun textRun = Utils.getRun(textList.get(i));
					runs.add(textRun);
				}
			}
		}
		return textPar;
	}

	
	@Override
	public String getCSS() {
		return "";
	}

	@Override
	public String getJS() {
		return "";
	}

	@Override
	public boolean hasJS() {
		return false;
	}

	@Override
	public boolean hasHTML() {
		return true;
	}

	@Override
	public boolean hasCSS() {
		return false;
	}

}
