/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.img;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

public class ImageMetaInfo {
	private int height;
	private int width;
	private String mime;


	public ImageMetaInfo(File file) throws IOException {
		InputStream is = new FileInputStream(file);
		mime = null;
		width = -1;
		height = -1;
		
		try {
			getInfo(is);
		} finally {
			is.close();
		}
	}

	public ImageMetaInfo(InputStream is) throws IOException {
		mime = null;
		width = -1;
		height = -1;
		getInfo(is);
	}

	public ImageMetaInfo(byte[] bytes) throws IOException {
		InputStream is = new ByteArrayInputStream(bytes);
		mime = null;
		width = -1;
		height = -1;
		try {
			getInfo(is);
		} finally {
			is.close();
		}
	}

	private void getInfo(InputStream is) throws IOException {
		int car1 = is.read();
		int car2 = is.read();
		int car3 = is.read();


		if (car1 == 'G' && car2 == 'I' && car3 == 'F') { // GIF
			is.skip(3);
			width = readInt(is,2,false);
			height = readInt(is,2,false);
			mime = "image/gif";
		} else if (car1 == 0xFF && car2 == 0xD8) { // JPG
			while (car3 == 255) {
				int marker = is.read();
				int len = readInt(is,2,true);
				if (marker == 192 || marker == 193 || marker == 194) {
					is.skip(1);
					height = readInt(is,2,true);
					width = readInt(is,2,true);
					mime = "image/jpeg";
					break;
				}
				is.skip(len - 2);
				car3 = is.read();
			}
		} else if (car1 == 137 && car2 == 80 && car3 == 78) { // PNG
			is.skip(15);
			width = readInt(is,2,true);
			is.skip(2);
			height = readInt(is,2,true);
			mime = "image/png";
		} else if (car1 == 66 && car2 == 77) { // BMP
			is.skip(15);
			width = readInt(is,2,false);
			is.skip(2);
			height = readInt(is,2,false);
			mime = "image/bmp";
		} else {
			int c4 = is.read();
			if ((car1 == 'M' && car2 == 'M' && car3 == 0 && c4 == 42)
					|| (car1 == 'I' && car2 == 'I' && car3 == 42 && c4 == 0)) { //TIFF
				boolean bigEndian = car1 == 'M';
				int ifd = 0;
				int entries;
				ifd = readInt(is,4,bigEndian);
				is.skip(ifd - 8);
				entries = readInt(is,2,bigEndian);
				for (int i = 1; i <= entries; i++) {
					int tag = readInt(is,2,bigEndian);
					int fieldType = readInt(is,2,bigEndian);
					int valOffset;
					if ((fieldType == 3 || fieldType == 8)) {
						valOffset = readInt(is,2,bigEndian);
						is.skip(2);
					} else {
						valOffset = readInt(is,4,bigEndian);
					}
					if (tag == 256) {
						width = valOffset;
					} else if (tag == 257) {
						height = valOffset;
					}
					if (width != -1 && height != -1) {
						mime = "image/tiff";
						break;
					}
				}
			}
		}
		if (mime == null) {
			throw new IOException("Unsupported image type");
		}
	}
	
	private int readInt(InputStream inputStream, int n_bytes, boolean big_endian) throws IOException {
		int out = 0;
		int sv = big_endian ? ((n_bytes - 1) * 8) : 0;
		int cnt = big_endian ? -8 : 8;
		for(int i=0;i<n_bytes;i++) {
			out |= inputStream.read() << sv;
			sv += cnt;
		}
		return out;
	}

	public int getHeight() {
		return height;
	}

	public void setHeight(int height) {
		this.height = height;
	}

	public int getWidth() {
		return width;
	}

	public void setWidth(int width) {
		this.width = width;
	}

	public String getMime() {
		return mime;
	}

	public void setMime(String mime) {
		this.mime = mime;
	}

//	@Override
//	public String toString() {
//		return "MIME Type : " + mimeType + "\t Width : " + width + "\t Height : " + height; 
//	}
}
