% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RmarineHeatWaves.R
\name{detect}
\alias{detect}
\title{Detect Marine Heat Waves and Marine Cold Spells.}
\usage{
detect(data, climatology_start = 1983, climatology_end = 2012,
  pctile = 90, window_half_width = 5, smooth_percentile = TRUE,
  smooth_percentile_width = 31, clim_only = FALSE, min_duration = 5,
  join_across_gaps = TRUE, max_gap = 2, max_pad_length = 3,
  cold_spells = FALSE)
}
\arguments{
\item{data}{A data frame with three columns. They are headed \code{doy},
\code{date} and \code{temp.} \code{doy} is the Julian day running from 1
to 366, but modified so that the day-of-year (doy) vector for non-leap-years
runs 1...59 and then 61...366. For leap years the 60th day is February 29.
The \code{date} column is a vector of dates of class \code{Date}, while
\code{temp} is the temperature. Data of the appropriate format are created
by the function \code{\link{make_whole}}, but your own data can be supplied
if they meet the criteria specified by \code{\link{make_whole}}.}

\item{climatology_start}{The first full year from which the (varying by
day-of-year) seasonal cycle and extremes threshold are calculated (full being
366 days if leap year, else 365 days). Note that a default value is provided
but that it may not be suitable for your own data. Default is \code{1983}.}

\item{climatology_end}{The last full year up to which the  seasonal cycle
and extremes threshold are calculated. Default is \code{2012} (but see comment
above).}

\item{pctile}{Threshold percentile (\%) for detection of extreme values.
Default is \code{90}th percentile. Please see \code{cold_spells} for more
information about the calculation of marine cold spells.}

\item{window_half_width}{Width of sliding window about day-of-year (to one
side of the center day-of-year) used for the pooling of values and
calculation of climatology and threshold percentile. Default is \code{5}
days, which gives a window width of 11 days centered on the 6th day of the
series of 11 days.}

\item{smooth_percentile}{Full width of moving average window for smoothing
climatology and threshold. Default is \code{31} days.}

\item{smooth_percentile_width}{Boolean switch selecting whether to smooth the
climatology and threshold percentile timeseries with a moving average of
width \code{smooth_percentile}. Default is \code{TRUE}.}

\item{clim_only}{Choose to calculate only the climatologies and not the
events. Default is \code{FALSE}.}

\item{min_duration}{Minimum duration for acceptance of detected MHWs.
Default is \code{5} days.}

\item{join_across_gaps}{Boolean switch indicating whether to join MHWs which
occur before/after a short gap as specified by \code{max_gap}. Default
is \code{TRUE}.}

\item{max_gap}{Maximum length of gap allowed for the joining of MHWs. Default
is \code{2} days.}

\item{max_pad_length}{Specifies the maximum length of days over which to
interpolate (pad) missing data (specified as \code{NA}) in the input
temperature time series; i.e., any consecutive blocks of NAs with length
greater than \code{max_pad_length} will be left as \code{NA}. Set as an
integer. Default is \code{3} days.}

\item{cold_spells}{Boolean specifying if the code should detect cold events
instead of heat events. Default is \code{FALSE}. Please note that the
climatological thresholds for cold-spells are calculated the same as for
heatwaves, meaning that \code{pctile} should be set the same regardless
if one is calculating heatwaves or cold-spells. For example, if one wants
to calculate heatwaves above the 90th percentile threshold
(the default) one sets \code{pctile} = 90. Likewise, if one would like
identify the most intense cold-spells one must also set \code{pctile} = 90,
even though cold spells are in fact simply the coldest extreme events in a
time series, which statistically equate to values below the 10th percentile.}
}
\value{
The function will return a list of two components, \code{clim} and
\code{event}, which are the climatology and MHW (or MCS) events, respectively.
The climatology contains the full time series of daily temperatures, as well
as the the seasonal climatology, the threshold and various aspects of the
events that were detected:
  \item{doy}{Julian day (day-of-year). For non-leap years it runs 1...59 and
  61...366, while leap years run 1...366.}
  \item{date}{The date of the temperature measurement.}
  \item{temp}{Seawater temperature on the specified date [deg. C].}
  \item{seas_clim_year}{Climatological seasonal cycle [deg. C].}
  \item{thresh_clim_year}{Seasonally varying threshold (e.g., 90th
  percentile) [deg. C].}
  \item{var_clim_year}{Seasonally varying variance (standard deviation) [deg. C].}
  \item{thresh_criterion}{Boolean indicating if \code{temp} exceeds
  \code{thresh_clim_year}.}
  \item{duration_criterion}{Boolean indicating whether periods of consecutive
  \code{thresh_criterion} are >= \code{min_duration}.}
  \item{event}{Boolean indicating if all criteria that define a MHW or MCS are
  met.}
  \item{event_no}{A sequential number indicating the ID and order of
  occurence of the MHWs or MCSs.}

The events are summarised using a range of event metrics:
  \item{index_start}{Start index of event.}
  \item{index_stop}{Stop index of event.}
  \item{event_no}{A sequential number indicating the ID and order of
  the events.}
  \item{duration}{Duration of event [days].}
  \item{date_start}{Start date of event [date].}
  \item{date_stop}{Stop date of event [date].}
  \item{date_peak}{Date of event peak [date].}
  \item{int_mean}{Mean intensity [deg. C].}
  \item{int_max}{Maximum (peak) intensity [deg. C].}
  \item{int_var}{Intensity variability (standard deviation) [deg. C].}
  \item{int_cum}{Cumulative intensity [deg. C x days].}
  \item{rate_onset}{Onset rate of event [deg. C / day].}
  \item{rate_decline}{Decline rate of event [deg. C / day].}

\code{int_max_rel_thresh}, \code{int_mean_rel_thresh},
\code{int_var_rel_thresh}, and \code{int_cum_rel_thresh}
are as above except relative to the threshold (e.g., 90th percentile) rather
than the seasonal climatology.

\code{int_max_abs}, \code{int_mean_abs}, \code{int_var_abs}, and
\code{int_cum_abs} are as above except as absolute magnitudes
rather than relative to the seasonal climatology or threshold.

\code{int_max_norm} and \code{int_mean_norm} are as above except
units are in multiples of threshold exceedances, i.e., a value of 1.5
indicates the event intensity (relative to the climatology) was 1.5 times the
value of the threshold (relative to climatology,
i.e., threshold - climatology.)

Note that \code{rate_onset} and \code{rate_decline} will return NA
when the event begins/ ends on the first/ last day of the time series. This
may be particularly evident when the function is applied to large gridded
data sets. Although the other metrics do not contain any errors and
provide sensible values, please take this into account in its
interpretation.
}
\description{
Applies the Hobday et al. (2016) marine heat wave definition to an input time
series of temperature along with a daily date vector.
}
\details{
\enumerate{
\item This function assumes that the input time series consists of continuous
daily values with few missing values. Time ranges which start and end
part-way through the calendar year are supported. The accompanying function
\code{\link{make_whole}} aids in the preparation of a time series that is
suitable for use with \code{detect}, although this may also be accomplished
'by hand' as long as the criteria are met as discussed in the documentation
to \code{\link{make_whole}}.
\item It is recommended that a climatology period of at least 30 years is
specified in order to capture decadal thermal periodicities. Currently the
function will only compute climatologies starting from 1 January of the
specified \code{climatology_start} and ending on 31 December of the specified
\code{climatology_end}. Even one day short of a full year (i.e. 365 day during
non-leap years and 366 days during leap years) at the beginning/end of the
climatology period will cause the function to fail. This may be changed in
future versions of the function.
\item This function supports leap years. This is done by ignoring Feb 29s
for the initial calculation of the climatology and threshold. The values for
Feb 29 are then linearly interpolated from the values for Feb 28 and Mar 1.
\item The calculation of onset and decline rates assumes that the events
started a half-day before the start day and ended a half-day after the
end-day. This is consistent with the duration definition as implemented,
which assumes duration = end day - start day + 1. As of version 0.15.7, an
event that is already present at the beginning of a time series, or an event
that is still present at the end of a time series, will report the rate of
onset or the rate of decline as \code{NA}, as it is impossible to know what
the temperature half a day before or after the start or end of the event is.
This may be a departure from the python marineHeatWaves function.
\item For the purposes of event detection, any missing temperature values not
interpolated over (through optional \code{max_pad_length}) will be set equal
to the seasonal climatology. This means they will trigger the end/start of
any adjacent temperature values which satisfy the event definition criteria.
\item If the code is used to detect cold events (\code{coldSpells} = TRUE),
then it works just as for heat waves except that events are detected as
deviations below the (100 - pctile)th percentile  (e.g., the 10th instead of
90th) for at least 5 days. Intensities are reported as negative values and
represent the temperature anomaly below climatology.
\item If only the climatology for the time series is required, and not the
events themselves, this may be done by setting \code{clim_only} = TRUE.
}
The original Python algorithm was written by Eric Oliver, Institute for
Marine and Antarctic Studies, University of Tasmania, Feb 2015, and is
documented by Hobday et al. (2016). The marine cold spell option was
implemented in version 0.13 (21 Nov 2015) of the Python module as a result
of our preparation of Schlegel et al. (submitted), wherein the cold events
receive a brief overview.
}
\examples{
t_dat <- make_whole(sst_WA)
res <- detect(t_dat, climatology_start = 1983, climatology_end = 2012)
# show a portion of the climatology:
res$clim[1:10, ]
# show some of the heat waves:
res$event[1:5, 1:10]
}
\references{
Hobday, A.J. et al. (2016). A hierarchical approach to defining
marine heatwaves, Progress in Oceanography, 141, pp. 227-238,
doi:10.1016/j.pocean.2015.12.014

Schlegel, R. W., Oliver, C. J., Wernberg, T. W., Smit, A. J. (2017).
Coastal and offshore co-occurrences of marine heatwaves and cold-spells.
Progress in Oceanography, 151, pp. 189-205, doi:10.1016/j.pocean.2017.01.004
}
\author{
Albertus J. Smit, Robert W. Schlegel, Eric C. J. Oliver
}
