% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/FeatureSelection.R
\name{FS.quickreduct.FRST}
\alias{FS.quickreduct.FRST}
\title{The fuzzy QuickReduct algorithm based on FRST}
\usage{
FS.quickreduct.FRST(decision.table, type.method = "fuzzy.dependency",
  type.QR = "fuzzy.QR", control = list())
}
\arguments{
\item{decision.table}{an object of a \code{"DecisionTable"} class representing a decision table. See \code{\link{SF.asDecisionTable}}.}

\item{type.method}{a string representing the type of methods.
The complete description can be found in Section \code{Details}.}

\item{type.QR}{a string expressing the type of QuickReduct algorithm which is one of the two following algorithms:
\itemize{
        \item \code{"fuzzy.QR"}: it is the original fuzzy rough QuickReduct algorithm based on (Jensen and Shen, 2002).
        \item \code{"modified.QR"}: it is the modified QuickReduct algorithm based on (Bhatt and Gopal, 2005).
}}

\item{control}{a list of other parameters as follows.
        \itemize{
         \item \code{type.aggregation}: a type of aggregation operator. See \code{\link{BC.IND.relation.FRST}}.
         \item \code{t.implicator}: a type of implicator function. See \code{\link{BC.LU.approximation.FRST}}.
               The default value is \code{"lukasiewicz"}.
         \item \code{type.relation}: a type of indiscernibility relation. See \code{\link{BC.IND.relation.FRST}}.
                The default value is \code{type.relation = c("tolerance", "eq.3")}.
         \item \code{alpha}: a real number between 0 and 1 expressing a threshold value or stopping criterion.
                The following methods use the parameter: \code{"vqrs"},

                \code{"min.positive.reg"}, and \code{"fuzzy.discernibility"}.
                The default value is 0.95.
         \item \code{alpha.precision}: a real number between 0 and 1 expressing variable precision (\eqn{\alpha}) for \code{"fvprs"}.
                See \code{\link{BC.LU.approximation.FRST}}. The default value is 0.05.
         \item \code{q.some}: a pair of numeric values for the alpha and beta parameter of VQRS for the quantifier \code{some}.
                The default value is \code{q.some = c(0.1, 0.6)}.

                See \code{\link{BC.LU.approximation.FRST}}.
         \item \code{q.most}: a pair of numeric values for the alpha and beta parameter of VQRS for the quantifier \code{most}.
                The default value is \code{q.most = c(0.2, 1)}.

                See \code{\link{BC.LU.approximation.FRST}}.
         \item \code{m.owa}: a numeric value to define the parameter in OWA. The default value is the mean number of objects.
         \item \code{type.rfrs}: a type of robust fuzzy rough sets.

                The default is \code{type.rfrs = "k.trimmed.min")}.

                See \code{\link{BC.LU.approximation.FRST}}.
         \item \code{k.rfrs}: a value between 0 and length of data representing index of considered data.
                The default is \code{k.rfrs = round(0.5*nrow(decision.table))}.
               See \code{\link{BC.LU.approximation.FRST}}.
         \item \code{beta.quasi}: a number between 0 and 1 representing \eqn{\beta}-precision t-norms and t-conorms.
                   The default value is 0.05.
         \item \code{randomize}: a boolean value to define whether selecting attributes randomly or not. For more detail,
               see in Section \code{Details}. The default value is \code{FALSE}.
         }
        It should be noted that instead of supplying all the above parameters, we only set
        those parameters needed by the considered method. See in Section \code{Details}.
        Also, we provide some examples to illustrate how the parameters are used.}
}
\value{
A class \code{"FeatureSubset"} that contains the following components:
\itemize{
\item \code{reduct}: a list representing a single reduct. In this case, it could be a superreduct or just a subset of feature.
\item \code{type.method}: a string representing the type of method.
\item \code{type.task}: a string showing the type of task which is \code{"feature selection"}.
\item \code{model}: a string representing the type of model. In this case, it is \code{"FRST"} which means fuzzy rough set theory.
}
}
\description{
It is a function implementing the fuzzy QuickReduct algorithm
for feature selection based on FRST.
The fuzzy QuickReduct is a modification of QuickReduct based on RST (see \code{\link{FS.quickreduct.RST}}).
}
\details{
In this function, we provide an algorithm proposed by
(Jensen and Shen, 2002) which is fuzzy QuickReduct. Then, the algorithm has been modified by (Bhatt and Gopal, 2005) to improve stopping criteria.
This function is aimed to implement both algorithms. These algorithms can be executed by assigning the parameter \code{type.QR}
with \code{"fuzzy.QR"} and \code{"modified.QR"} for fuzzy quickreduct and modified fuzzy quickreduct
algorithms, respectively. Additionally, in the \code{control} parameter, we provide one component which is
\code{randomize} having boolean values: \code{TRUE} or \code{FALSE}. \code{randomize = TRUE} means that
we evaluate some (or not all) attributes randomly along iteration. It will be useful if we have a large number of attributes
in a decision table.

In this function, we have considered many approaches of the lower and upper approximations.
The following list shows considered methods and their descriptions. Additionally, those approaches can be executed by
assigning the following value to the parameter \code{type.method}.
\itemize{
   \item \code{"fuzzy.dependency"}: It is based on the degree of dependency using the implication/t-norm model approximation (Jensen and Shen, 2009).
               The detailed concepts about this approximation have been explained in \code{\link{B.Introduction-FuzzyRoughSets}}
               and

\code{\link{BC.LU.approximation.FRST}}.
   \item \code{"fuzzy.boundary.reg"}: It is based on the fuzzy boundary region proposed by (Jensen and Shen, 2009).
               This algorithm introduced the usage of the total uncertainty degree \eqn{\lambda_B(Q)}
               for all concepts of feature subset \eqn{B} and decision attribute \eqn{Q}.
               The total uncertainty degree is used as a parameter to select appropriate features.
  \item \code{"vqrs"}: It is based on vaquely quantified rough set (VQRS)
               proposed by (Cornelis and Jensen, 2008). See also \code{\link{BC.LU.approximation.FRST}}.
  \item \code{"owa"}: Based on ordered weighted average (OWA) based fuzzy rough set, (Cornelis et al, 2010) proposed
               the degree of dependency as a parameter employed in the algorithm to select appropriate features. The explanation
               about lower and upper approximations based on OWA can be found in \code{\link{BC.LU.approximation.FRST}}.
  \item \code{"rfrs"}: It is based on degree of dependency that is obtained by performing
               the robust fuzzy rough sets proposed by (Hu et al, 2012).
               The detailed concepts about this approximation have been explained in \code{\link{BC.LU.approximation.FRST}}.
  \item \code{"min.positive.reg"}: Based on measure introduced in (Cornelis et al, 2010) which considers the most problematic element in
             the positive region, defined using the implicator/t-norm model.
  \item \code{"fvprs"}: It is based on degree of dependency proposed by (Zhao et al, 2009).
               The degree is obtained by using fuzzy lower approximation based on
               fuzzy variable precision rough set model.
  \item \code{"fuzzy.discernibility"}: This approach attempts to combine the the decision-relative discernibility matrix
              and the fuzzy QuickReduct algorithm. (Jensen and Shen, 2009) introduced a measurement which is the degree of satisfaction to select the attributes.
  \item \code{"beta.pfrs"}: Based on \eqn{\beta}-precision fuzzy rough sets (\eqn{\beta}-PFRS) proposed by (Salido and Murakami, 2003),
               the degree of dependency as a parameter employed in the algorithm to select appropriate features. The explanation
               about lower and upper approximations based on \eqn{\beta}-PFRS can be found in \code{\link{BC.LU.approximation.FRST}}.
}

It should be noted that the parameter \code{type.method} is related to parameter \code{control}.
In other words, we only set the components in the \code{control} parameter that related to the chosen type of method.
The following is a list showing the components of \code{control} needed by each type of methods.
\itemize{
\item \code{type.method = "fuzzy.dependency"}:

\code{control <- list(t.implicator, type.relation, type.aggregation)}

\item \code{type.method = "fuzzy.boundary.reg"}:

\code{control <- list(t.implicator, type.relation, type.aggregation)}

\item \code{type.method = "vqrs"}:

\code{control <- list(alpha, q.some, q.most, type.aggregation)}

\item \code{type.method = "owa"}:

\code{control <- list(t.implicator, type.relation, m.owa, type.aggregation)}

\item \code{type.method = "rfrs"}:

\code{control <- list(t.implicator, type.relation, type.rfrs,}

\code{k.rfrs, type.aggregation)}

\item \code{type.method = "min.positive.reg"}:

\code{control <- list(alpha, t.implicator, type.relation, type.aggregation)}

\item \code{type.method = "fuzzy.discernibility"}:

\code{control <- list(alpha, t.implicator, type.relation, type.aggregation)}

\item \code{type.method = "fvprs"}:

\code{control <- list(alpha.precision, t.implicator, type.relation, type.aggregation)}

\item \code{type.method = "beta.pfrs"}:

\code{control <- list(t.implicator, type.relation, beta.quasi, type.aggregation)}
}
The descriptions of each component can be seen in the documentation of the \code{control} parameter.

It should be noted that this function does not give the new decision table directly.
An additional function called \code{\link{SF.applyDecTable}} is used to produce new decision table based on
information about the reduct from this function. See Section \code{Examples}.
}
\examples{
##########################################################
## Example 1: Dataset containing nominal values on all attributes
##########################################################

data(RoughSetData)
decision.table <- RoughSetData$housing7.dt

########## using fuzzy lower approximation ##############
control <- list(t.implicator = "lukasiewicz", type.relation = c("tolerance", "eq.1"),
               type.aggregation = c("t.tnorm", "lukasiewicz"))
reduct.1 <- FS.quickreduct.FRST(decision.table, type.method = "fuzzy.dependency",
                            type.QR = "fuzzy.QR", control = control)

########## using fuzzy boundary region ##############
\dontrun{control <- list(t.implicator = "lukasiewicz", type.relation = c("tolerance", "eq.1"),
                type.aggregation = c("t.tnorm", "lukasiewicz"))
reduct.2 <- FS.quickreduct.FRST(decision.table, type.method = "fuzzy.boundary.reg",
                            type.QR = "fuzzy.QR", control = control)

########## using vaguely quantified rough sets (VQRS) #########
control <- list(alpha = 0.9, q.some = c(0.1, 0.6), q.most = c(0.2, 1),
                type.aggregation = c("t.tnorm", "lukasiewicz"))
reduct.3 <- FS.quickreduct.FRST(decision.table, type.method = "vqrs",
                            type.QR = "fuzzy.QR", control = control)

########## ordered weighted average (OWA) #########
control <- list(t.implicator = "lukasiewicz", type.relation = c("tolerance", "eq.1"),
                m.owa = 3, type.aggregation = c("t.tnorm","lukasiewicz"))
reduct.4 <- FS.quickreduct.FRST(decision.table, type.method = "owa",
                            type.QR = "fuzzy.QR", control = control)

########## robust fuzzy rough sets (RFRS) #########
control <- list(t.implicator = "lukasiewicz", type.relation = c("tolerance", "eq.1"),
               type.rfrs = "k.trimmed.min", type.aggregation = c("t.tnorm", "lukasiewicz"),
               k.rfrs = 0)
reduct.5 <- FS.quickreduct.FRST(decision.table, type.method = "rfrs",
                            type.QR = "fuzzy.QR", control = control)

########## using min positive region (delta) ###########
control <- list(alpha = 1, t.implicator = "lukasiewicz",
                type.relation = c("tolerance", "eq.1"), type.aggregation =
                                c("t.tnorm", "lukasiewicz"))
reduct.6 <- FS.quickreduct.FRST(decision.table, type.method = "min.positive.reg",
                            type.QR = "fuzzy.QR", control = control)

########## using FVPRS approximation ##############
control <- list(alpha.precision = 0.05, t.implicator = "lukasiewicz",
               type.aggregation = c("t.tnorm", "lukasiewicz"),
               type.relation = c("tolerance", "eq.1"))
reduct.7 <- FS.quickreduct.FRST(decision.table, type.method = "fvprs",
                            type.QR = "fuzzy.QR", control = control)

########## using beta.PFRS approximation ##############
control <- list(t.implicator = "lukasiewicz", type.relation = c("tolerance", "eq.1"),
                beta.quasi = 0.05, type.aggregation = c("t.tnorm", "lukasiewicz"))
reduct.8 <- FS.quickreduct.FRST(decision.table, type.method = "beta.pfrs",
                            type.QR = "fuzzy.QR", control = control)

########## using fuzzy discernibility matrix ##############
control <- list(alpha = 1, type.relation = c("tolerance", "eq.1"),
               type.aggregation = c("t.tnorm", "lukasiewicz"),
                t.implicator = "lukasiewicz")
reduct.9 <- FS.quickreduct.FRST(decision.table, type.method = "fuzzy.discernibility",
                            type.QR = "fuzzy.QR", control = control)}

##########################################################
## Example 2: Dataset containing nominal and continuous values
## In this case, we only provide one method but others work in
## the same way.
## In this example, we will show how to get the
## new decision table as well
##########################################################
data(RoughSetData)
decision.table <- RoughSetData$hiring.dt

########## using fuzzy lower approximation ##############
control <- list(type.aggregation = c("t.tnorm", "lukasiewicz"),
               t.implicator = "lukasiewicz", type.relation = c("tolerance", "eq.1"))
reduct.1 <- FS.quickreduct.FRST(decision.table, type.method = "fuzzy.dependency",
                            type.QR = "fuzzy.QR", control = control)

## get new decision table based on reduct
new.decTable <- SF.applyDecTable(decision.table, reduct.1)
}
\author{
Lala Septem Riza
}
\references{
C. Cornelis, G. Hurtado Martin, R. Jensen, and D. Slezak,
"Feature Selection with Fuzzy Decision Reducts", Information Sciences, vol. 180, no. 2, p. 209 - 224 (2010).

C. Cornelis, N. Verbiest, and R. Jensen, "Ordered Weighted Average Based Fuzzy Rough Sets",
Proceedings of the 5th International Conference on Rough Sets and Knowledge Technology (RSKT 2010),
p. 78 - 85 (2010).

C. Cornelis and R. Jensen, "A Noise-tolerant Approach to Fuzzy-rough Feature Selection",
Proceedings of the 2008 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2008),
p. 1598 - 1605 (2008).

J. M. F. Salido and S. Murakami, "Rough Set Analysis of a General Type of Fuzzy Data
Using Transitive Aggregations of Fuzzy Similarity Relations",
Fuzzy Sets Syst., vol. 139, p. 635 - 660 (2003).

Q. Hu, L. Zhang, S. An, D. Zhang, and D. Yu, "On Robust Fuzzy Rough Set Models",
IEEE Trans. on Fuzzy Systems, vol. 20, no. 4, p. 636 - 651 (2012).

R. B. Bhatt and M. Gopal, "On Fuzzy-rough Sets Approach to Feature Selection",
Pattern Recognition Letters, vol. 26, no. 7, p. 965 - 975 (2005).

R. Jensen and Q. Shen, "Fuzzy-rough Sets for Descriptive Dimensionality Reduction",
In: Proceedings of IEEE International Conference on Fuzzy System, FUZZ-IEEE, p. 29 - 34 (2002).

R. Jensen and Q. Shen, "New Approaches to Fuzzy-rough Feature Selection",
IEEE Transactions on Fuzzy Systems, vol. 17, no. 4, p. 824 - 838 (2009).

S. Y. Zhao, E. C. C. Tsang, and D. G. Chen,
"The Model of Fuzzy Variable Precision Rough Sets",
IEEE Trans. Fuzzy Systems, vol. 17, no. 2,
p. 451 - 467 (2009).
}
\seealso{
\code{\link{FS.quickreduct.RST}} and \code{\link{FS.feature.subset.computation}}.
}

