% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PLSrounding.R
\name{PLSrounding}
\alias{PLSrounding}
\title{PLS inspired rounding}
\usage{
PLSrounding(data, freqVar, roundBase = 3, hierarchies = NULL,
  formula = NULL, ...)
}
\arguments{
\item{data}{Input data as a data frame (inner cells)}

\item{freqVar}{Variable holding counts (inner cells frequencies)}

\item{roundBase}{Rounding base}

\item{hierarchies}{List of hierarchies}

\item{formula}{Model formula defining publishable cells}

\item{...}{Further parameters sent to \code{RoundViaDummy}}
}
\value{
Output is a four-element list with class attribute "PLSrounded" (to ensure informative printing).
   \item{inner}{Data frame corresponding to input data with the main dimensional variables and with cell 
               frequencies (original, rounded, difference).}
   \item{publish}{Data frame of publishable data with the main dimensional variables and with cell frequencies 
                  (original, rounded, difference).}
   \item{metrics}{A named character vector of various statistics calculated from the two output data frames 
   ("\code{inner_}" used to distinguish). See examples below and the function \code{\link{HDutility}}.}
   \item{freqTable}{Matrix of frequencies of cell frequencies and absolute differences.
   For example, row "\code{rounded}" and column "\code{pub.4+}" is the number of rounded 
   inner cell frequencies greater than or equal to \code{4}.}
}
\description{
Small count rounding of necessary inner cells are performed so that all small frequencies of cross-classifications to be published 
(publishable cells) are rounded. The publishable cells can be defined from a model formula, hierarchies or automatically from data.
}
\details{
This function is a user-friendly wrapper for \code{RoundViaDummy} with data frame output and with computed summary of the results.
See \code{\link{RoundViaDummy}} for more details.
}
\examples{
# Small example data set
z <- SmallCountData("e6")
print(z)

# Publishable cells by formula interface
a <- PLSrounding(z, "freq", roundBase = 5,  formula = ~geo + eu + year)
print(a)
print(a$inner)
print(a$publish)
print(a$metrics)
print(a$freqTable)

# Recalculation of maxdiff, HDutility, meanAbsDiff and rootMeanSquare
max(abs(a$publish[, "difference"]))
HDutility(a$publish[, "original"], a$publish[, "rounded"])
mean(abs(a$publish[, "difference"]))
sqrt(mean((a$publish[, "difference"])^2))

# Four lines below produce equivalent results 
# Ordering of rows can be different
PLSrounding(z, "freq")
PLSrounding(z, "freq", formula = ~eu * year + geo * year)
PLSrounding(z[, -2], "freq", hierarchies = SmallCountData("eHrc"))
PLSrounding(z[, -2], "freq", hierarchies = SmallCountData("eDimList"))

# Define publishable cells differently by making use of formula interface
PLSrounding(z, "freq", formula = ~eu * year + geo)

# Define publishable cells differently by making use of hierarchy interface
eHrc2 <- list(geo = c("EU", "@Portugal", "@Spain", "Iceland"), year = c("2018", "2019"))
PLSrounding(z, "freq", hierarchies = eHrc2)

# Package sdcHierarchies can be used to create hierarchies. 
# The small example code below works if this package is available. 
if (require(sdcHierarchies)) {
  z2 <- cbind(geo = c("11", "21", "22"), z[, 3:4], stringsAsFactors = FALSE)
  h2 <- list(
    geo = hier_compute(inp = unique(z2$geo), dim_spec = c(1, 1), root = "Tot", as = "df"),
    year = hier_convert(hier_create(root = "Total", nodes = c("2018", "2019")), as = "df"))
  PLSrounding(z2, "freq", hierarchies = h2)
}
}
