#' Calculate the Logarithmic (Ignorance) Score for a mixture of Normal distributions, for example generated by ensemble dressing
#'
#' @param dressed.ens a list with elements `ens`, a N*R matrix representing N time instances of kernel centers, and `ker.wd`, a N*R matrix with corresponding kernel standard deviations. See function `DressEnsemble` 
#' @param obs a numeric vector of length N with real-valued observations
#' @return numeric vector of length N with the Ignorance score values
#' @examples
#' data(eurotempforecast)
#' d.ens <- DressEnsemble(ens)
#' DressIgn(d.ens, obs)
#' @seealso DressEnsemble, DressCrps
#' @references Roulston and Smith (2002) Evaluating Probabilistic Forecasts Using Information Theory, \doi{10.1175/1520-0493(2002)130<1653:EPFUIT>2.0.CO;2}
#' @export

# return a vector of Ignorance scores for dressed ensembles
DressIgn <- function(dressed.ens, obs) {
  N <- nrow(dressed.ens[["ens"]])

  ign <- with(dressed.ens, {
    sapply(1:N, function(ii) {
      s <- as.numeric(ker.wd[ii, ])
      e <- as.numeric(ens[ii, ])
      o <- as.numeric(obs[ii])
      -log2(mean(dnorm(o, e, s), na.rm=TRUE))
    })
  })

  # return
  ign
}


