% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SOPC_TFM.R
\name{SOPC_TFM}
\alias{SOPC_TFM}
\title{Sparse Online Principal Component Analysis}
\usage{
SOPC_TFM(data, m, p, gamma, eta, A, D)
}
\arguments{
\item{data}{The data used in the SOPC analysis.}

\item{m}{the number of common factors}

\item{p}{the number of variables}

\item{gamma}{Tuning parameter for the sparseness of the loadings matrix.}

\item{eta}{Tuning parameter for the sparseness of the uniquenesses matrix.}

\item{A}{The true A matrix.}

\item{D}{The true D matrix.}
}
\value{
A list of metrics including:
\item{Aso}{Estimated factor loadings matrix obtained from the SOPC analysis.}
\item{Dso}{Estimated uniquenesses vector obtained from the SOPC analysis.}
\item{MSEA}{Mean squared error of the estimated factor loadings (Aso) compared to the true loadings (A).}
\item{MSED}{Mean squared error of the estimated uniquenesses (Dso) compared to the true uniquenesses (D).}
\item{LSA}{Loss metric for the estimated factor loadings (Aso), indicating the relative error compared to the true loadings (A).}
\item{LSD}{Loss metric for the estimated uniquenesses (Dso), indicating the relative error compared to the true uniquenesses (D).}
\item{tauA}{Proportion of zero factor loadings in the estimated loadings matrix (Aso), indicating the sparsity of the loadings.}
}
\description{
This function calculates various metrics for the Sparse Online Principal Component Analysis (SOPC) method. It estimates the factor loadings and uniquenesses while calculating mean squared errors and loss metrics for comparison with true values. Additionally, it computes the proportion of zero factor loadings in the estimated loadings matrix.
}
\examples{
library(MASS)
library(relliptical)
library(SOPC)

SOPC_MSEA <- c()
SOPC_MSED <- c()
SOPC_LSA <- c()
SOPC_LSD <- c()
SOPC_TAUA <- c()

p = 10; m = 5
n = 2000  # Set n to 2000
mu = t(matrix(rep(runif(p, 0, 1000), n), p, n))
mu0 = as.matrix(runif(m, 0))
sigma0 = diag(runif(m, 1))
F = matrix(mvrnorm(n, mu0, sigma0), nrow = n)
A = matrix(runif(p * m, -1, 1), nrow = p)

# Sampling from the Truncated Normal distribution
lower = c(rep(-0.5, p - 3), -5, -5, -Inf)
upper = c(rep(0.5, p - 3), 5, 5, Inf)
Sigma = as.matrix(diag(rep(runif(p, 0, 1))))
mut = runif(p, 0, 10)
trnor = rtelliptical(n, mut, Sigma, lower, upper, dist = "Normal")
epsilon = matrix(trnor, nrow = n)
D = Sigma

data = mu + F \%*\% t(A) + epsilon

Z = data.frame(SOPC_TFM(data, m = m, p = p, gamma = 0.1, eta = 0.8, A = A, D = D))
SOPC_MSEA = c(SOPC_MSEA, Z$MSEA)
SOPC_MSED = c(SOPC_MSED, Z$MSED)
SOPC_LSA = c(SOPC_LSA, Z$LSA)
SOPC_LSD = c(SOPC_LSD, Z$LSD)
SOPC_TAUA = c(SOPC_TAUA, Z$tauA)

# Ensure the data frame has the correct column structure, even with one value
data_F = data.frame(n = rep(n, length(SOPC_MSEA)), MSEA = SOPC_MSEA, MSED = SOPC_MSED,
 LSA = SOPC_LSA, LSD = SOPC_LSD, tauA = SOPC_TAUA)
data_F

}
