% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TLMoments.R
\name{TLMoments}
\alias{TLMoments}
\alias{TLMoments.PWMs}
\alias{TLMoments.data.frame}
\alias{TLMoments.list}
\alias{TLMoments.matrix}
\alias{TLMoments.numeric}
\alias{TLMoments.parameters}
\title{Trimmed L-moments}
\usage{
TLMoments(x, ...)

\method{TLMoments}{numeric}(x, leftrim = 0L, rightrim = 0L,
  max.order = 4L, na.rm = FALSE, computation.method = "auto", ...)

\method{TLMoments}{matrix}(x, leftrim = 0L, rightrim = 0L, max.order = 4L,
  na.rm = FALSE, computation.method = "auto", ...)

\method{TLMoments}{list}(x, leftrim = 0L, rightrim = 0L, max.order = 4L,
  na.rm = FALSE, computation.method = "auto", ...)

\method{TLMoments}{data.frame}(x, formula, leftrim = 0L, max.order = 4L,
  rightrim = 0L, na.rm = FALSE, computation.method = "auto", ...)

\method{TLMoments}{PWMs}(x, leftrim = 0L, rightrim = 0L, ...)

\method{TLMoments}{parameters}(x, leftrim = attr(x, "source")$trimmings[1],
  rightrim = attr(x, "source")$trimmings[2], max.order = 4L, ...)
}
\arguments{
\item{x}{numeric data in form of vector, matrix, list, or data.frame OR an object
of class parameters or PWMs.}

\item{...}{additional arguments}

\item{leftrim}{integer indicating lower trimming parameter, has to be greater than 0.}

\item{rightrim}{integer indicating upper trimming parameter, has to be greater than 0.}

\item{max.order}{integer, maximum order of Trimmed L-moments/ratios, has to be
greater than 1.}

\item{na.rm}{logical, indicates if NAs should be removed. Only if empirical moments
are calculated.}

\item{computation.method}{character, indicating if the computation is performed via
PWMs, direct, recursive, or recurrence (see References Hosking & Balakrishnan, 2015).
Possible values are \code{auto} (default, automatically choose appropriate method), \code{pwm},
\code{direct}, \code{recursive}, or \code{recurrence}. Only if empirical moments are calculated.}

\item{formula}{if \code{x} is data.frame. See examples.}
}
\value{
a list of two dimensions: \code{lambdas}/\code{ratios} are a numeric vector, matrix,
list, or data.frame consisting of the TL-moments/TL-moment-ratios. The list has the class
\code{TLMoments}.
The object contains the following attributes: \itemize{
 \item \code{leftrim}: a numeric giving the used leftrim-argument
 \item \code{rightrim}: a numeric giving the used rightrim-argument
 \item \code{order}: a integer vector with corresponding TL-moment orders
 \item \code{computation.method} a character giving the used computation method
 \item \code{source}: a list with background information (used function, data, n, formula;
 mainly for internal purposes)
}
The attributes are hidden in the print-function for a clearer presentation.
}
\description{
Calculates empirical or theoretical Trimmed L-moments and -ratios up to a specific order.
If empirical moments should be calculated, acceptable input types are numeric vectors,
matrices, lists, data.frames. TLMoments is type-preservative, so the input type is also
the output type. If theoretical moments should be calculated, the input type has to be
of class parameters or PWMs, so an object returned by parameters, as.parameters or
PWMs, as.PWMs.
}
\section{Methods (by class)}{
\itemize{
\item \code{numeric}: TLMoments for numeric vector of data

\item \code{matrix}: TLMoments for numeric matrix of data

\item \code{list}: TLMoments for numeric list of data

\item \code{data.frame}: TLMoments for numeric data.frame of data

\item \code{PWMs}: TLMoments for PWMs-object

\item \code{parameters}: TLMoments for parameters-object
}}
\examples{
# Generating data sets:
xmat <- matrix(rnorm(100), nc = 4)
xvec <- xmat[, 3]
xlist <- lapply(1L:ncol(xmat), function(i) xmat[, i])
xdat <- data.frame(
 station = rep(letters[1:2], each = 50),
 season = rep(c("S", "W"), 50),
 hq = as.vector(xmat)
)

# Calculating TL-moments from data:
TLMoments(xvec, leftrim = 0, rightrim = 1)
TLMoments(xmat, leftrim = 1, rightrim = 1)
TLMoments(xlist)
TLMoments(xdat, hq ~ station, leftrim = 0, rightrim = 2)
TLMoments(xdat, hq ~ season, leftrim = 0, rightrim = 2)
TLMoments(xdat, hq ~ ., leftrim = 0, rightrim = 2)

# Calculating TL-moments from PWMs:
TLMoments(PWMs(xvec))
TLMoments(PWMs(xmat), rightrim = 1)
TLMoments(PWMs(xlist), leftrim = 1, rightrim = 1)
TLMoments(PWMs(xdat, hq ~ station), leftrim = 0, rightrim = 2)
TLMoments(PWMs(xdat, hq ~ station + season), leftrim = 0, rightrim = 2)

# Calculating TL-moments from parameters:
(tlm <- TLMoments(xmat, leftrim = 0, rightrim = 1))
TLMoments(parameters(tlm, "gev"))

p <- as.parameters(loc = 3, scale = 2, shape = .4, distr = "gev")
TLMoments(p, rightrim = 1)
p <- as.parameters(cbind(loc = 10, scale = 4, shape = seq(0, .4, .1)), distr = "gev")
TLMoments(p, max.order = 6)
p <- as.parameters(list(
 list(loc = 3, scale = 2, shape = .4),
 list(loc = 3, scale = 2, shape = .2)
), distr = "gev")
TLMoments(p)
p <- as.parameters(data.frame(
 station = letters[1:2],
 loc = c(2, 3),
 scale = c(2, 2),
 shape = c(.4, .2)
), .~station, distr = "gev")
TLMoments(p)
}
\references{
Elamir, E. A., & Seheult, A. H. (2003). Trimmed L-moments. Computational Statistics & Data Analysis, 43(3), 299-314.

Hosking, J. R. (1990). L-moments: analysis and estimation of distributions using linear combinations of order statistics. Journal of the Royal Statistical Society. Series B (Methodological), 105-124.

Hosking, J. R. M. (2007). Some theory and practical uses of trimmed L-moments. Journal of Statistical Planning and Inference, 137(9), 3024-3039.

Hosking, J. R. M., & Balakrishnan, N. (2015). A uniqueness result for L-estimators, with applications to L-moments. Statistical Methodology, 24, 69-80.
}

