compare.trtsel <-
function(trtsel1, trtsel2, bootstraps = 500, alpha = .05, plot = TRUE, 
                           ci   = "horizontal", fixed.values =  NULL, offset = .01,
                            conf.bands = TRUE, conf.bandsN =100, marker.names = c("Marker 1", "Marker 2"), 
                           xlab = NULL, 
                           ylab = NULL, 
                           xlim = NULL, 
                           ylim = NULL, 
                           main = NULL, annotate.plot = TRUE){

  # assume paired data here, so each individual has a measurement on y1 and y2. Also I am assuming each data set is ordered the same way. 


  if(!is.trtsel(trtsel1)) stop("trtsel1 must be an object of class 'trtsel' created by using the function 'trtsel' see ?trtsel for more help")
  if(!is.trtsel(trtsel2)) stop("trtsel2 must be an object of class 'trtsel' created by using the function 'trtsel' see ?trtsel for more help")
  
 
  
  if(alpha<0 | alpha > 1) stop("Error: alpha should be between 0 and 1")
  if(bootstraps ==0 ) cat("bootstrap confidence intervals will not be calculated\n")
  if(bootstraps == 1) {warning("Number of bootstraps must be greater than 1, bootstrap confidence intervals will not be computed"); bootstraps <- 0;}  
  
  
  study.design  <-trtsel1$model.fit$study.design
  rho   <-trtsel1$model.fit$cohort.attributes #because of paired data, rho should be the same for each trtsel object
  link <- trtsel1$model.fit$link
  
  data1 <- trtsel1$derived.data 
  data2 <- trtsel2$derived.data
  
  if(names(data1)[6]!=names(data2)[6]) stop("default.trt is different between markers. Summary measure comparison would not be valid.")
  
  #cant compare a biomarker- provided trtsel object with one that was generated by fitted risks being input
  if(ncol(data1) != ncol(data2)) stop("cannot compare a trtsel object that was created by providing fitted risk to another trtsel object that was not. Bootstrapping methods are not comparable.") 
  
  if(nrow(data1) != nrow(data2)) stop("trtsel objects must have the same number of observations for comparison")
  if(!all.equal(data1[,1:2], data2[,1:2])) stop("trt and event data must be identical to compare markers!")
  boot.sample <- trtsel1$functions$boot.sample
  get.summary.measures <- trtsel1$functions$get.summary.measures
  
  if(bootstraps > 1){
  #get bootstrap data
  boot.data <- replicate(bootstraps, one.boot.compare(data1 = data1, data2 = data2, rho = rho, study.design = study.design, obe.boot.sample = boot.sample, obe.get.summary.measures = get.summary.measures, link = link, d = trtsel1$model.fit$thresh, disc.marker.neg = trtsel1$model.fit$disc.marker.neg))
  
  boot.data1 <- boot.data[c(1:4, 9:18),]
  boot.data2 <- boot.data[c(5:8, 25:34),]

  boot.data = boot.data1 - boot.data2

  ## Estimate summary measures
  
  sm.m1 <- get.summary.measures(data1, rho)
  sm.m2 <- get.summary.measures(data2, rho)
  sm.diff <- as.data.frame(t(unlist(sm.m1) - unlist(sm.m2) ))
  
  ci.m1   <- apply(boot.data1[-c(1:4),], 1, quantile, probs = c(alpha/2, 1-alpha/2), na.rm = TRUE)
  ci.m2   <- apply(boot.data2[-c(1:4),], 1, quantile, probs = c(alpha/2, 1-alpha/2), na.rm = TRUE)

  ci.diff <- apply(boot.data[-c(1:4),], 1, quantile, probs = c(alpha/2, 1-alpha/2), na.rm = TRUE)


  ## Get p-values for differences in all performance measures (ie boot.data[5:13,])

  potential.pvals <- (1:bootstraps)/bootstraps
  p.vals <- rep(0, 10)  

  for( i in 1:10 ){

   tmp.ind <- c(5:14)[i]
   tmp.boot.data <- boot.data[tmp.ind,]
   tmp.boot.data <- tmp.boot.data[is.finite(tmp.boot.data)]

   if(!cover(min(tmp.boot.data), max(tmp.boot.data), 0) ){ 
     
     p.vals[i] <- 0


   }else{

     reject.all <- unname( mapply( cover, 
                                   quantile(tmp.boot.data, potential.pvals/2, type = 1, na.rm = TRUE),
                                   quantile(tmp.boot.data, 1 - potential.pvals/2, type = 1, na.rm = TRUE), 
                                   rep(0, bootstraps))  )
     reject.all <- c(reject.all, FALSE)
     tmp.reject <- which(reject.all==FALSE)[1] 
     p.vals[i] <- potential.pvals[ifelse(tmp.reject==1, 1, tmp.reject - 1)]
   }
    
  }

  p.vals <- data.frame(t(p.vals))
  names(p.vals) <- names(sm.diff)[1:10]
  row.names(p.vals) <- c("p.values")
  row.names(ci.m1) <- c("lower", "upper")
  row.names(ci.m2) <- c("lower", "upper")
  row.names(ci.diff) <- c("lower", "upper")
  result <- list(estimates.marker1   = data.frame(sm.m1),
                 estimates.marker2   = data.frame(sm.m2), 
                 estimates.diff = data.frame(sm.diff), 
                 ci.marker1   = data.frame(ci.m1), 
                 ci.marker2   = data.frame(ci.m2), 
                 ci.diff   = data.frame(ci.diff),
                 trtsel1 = trtsel1, 
                 trtsel2 = trtsel2, 
                 p.values = p.vals,  
                 bootstraps = bootstraps)

  }else{
  sm.m1 <- get.summary.measures(data1, rho)
  sm.m2 <- get.summary.measures(data2, rho)
  sm.diff <- as.data.frame(t(unlist(sm.m1) - unlist(sm.m2) ))

result <- list(estimates.marker1   = data.frame(sm.m1),
                 estimates.marker2   = data.frame(sm.m2), 
                 estimates.diff = data.frame(sm.diff), 
                 trtsel1 = trtsel1, 
                 trtsel2 = trtsel2, 
                 bootstraps = bootstraps)
  }

  #for plotting, we can only compare marker types that are the same...ie discrete to discrete, continuous to continuous
  same.marker.type = (is.null(trtsel1$model.fit$disc.marker.neg) == is.null(trtsel2$model.fit$disc.marker.neg))
  if(plot & !same.marker.type) {
    warning("Can not generate plots to compare a discrete marker to a continuous marker (bootstrap methods are not comparable). No plots will be produced!")
    plot = FALSE
  }
  

  if(plot & is.null(trtsel1$model.fit$disc.marker.neg)){ 
    if(!is.element(ci, c("vertical", "horizontal"))) stop("If plotting, ci must be one of `vertical` or `horizontal`")
    
  if(length(fixed.values>0)) conf.bands = FALSE 
  if(conf.bands){
    
   if(substr(ci, 1,1 )=="v"){
      fixed.values1 = seq(from = 0, to = 100, length.out = conf.bandsN)
      fixed.values2 = seq(from = 0, to = 100, length.out = conf.bandsN)
      offset = 0

   }else{
      fixed.values1 = seq(from = min(c(data1$trt.effect)), to = max(c(data1$trt.effect)), length.out = conf.bandsN)
      fixed.values2 = seq(from = min(c(data2$trt.effect)), to = max(c(data2$trt.effect)), length.out = conf.bandsN)
      offset = 0

   }

  }else{
   fixed.values1 <- fixed.values
   fixed.values2 <- fixed.values
  } 
  
  curves <-  myplotcompare.trtsel( x = result, bootstraps =bootstraps, alpha  = alpha, ci = ci, marker.names = marker.names,
                           fixeddeltas.y1 =  fixed.values1, fixeddeltas.y2 = fixed.values2, 
                           xlab = xlab, 
                           ylab = ylab, 
                           xlim = xlim, 
                           ylim = ylim, 
                           main = main, offset = offset, conf.bands=conf.bands)
  result$plot <- curves$plot

  result$plot.ci.marker1 <- curves$trtsel1$conf.intervals
  result$plot.ci.marker2 <- curves$trtsel2$conf.intervals
  }else if(plot & !is.null(trtsel1$model.fit$disc.marker.neg)){
    
    curves <-  myplotcompare.trtsel_disc( x = result, bootstraps =bootstraps, alpha  = alpha, ci = ci, marker.names = marker.names, 
                                     xlab = xlab, 
                                     ylab = ylab, 
                                     xlim = xlim, 
                                     ylim = ylim, 
                                     main = main, offset = offset, conf.bands=conf.bands, annotate.plot)
    result$plot <- curves$plot
    result$plot.ci <- curves$ci.bounds

    
    
  }
  
  
  
  result$trtsel1 <- NULL
  result$trtsel2 <- NULL
  class(result) <- "compare.trtsel"
  result$alpha = alpha

  return(result) 

}
