% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vca.R
\name{anovaDF}
\alias{anovaDF}
\title{ANOVA Type-I Degrees of Freedom.}
\usage{
anovaDF(form, Data, Zmat, Amat, tol = 1e-08)
}
\arguments{
\item{form}{(formula) object specifying the model for which ANOVA DFs are requested}

\item{Data}{(data.frame) with all variables appearing in 'form'}

\item{Zmat}{(list) of Z-matrices, representing the design matrices of all model-terms, interpreted 
as fixed effects (number of columns represent number of unique levels)}

\item{Amat}{(list) of A-matrices, generating ANOVA sums of squares as quadratic forms in \eqn{y},
see \code{\link{anovaVCA}} for details}

\item{tol}{(numeric) constant, representing a numeric tolerance used in computing the rank of
\eqn{A}-matrices}
}
\description{
Depending on the type of model, e.g. fully-nested, crossed-nested, etc. algorithms
are applied which are believed to be reasonably fast for the respective type of model, whenever
ANOVA sums of squares are constructed via quadradic forms in y (SSQ.method="qf").
}
\details{
This function is not meant to be called directly. It is invoked by functions \code{\link{anovaVCA}}
and \code{\link{anovaMM}}.

Computing the rank of a corresponding \code{A}-matrix, which generates ANOVA sum of squares as
quadratic form in \eqn{y} is a general method applicable to all types of models. This usually
envokes a singular value decomposition of \eqn{A} which makes it rather slow. 
Here, we try to speed up things by differentiating three classes of models, 1) fully-nested 
models where DFs are computed as the number of factor-levels minus the sum of higher order terms
minus 1, 2) models with only main factors (in this case \code{\link{anova.lm}} is used), 3) models
with main factors and interaction terms. Main factors DFs are computed employing function \code{\link{anova.lm}}.
DFs for interaction terms are computed by determining the rank of \eqn{A}-matrices. There are certain 
designs for which \code{anova.lm} becomes very fast (see examples section of \code{\link{anovaMM}}).
}
\examples{
\dontrun{
# fully-nested design
data(realData)
datP1 <- realData[realData$PID==1,]
system.time(anova.lm.Tab1 <- anova(lm(y~lot/calibration/day/run, datP1)))
system.time(anovaMM.Tab1  <- anovaMM(y~lot/calibration/day/run, datP1))
anova.lm.Tab1
anovaMM.Tab1

# use SSQ.method="qf" (based on quadratic forms)
system.time(anovaMM.Tab1.qf  <- anovaMM(y~lot/calibration/day/run, datP1, SSQ.method="qf"))

# compute degrees of freedom
VCA:::anovaDF( y~lot/calibration/day/run, datP1,
				  Zmat=anovaMM.Tab1.qf$Matrices$Z,
				  Amat=anovaMM.Tab1.qf$Matrices$A)

# design with only main-factors
system.time(anova.lm.Tab2 <- anova(lm(y~lot+calibration+day+run, datP1)))
system.time(anovaMM.Tab2  <- anovaMM(y~lot+calibration+day+run, datP1))
anova.lm.Tab2
anovaMM.Tab2

# use SSQ.method="qf" (based on quadratic forms)
system.time(anovaMM.Tab2.qf  <- anovaMM(y~lot+calibration+day+run, datP1, SSQ.method="qf"))

# compute degrees of freedom
VCA:::anovaDF( y~lot+calibration+day+run, datP1,
				  Zmat=anovaMM.Tab2.qf$Matrices$Z,
				  Amat=anovaMM.Tab2.qf$Matrices$A)

# design with main-factors and interactions
system.time(anova.lm.Tab3 <- anova(lm(y~(lot+calibration)/day/run, datP1)))
system.time(anovaMM.Tab3  <- anovaMM( y~(lot+calibration)/day/run, datP1))
anova.lm.Tab3
anovaMM.Tab3

# use SSQ.method="qf" (based on quadratic forms)
system.time(anovaMM.Tab3.qf  <- anovaMM(y~(lot+calibration)/day/run, datP1, SSQ.method="qf"))

# compute degrees of freedom
VCA:::anovaDF( y~(lot+calibration)/day/run, datP1,
				  Zmat=anovaMM.Tab3.qf$Matrices$Z,
				  Amat=anovaMM.Tab3.qf$Matrices$A)
}
}
\author{
Andre Schuetzenmeister \email{andre.schuetzenmeister@roche.com}
}
