\name{setx}

\alias{setx}

\title{Setting Explanatory Variable Values}

\description{ The \code{setx} command uses the variables identified in
  the \code{formula} generated by \code{zelig} and sets the values of
  the explanatory variables to the selected values.  Use \code{setx}
  after \code{zelig} and before \code{sim} to simulate quantities of
  interest.  }

\usage{
x.out <- setx(object, fn = list(numeric = mean, ordered = median, 
                                others = mode), 
              data = NULL, cond = FALSE, \dots)
}

\arguments{
  \item{object}{the saved output from \code{\link{zelig}}. }
  \item{fn}{a list of functions to apply to three types of variables:
      \itemize{
      \item{numeric}{\code{numeric} variables are set to their mean by
      default, but you may select any mathematical function to apply to
      numeric variables.}
      \item{ordered}{\code{ordered} factors are set to their meidan by
      default, and most mathematical operations will work on them.  If
      you select \code{ordered = mean}, however, \code{setx} will
      default to median with a warning.}
      \item{other}{variables may consist of unordered factors, character
	strings, or logical variables.  The \code{other} variables may
	only be set to their mode.  If you wish to set one of the other
	variables to a specific value, you may do so using \code{\dots}
	below. }
      }
    In the special case \code{fn = NULL}, \code{setx} will return all
    of the observations without applying any function to the data.  }
  \item{data}{a new data frame used to set the values of
    explanatory variables. If \code{data = NULL} (the default), the
    data frame called in \code{zelig} is used. }
  \item{cond}{a logical value indicating whether unconditional
    (default) or conditional (choose \code{cond = TRUE}) prediction
    should be performed.  If you choose \code{cond = TRUE}, \code{setx}
    will coerce \code{fn = NULL} and ignore the additional arguments in 
    \code{\dots}.  If \code{cond = TRUE} and \code{data = NULL},
    \code{setx} will prompt you for a data frame.  }  
  \item{\dots}{user-defined values of specific variables
    overwriting the default values set by the function \code{fn}.  For
    example, adding \code{var1 = mean(data\$var1)} or \code{x1 = 12}
    explicitly sets the value of \code{x1} to 12.  In addition, you may
    specify one explanatory variable as a range of values, creating one
    observation for every unique value in the range of values. }
}

\value{
  For unconditional prediction, \code{x.out} is a model matrix based
  on the specified values for the explanatory variables.  For multiple
  analyses (i.e., when choosing the \code{by} option in \code{\link{zelig}},
  \code{setx} returns the selected values calculated over the entire
  data frame.  If you wish to calculate values over just one subset of
  the data frame, the 5th subset for example, you may use:  
\code{x.out <- setx(z.out[[5]])}

For conditional prediction, \code{x.out} includes the model matrix
  and the dependent variables.  For multiple analyses (when choosing
  the \code{by} option in \code{zelig}), \code{setx} returns the
  observed explanatory variables in each subset.
}

\examples{
# Unconditional prediction:
data(turnout)
z.out <- zelig(vote ~ race + educate, model = "logit", data = turnout)
x.out <- setx(z.out)
s.out <- sim(z.out, x = x.out)

# Unconditional prediction with all observations:
x.out <- setx(z.out, fn = NULL)
s.out <- sim(z.out, x = x.out)

# Unconditional prediction with out of sample data:
z.out <- zelig(vote ~ race + educate, model = "logit",
               data = turnout[1:1000,])
x.out <- setx(z.out, data = turnout[1001:2000,])
s.out <- sim(z.out, x = x.out)

# Using a user-defined function in fn:
\dontrun{
quants <- function(x)
  quantile(x, 0.25)
x.out <- setx(z.out, fn = list(numeric = quants))
}

# Conditional prediction:  
\dontrun{library(MatchIt)
data(lalonde)
match.out <- matchit(treat ~ age + educ + black + hispan + married + 
                     nodegree + re74 + re75, data = lalonde)
z.out <- zelig(re78 ~ distance, data = match.data(match.out, "control"), 
               model = "ls")
x.out <- setx(z.out, fn = NULL, data = match.data(match.out, "treat"),
	      cond = TRUE)
s.out <- sim(z.out, x = x.out)
}
}

\seealso{ The full Zelig manual may be accessed online at
  \url{http://gking.harvard.edu/zelig}.  }

\author{
  Kosuke Imai <\email{kimai@princeton.edu}>; Gary King
  <\email{king@harvard.edu}>; Olivia Lau <\email{olau@fas.harvard.edu}>
}

\keyword{file}
