% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/abclass.R
\name{abclass}
\alias{abclass}
\title{Angle-Based Classification}
\usage{
abclass(
  x,
  y,
  intercept = TRUE,
  weight = NULL,
  loss = c("logistic", "boost", "hinge-boost", "lum"),
  lambda = NULL,
  alpha = 0.5,
  nlambda = 50,
  lambda_min_ratio = NULL,
  nfolds = 0,
  stratified_cv = TRUE,
  lum_a = 1,
  lum_c = 0,
  boost_umin = -3,
  max_iter = 1e+05,
  rel_tol = 1e-04,
  standardize = TRUE,
  varying_active_set = TRUE,
  verbose = 0,
  ...
)
}
\arguments{
\item{x}{A numeric matrix representing the design matrix.  No missing valus
are allowed.  The coefficient estimates for constant columns will be
zero.  Thus, one should set the argument \code{intercept} to \code{TRUE}
to include an intercept term instead of adding an all-one column to
\code{x}.}

\item{y}{An integer vector, a character vector, or a factor vector
representing the response label.}

\item{intercept}{A logical value indicating if an intercept should be
considered in the model.  The default value is \code{TRUE} and the
intercept is excluded from regularization.}

\item{weight}{A numeric vector for nonnegative observation weights. Equal
observation weights are used by default.}

\item{loss}{A character value specifying the loss function.  The available
options are \code{"logistic"} for the logistic deviance loss,
\code{"boost"} for the exponential loss approximating Boosting machines,
\code{"hinge-boost"} for hybrid of SVM and AdaBoost machine, and
\code{"lum"} for largin-margin unified machines (LUM).  See Liu, et
al. (2011) for details.}

\item{lambda}{A numeric vector specifying the tuning parameter \emph{lambda}
of elastic-net penalty.  A data-driven \emph{lambda} sequence will be
generated and used according to specified \code{alpha}, \code{nlambda}
and \code{lambda_min_ratio} if this argument is \code{NULL} by default.
The specified \code{lambda} will be sorted in decreasing order
internally and only the unique values will be kept.}

\item{alpha}{A numeric value in [0, 1] representing the mixing parameter
\emph{alpha} in elastic-net penalty.}

\item{nlambda}{A positive integer specifying the length of the internally
generated \emph{lambda} sequence.  This argument will be ignored if a
valid \code{lambda} is specified.  The default value is \code{50}.}

\item{lambda_min_ratio}{A positive number specifying the ratio of the
smallest lambda parameter to the largest lambda parameter.  The default
value is set to \code{1e-4} if the sample size is larger than the number
of predictors, and \code{1e-2} otherwise.}

\item{nfolds}{A nonnegative integer specifying the number of folds for
cross-validation.  The default value is \code{0} and no cross-validation
will be performed if \code{nfolds < 2}.}

\item{stratified_cv}{A logical value indicating if the cross-validation
procedure should be stratified by the response label. The default value
is \code{TRUE}.}

\item{lum_a}{A positive number specifying the parameter \emph{a} in LUM,
which will be used only if \code{loss = "lum"}.  The default value is
\code{1.0}.}

\item{lum_c}{A nonnegative number specifying the parameter \emph{c} in LUM,
which will be used only if \code{loss = "hinge-boost"} or \code{loss =
"lum"}.  The default value is \code{0.0}.}

\item{boost_umin}{A negative number for adjusting the boosting loss for the
internal majorization procedure.  The default value is \code{-3.0}.}

\item{max_iter}{A positive integer specifying the maximum number of
iteration.  The default value is \code{10^5}.}

\item{rel_tol}{A positive number specifying the relative tolerance that
determines convergence.  The default value is \code{1e-4}.}

\item{standardize}{A logical value indicating if each column of the design
matrix should be standardized internally to have mean zero and standard
deviation equal to the sample size.  The default value is \code{TRUE}.
Notice that the coefficient estimates are always returned on the
original scale.}

\item{varying_active_set}{A logical value indicating if the active set
should be updated after each cycle of coordinate-majorization-descent
algorithm.  The default value is \code{TRUE} for usually more efficient
estimation procedure.}

\item{verbose}{A nonnegative integer specifying if the estimation procedure
should print out intermediate steps/results.  The default value is
\code{0} for silent estimation procedure.}

\item{...}{Other arguments not used now.}
}
\value{
An object of class {abclass}.
}
\description{
Multi-category angle-based large-margin classifiers with regularization by
the elastic-net penalty.
}
\examples{
library(abclass)

## follow example 1 in Zhang and Liu (2014)
ntrain <- 100 # size of training set
ntest <- 1000 # size of testing set
p <- 100      # number of predictors
k <- 5        # number of categories

n <- ntrain + ntest
train_idx <- seq_len(ntrain)
set.seed(123)
y <- sample(k, size = n, replace = TRUE)       # response
mu <- matrix(rnorm(p * k), nrow = k, ncol = p) # mean vector
## normalize the mean vector so that they are distributed on the unit circle
mu <- mu / apply(mu, 1, function(a) sqrt(sum(a ^ 2)))
x <- t(sapply(y, function(i) rnorm(p, mean = mu[i, ], sd = 0.25)))
train_x <- x[train_idx, ]
test_x <- x[- train_idx, ]
y <- factor(paste0("label_", y))
train_y <- y[train_idx]
test_y <- y[- train_idx]

## logistic deviance loss by default
model1 <- abclass(train_x, train_y, nlambda = 10, nfolds = 3,
                  lambda_min_ratio = 1e-3, rel_tol = 1e-3)
pred1 <- predict(model1, test_x)
table(test_y, pred1)
mean(test_y == pred1) # accuracy

## exponential loss approximating AdaBoost
model2 <- abclass(train_x, train_y, nlambda = 10, nfolds = 3,
                  loss = "boost", rel_tol = 1e-3)
pred2 <- predict(model2, test_x)
table(test_y, pred2)
mean(test_y == pred2) # accuracy
}
\references{
Zhang, C., & Liu, Y. (2014). Multicategory Angle-Based Large-Margin
Classification. \emph{Biometrika}, 101(3), 625--640.

Liu, Y., Zhang, H. H., & Wu, Y. (2011). Hard or soft classification?
large-margin unified machines. \emph{Journal of the American Statistical
Association}, 106(493), 166--177.
}
