% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_var_merged_ef_msrc.R
\name{derive_var_merged_ef_msrc}
\alias{derive_var_merged_ef_msrc}
\title{Merge an Existence Flag From Multiple Sources}
\usage{
derive_var_merged_ef_msrc(
  dataset,
  by_vars,
  flag_events,
  source_datasets,
  new_var,
  true_value = "Y",
  false_value = NA_character_,
  missing_value = NA_character_
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{by_vars} argument are expected to be in the dataset.}

\item{by_vars}{Grouping variables

\emph{Permitted Values}: list of variables created by \code{exprs()}
e.g. \code{exprs(USUBJID, VISIT)}}

\item{flag_events}{Flag events

A list of \code{flag_event()} objects is expected. For each event the condition
(\code{condition} field) is evaluated in the source dataset referenced by the
\code{dataset_name} field. If it evaluates to \code{TRUE} at least once, the new
variable is set to \code{true_value}.}

\item{source_datasets}{Source datasets

A named list of datasets is expected. The \code{dataset_name} field of
\code{flag_event()} refers to the dataset provided in the list.}

\item{new_var}{New variable

The specified variable is added to the input dataset.}

\item{true_value}{True value

The new variable (\code{new_var}) is set to the specified value for all by
groups for which at least one of the source object (\code{sources}) has the
condition evaluate to \code{TRUE}.

The values of \code{true_value}, \code{false_value}, and \code{missing_value} must be of
the same type.}

\item{false_value}{False value

The new variable (\code{new_var}) is set to the specified value for all by
groups which occur in at least one source (\code{sources}) but the condition
never evaluates to \code{TRUE}.

The values of \code{true_value}, \code{false_value}, and \code{missing_value} must be of
the same type.}

\item{missing_value}{Values used for missing information

The new variable is set to the specified value for all by groups without
observations in any of the sources (\code{sources}).

The values of \code{true_value}, \code{false_value}, and \code{missing_value} must be of
the same type.}
}
\value{
The output dataset contains all observations and variables of the
input dataset and additionally the variable specified for \code{new_var}.
}
\description{
Adds a flag variable to the input dataset which indicates if
there exists at least one observation in one of the source datasets
fulfilling a certain condition. For example, if a dose adjustment flag
should be added to \code{ADEX} but the dose adjustment information is collected
in different datasets, e.g., \code{EX}, \code{EC}, and \code{FA}.
}
\details{
\enumerate{
\item For each \code{flag_event()} object specified for \code{flag_events}: The
condition (\code{condition}) is evaluated in the dataset referenced by
\code{dataset_name}. If the \code{by_vars} field is specified the dataset is grouped
by the specified variables for evaluating the condition. If named elements
are used in \code{by_vars} like \code{by_vars = exprs(USUBJID, EXLNKID = ECLNKID)},
the variables are renamed after the evaluation. If the \code{by_vars} element is
not specified, the observations are grouped by the variables specified for
the \code{by_vars} argument.
\item The new variable (\code{new_var}) is added to the input dataset and set to
the true value (\code{true_value}) if for the by group at least one condition
evaluates to \code{TRUE} in one of the sources. It is set to the false value
(\code{false_value}) if for the by group at least one observation exists and for
all observations the condition evaluates to \code{FALSE} or \code{NA}. Otherwise, it
is set to the missing value (\code{missing_value}).
}
}
\examples{
library(dplyr)

# Derive a flag indicating anti-cancer treatment based on CM and PR
adsl <- tribble(
  ~USUBJID,
  "1",
  "2",
  "3",
  "4"
)

cm <- tribble(
  ~USUBJID, ~CMCAT,        ~CMSEQ,
  "1",      "ANTI-CANCER",      1,
  "1",      "GENERAL",          2,
  "2",      "GENERAL",          1,
  "3",      "ANTI-CANCER",      1
)

# Assuming all records in PR indicate cancer treatment
pr <- tibble::tribble(
  ~USUBJID, ~PRSEQ,
  "2",      1,
  "3",      1
)

derive_var_merged_ef_msrc(
  adsl,
  by_vars = exprs(USUBJID),
  flag_events = list(
    flag_event(
      dataset_name = "cm",
      condition = CMCAT == "ANTI-CANCER"
    ),
    flag_event(
      dataset_name = "pr"
    )
  ),
  source_datasets = list(cm = cm, pr = pr),
  new_var = CANCTRFL
)

# Using different by variables depending on the source
# Add a dose adjustment flag to ADEX based on ADEX, EC, and FA
adex <- tribble(
  ~USUBJID, ~EXLNKID, ~EXADJ,
  "1",      "1",      "AE",
  "1",      "2",      NA_character_,
  "1",      "3",      NA_character_,
  "2",      "1",      NA_character_,
  "3",      "1",      NA_character_
)

ec <- tribble(
  ~USUBJID, ~ECLNKID, ~ECADJ,
  "1",      "3",      "AE",
  "3",      "1",      NA_character_
)

fa <- tribble(
  ~USUBJID, ~FALNKID, ~FATESTCD, ~FAOBJ,            ~FASTRESC,
  "3",      "1",      "OCCUR",   "DOSE ADJUSTMENT", "Y"
)

derive_var_merged_ef_msrc(
  adex,
  by_vars = exprs(USUBJID, EXLNKID),
  flag_events = list(
    flag_event(
      dataset_name = "ex",
      condition = !is.na(EXADJ)
    ),
    flag_event(
      dataset_name = "ec",
      condition = !is.na(ECADJ),
      by_vars = exprs(USUBJID, EXLNKID = ECLNKID)
    ),
    flag_event(
      dataset_name = "fa",
      condition = FATESTCD == "OCCUR" & FAOBJ == "DOSE ADJUSTMENT" & FASTRESC == "Y",
      by_vars = exprs(USUBJID, EXLNKID = FALNKID)
    )
  ),
  source_datasets = list(ex = adex, ec = ec, fa = fa),
  new_var = DOSADJFL
)
}
\seealso{
\code{\link[=flag_event]{flag_event()}}

General Derivation Functions for all ADaMs that returns variable appended to dataset:
\code{\link{derive_var_extreme_flag}()},
\code{\link{derive_var_joined_exist_flag}()},
\code{\link{derive_var_merged_exist_flag}()},
\code{\link{derive_var_merged_summary}()},
\code{\link{derive_var_obs_number}()},
\code{\link{derive_var_relative_flag}()},
\code{\link{derive_vars_computed}()},
\code{\link{derive_vars_joined}()},
\code{\link{derive_vars_merged_lookup}()},
\code{\link{derive_vars_merged}()},
\code{\link{derive_vars_transposed}()}
}
\concept{der_gen}
\keyword{der_gen}
