% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scores.fluxnet.nc.R
\name{scores.fluxnet.nc}
\alias{scores.fluxnet.nc}
\title{Scores for FLUXNET reference data in NetCDF format}
\usage{
scores.fluxnet.nc(long.name, nc.mod, nc.ref, mod.id, ref.id, unit.conv.mod,
  unit.conv.ref, variable.unit, score.weights, rotate.me = TRUE,
  irregular = FALSE, my.projection = "+proj=longlat +ellps=WGS84",
  shp.filename = system.file("extdata/ne_110m_land/ne_110m_land.shp",
  package = "amber"), my.xlim = c(-180, 180), my.ylim = c(-60, 85),
  plot.width = 8, plot.height = 3.8, numCores = 2,
  outputDir = FALSE)
}
\arguments{
\item{long.name}{A string that gives the full name of the variable, e.g. 'Gross primary productivity'}

\item{nc.mod}{A string that gives the path and name of the netcdf file that contains the model output, e.g. '/home/model_gpp.nc'}

\item{nc.ref}{A string that gives the path and name of the netcdf file that contains the reference data output, e.g. '/home/reference_gpp.nc'}

\item{mod.id}{A string that identifies the source of the reference data set, e.g. 'CanESM2'}

\item{ref.id}{A string that identifies the source of the reference data set, e.g. 'MODIS'}

\item{unit.conv.mod}{A number that is used as a factor to convert the unit of the model data, e.g. 86400}

\item{unit.conv.ref}{A number that is used as a factor to convert the unit of the reference data, e.g. 86400}

\item{variable.unit}{A string that gives the final units using LaTeX notation, e.g. 'gC m$^{-2}$ day$^{-1}$'}

\item{score.weights}{R object that gives the weights of each score (\eqn{S_{bias}}, \eqn{S_{rmse}}, \eqn{S_{phase}}, \eqn{S_{iav}}, \eqn{S_{dist}})
that are used for computing the overall score, e.g. c(1,2,1,1,1)}

\item{rotate.me}{logical: TRUE if you want longitudes to range from -180 to
180 degrees and FALSE if you want longitudes to range from 0 to 360 degrees}

\item{irregular}{logical: TRUE if data is on an irregular grid and FALSE if
data is on a regular grid}

\item{my.projection}{A string that gives the projection of the irregular grid}

\item{shp.filename}{A string that gives the coastline shapefile}

\item{my.xlim}{An R object that gives the longitude range that you wish to
plot, e.g. c(-180, 180)}

\item{my.ylim}{An R object that gives the longitude range that you wish to
plot, e.g. c(-90, 90)}

\item{plot.width}{Number that gives the plot width, e.g. 8}

\item{plot.height}{Number that gives the plot height, e.g. 4}

\item{numCores}{An integer that defines the number of cores, e.g. 2}

\item{outputDir}{A string that gives the output directory, e.g. '/home/project/study'. The output will only be written if the user specifies an output directory.}
}
\value{
(1) Figures in PDF format that show global maps of
the model data at the location of FLUXNET sites
(mean, \eqn{mod.mean}; interannual-variability, \eqn{mod.iav}; month of annual cycle maximum, \eqn{mod.max.month}),
the reference data
(mean, \eqn{ref.mean}; interannual-variability, \eqn{ref.iav}; month of annual cycle maximum, \eqn{ref.max.month}),
statistical metrics
(bias, \eqn{bias}; root mean square error, \eqn{rmse}; time difference of the annual cycle maximum, \eqn{phase}),
and scores
(bias score, \eqn{bias.score}; root mean square error score, \eqn{rmse.score}; inter-annual variability score \eqn{iav.score}; annual cycle score (\eqn{phase.score}).

(2) Four text files: (i) score values and (ii) score inputs for each individual
site, and (iii) score values and (iv) score inputs averaged across sites.
when averaging over all station.
}
\description{
This function compares model output against
FLUXNET measurements in NetCDF format. The performance of a model is
expressed through scores that range from zero to one, where increasing values
imply better performance. These scores are computed in five steps:
\eqn{(i)} computation of a statistical metric,
\eqn{(ii)} nondimensionalization,
\eqn{(iii)} conversion to unit interval,
\eqn{(iv)} spatial integration, and
\eqn{(v)} averaging scores computed from different statistical metrics.
The latter includes the bias, root-mean-square error, phase shift,
inter-annual variability, and spatial distribution. The corresponding equations
are documented in \code{\link{amber-package}}.
}
\examples{

library(amber)
library(doParallel)
library(foreach)
library(ncdf4)
library(parallel)
library(raster)
long.name <- 'Gross primary productivity'
nc.mod <- system.file('extdata/modelRegular', 'gpp_monthly.nc', package = 'amber')
nc.ref <- system.file('extdata/referenceRegular', 'gpp_FLUXNET.nc', package = 'amber')
mod.id <- 'CLASSIC' # define a model experiment ID
ref.id <- 'FLUXNET' # give reference dataset a name
unit.conv.mod <- 86400*1000 # optional unit conversion for model data
unit.conv.ref <- 86400*1000 # optional unit conversion for reference data
variable.unit <- 'gC m$^{-2}$ day$^{-1}$' # unit after conversion (LaTeX notation)
score.weights <- c(1,2,1,1,1) # score weights of S_bias, S_rmse, S_phase, S_iav, S_dist
# global plot
scores.fluxnet.nc(long.name, nc.mod, nc.ref, mod.id, ref.id, unit.conv.mod,
 unit.conv.ref, variable.unit, score.weights)

\donttest{
# regional plot
scores.fluxnet.nc(long.name, nc.mod, nc.ref, mod.id, ref.id, unit.conv.mod,
unit.conv.ref, variable.unit, score.weights, my.xlim = c(-150, -60), my.ylim = c(20, 60),
plot.width = 6, plot.height = 3.8)
scores.fluxnet.nc(long.name, nc.mod, nc.ref, mod.id, ref.id, unit.conv.mod,
unit.conv.ref, variable.unit, score.weights)

# (2) Example for data on a rotated grid
nc.mod <- system.file('extdata/modelRotated', 'gpp_monthly.nc', package = 'amber')
nc.ref <- system.file('extdata/referenceRegular', 'gpp_FLUXNET.nc', package = 'amber')
mod.id <- 'CLASSIC' # define a model experiment ID
ref.id <- 'FLUXNET' # give reference dataset a name
unit.conv.mod <- 86400*1000 # optional unit conversion for model data
unit.conv.ref <- 86400*1000 # optional unit conversion for reference data
variable.unit <- 'gC m$^{-2}$ day$^{-1}$' # unit after conversion (LaTeX notation)
score.weights <- c(1,2,1,1,1) # score weights of S_bias, S_rmse, S_phase, S_iav, S_dist
rotate.me <- FALSE
irregular <- TRUE
my.projection <-'+proj=ob_tran +o_proj=longlat +o_lon_p=83. +o_lat_p=42.5 +lon_0=263.'
shp.filename <- system.file('extdata/ne_50m_admin_0_countries/ne_50m_admin_0_countries.shp',
 package = 'amber')
my.xlim <- c(-171, 0) # longitude range that you wish to plot
my.ylim <- c(32, 78) # latitude range that you wish to plot
plot.width <- 7 # plot width
plot.height <- 3.8 # plot height
numCores = 2

scores.fluxnet.nc(long.name, nc.mod, nc.ref, mod.id, ref.id, unit.conv.mod,
unit.conv.ref, variable.unit, score.weights, rotate.me, irregular,
my.projection,
shp.filename, my.xlim, my.ylim)
}
}
