\name{ROSETTA.centroids}
\alias{ROSETTA.centroids}
\docType{data}

\title{Average Hydraulic Parameters from the ROSETTA Model by USDA Soil Texture Class}

\description{Average soil hydraulic parameters generated by the first stage predictions of the ROSETTA model by USDA soil texture class. These data were extracted from \href{https://www.ars.usda.gov/pacific-west-area/riverside-ca/agricultural-water-efficiency-and-salinity-research-unit/docs/model/rosetta-class-average-hydraulic-parameters/}{ROSETTA documentation} and re-formated for ease of use.

Theoretical water-retention parameters for uniform soil material of each texture class have been estimated via van Genuchten model.
}

\usage{data("ROSETTA.centroids")}

\format{
  A data frame:
  \describe{
    
    \item{\code{texture}}{soil texture class, ordered from low to high available water holding capacity}
    \item{\code{theta_r}}{average saturated water content}
    \item{\code{theta_s}}{average residual water content}
    \item{\code{alpha}}{average value, related to the inverse of the air entry suction, log10-tranformed values with units of cm}
    \item{\code{npar}}{average value, index of pore size distribution, log10-tranformed values with units of 1/cm}
    
    \item{\code{theta_r_sd}}{1 standard deviation of \code{theta_r}}
    \item{\code{theta_s_sd}}{1 standard deviation of \code{theta_s}}
    \item{\code{alpha_sd}}{1 standard deviation of \code{alpha}}
    \item{\code{npar_sd}}{1 standard deviation of \code{npar}}
    
    \item{\code{sat}}{approximate volumetric water content at which soil material is saturated}
    \item{\code{fc}}{approximate volumetric water content at which matrix potential = -33kPa}
    \item{\code{pwp}}{approximate volumetric water content at which matrix potential = -1500kPa}
    \item{\code{awc}}{approximate available water holding capacity: VWC(-33kPa) - VWC(-1500kPa)}
  }
}

\details{See \href{https://ncss-tech.github.io/AQP/soilDB/fetchKSSL-VG-demo.html}{the related tutorial}.}

\references{
van Genuchten, M.Th. (1980). "A closed-form equation for predicting the hydraulic conductivity of unsaturated soils". Soil Science Society of America Journal. 44 (5): 892-898. 

Schaap, M.G., F.J. Leij, and M.Th. van Genuchten. 2001. ROSETTA: A computer program for estimating soil hydraulic parameters with hierarchical pedotransfer functions. Journal of Hydrology 251(3–4): 163-176.
}

\source{
\href{https://www.ars.usda.gov/pacific-west-area/riverside-ca/agricultural-water-efficiency-and-salinity-research-unit/docs/model/rosetta-class-average-hydraulic-parameters/}{ROSETTA documentation}
}

\examples{

\dontrun{

library(aqp)
library(soilDB)
library(latticeExtra)

data("ROSETTA.centroids")

# iterate over horizons and generate VG model curve
res <- lapply(1:nrow(ROSETTA.centroids), function(i) {
  m <- KSSL_VG_model(VG_params = ROSETTA.centroids[i, ], phi_min = 10^-3, phi_max=10^6)$VG_curve
  # copy generalized hz label
  m$hz <- ROSETTA.centroids$hz[i]
  # copy ID
  m$texture_class <- ROSETTA.centroids$texture[i]
  return(m)
})

# copy over lab sample number as ID
res <- do.call('rbind', res)

# check: OK
str(res)


# visual check: OK
xyplot(
  phi ~ theta | texture_class, data=res, 
  type=c('l', 'g'), 
  scales=list(alternating=3, x=list(tick.number=10), y=list(log=10, tick.number=10)), 
  yscale.components=yscale.components.logpower, 
  ylab=expression(Suction~~(kPa)), 
  xlab=expression(Volumetric~Water~Content~~(cm^3/cm^3)), 
  par.settings = list(superpose.line=list(col='RoyalBlue', lwd=2)), 
  strip=strip.custom(bg=grey(0.85)), 
  as.table=TRUE
)


}

}
\keyword{datasets}
