% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sa_pairing_generalized.R
\name{sa_pairing_generalized}
\alias{sa_pairing_generalized}
\title{Automatic Item Pairing Method in Forced-Choice Test Construction}
\usage{
sa_pairing_generalized(block, total_items, Temperature,
                              eta_Temperature = 0.01, r = 0.999,
                              end_criteria = 10^(-6),
                              item_chars, weights, FUN,
                              n_exchange = 2, prob_newitem = 0.25,
                              use_IIA = FALSE, rater_chars,
                              iia_weights = c(BPlin = 1, BPquad = 1,
                              AClin = 1, ACquad = 1))
}
\arguments{
\item{block}{An \emph{n} by \emph{k} integer matrix,
where \emph{n} is the number of item blocks
and \emph{k} is the number of items per block.

Serves as the initial starting blocks for the automatic pairing method.}

\item{total_items}{Integer value. How many items do we sample from
in order to build this \code{block}? Should be more than number of unique values
in \code{block}.}

\item{Temperature}{Initial temperature value. Can be left blank and be computed based on
the absolute value of initial energy of \code{block} (Recommended), and scaled
by \code{eta_Temperature}.

In general, higher temperature represents a higher probability of
accepting an inferior solution.}

\item{eta_Temperature}{A positive numeric value. The ratio of initial temperature to
initial energy of \code{block}, if \code{Temperature} is not designated.}

\item{r}{A positive numeric value less than 1.
Determines the reduction rate of \code{Temperature} after each iteration.}

\item{end_criteria}{A positive numeric value less than 1.
Iteration stops when temperature drops to below \code{end_criteria * Temperature}.
Default to be \eqn{10^-6}.}

\item{item_chars}{An \emph{m} by \emph{r} data frame,
where \emph{m} is the total number of items to sample from,
whether it is included in the block or not,
whereas \emph{r} is the number of item characteristics.}

\item{weights}{A vector of length \emph{r} with weights for each
item characteristics in \code{item_chars}.
Should provide a weight of 0 for specific characteristics
not of interest, such as item ID.}

\item{FUN}{A vector of customized function names for optimizing each
item characteristic within each block, with length \emph{r}.}

\item{n_exchange}{Integer value. Determines how many blocks are exchanged
in order to produce a new solution for each iteration.
Should be a value larger than 1 and less than \code{nrow(block)}.}

\item{prob_newitem}{A value between \emph{0} and \emph{1}.
Probability of choosing the strategy of picking a new item,
when not all candidate items are used to build the FC scale.}

\item{use_IIA}{Logical. Are IIA metrics used when performing automatic pairing?}

\item{rater_chars}{A \emph{p} by \emph{m} numeric matrix with scores of each of the
\emph{p} participants for the \emph{m} items. Ignored when \code{use_IIA == FALSE}.}

\item{iia_weights}{A vector of length 4 indicating weights given to each IIA metric:

Linearly weighted AC (Gwet, 2008; 2014);

Quadratic weighted AC;

Linearly weighted Brennan-Prediger (BP) Index(Brennan & Prediger, 1981; Gwet, 2014);

Quadratic weighted BP.}
}
\value{
A list containing:
\itemize{
  \code{block_initial} Initial starting block

  \code{energy_initial} Initial energy for \code{block_initial}

  \code{block_final} Final paired block after optimization by SA

  \code{energy_final} Final energy for \code{block_final}
}
}
\description{
Automatic construction of forced-choice tests based on
Simulated Annealing algorithm. Allows items to be:

1. Matched in either pairs, triplets, quadruplets or blocks of any size;

2. Matched based on any number of item-level characteristics
(e.g. Social desirability, factor) based on any customized criteria;

3. Matched based on person-level inter-item agreement (IIA) metrics.
}
\note{
The essence of SA is the probablistic acceptance of solutions inferior to
the current state, which avoids getting stuck in local maxima/minima.
It is also recommended to try out different values of
\code{weights, iia_weights, eta_Temperature} to find out the best
combination of initial temperature and energy value
in order to provide optimally paired blocks.

Use \code{cal_block_energy_with_iia} if inter-item agreement
(IIA) metrics are needed.
}
\examples{
## Simulate 60 items loading on different Big Five dimensions,
## with different mean and item difficulty

item_dims <- sample(c("Openness","Conscientiousness","Neuroticism",
                     "Extraversion","Agreeableness"), 60, replace = TRUE)
item_mean <- rnorm(60, 5, 2)
item_difficulty <- runif(60, -1, 1)


item_df <- data.frame(Dimensions = item_dims,
                     Mean = item_mean, Difficulty = item_difficulty)
solution <- make_random_block(60, 60, 3)


item_responses <- matrix(sample(seq(1:5), 600*60, replace = TRUE), nrow = 60, byrow = TRUE)

## Automatic pairing, without use of IIAs
## See ?facfun for information about what it does
\donttest{
sa_pairing_generalized(solution, 60, eta_Temperature = 0.01,
                                   r = 0.999, end_criteria = 0.001,
                                   weights = c(1,1,1),
                                   item_chars = item_df,
                                   FUN = c("facfun", "var", "var"))
}

## Automatic pairing, with IIAs
\donttest{
sa_pairing_generalized(solution, 60, eta_Temperature = 0.01,
                                   r = 0.999, end_criteria = 0.001,
                                   weights = c(1,1,1),
                                   item_chars = item_df,
                                   FUN = c("facfun", "var", "var"),
                                   use_IIA = TRUE,
                                   rater_chars = item_responses,
                                   iia_weights = c(BPlin = 1, BPquad = 1,
                                   AClin = 1, ACquad = 1))
}


}
\references{
Brennan, R. L., & Prediger, D. J. (1981). Coefficient kappa: Some uses, misuses,
and alternatives. \emph{Educational and Psychological Measurement, 41}(3),
687-699. https://doi.org/10.1177/001316448104100307

Gwet, K. L. (2008). Computing inter rater reliability and its
variance in the presence of high agreement.
\emph{British Journal of Mathematical and Statistical Psychology, 61}(1),
29-48. https://doi.org/10.1348/000711006X126600

Gwet, K. L. (2014). \emph{Handbook of inter-rater reliability (4th ed.):
The definitive guide to measuring the extent of agreement among raters}.
Gaithersburg, MD: Advanced Analytics Press.
}
\author{
Mengtong Li
}
