% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reconc_MH.R
\name{reconc_MCMC}
\alias{reconc_MCMC}
\title{MCMC for Probabilistic Reconciliation of forecasts via conditioning}
\usage{
reconc_MCMC(
  S,
  base_forecasts,
  distr,
  num_samples = 10000,
  tuning_int = 100,
  init_scale = 1,
  burn_in = 1000,
  seed = NULL
)
}
\arguments{
\item{S}{summing matrix (n x n_bottom).}

\item{base_forecasts}{list of the parameters of the base forecast distributions, see details.}

\item{distr}{a string describing the type of predictive distribution.}

\item{num_samples}{number of samples to draw using MCMC.}

\item{tuning_int}{number of iterations between scale updates of the proposal.}

\item{init_scale}{initial scale of the proposal.}

\item{burn_in}{number of initial samples to be discarded.}

\item{seed}{seed for reproducibility.}
}
\value{
A list containing the reconciled forecasts. The list has the following named elements:
\itemize{
\item \code{bottom_reconciled_samples}: a matrix (n_bottom x \code{num_samples}) containing reconciled samples for the bottom time series;
\item \code{upper_reconciled_samples}: a matrix (n_upper x \code{num_samples}) containing reconciled samples for the upper time series;
\item \code{reconciled_samples}: a matrix (n x \code{num_samples}) containing the reconciled samples for all time series.
}
}
\description{
Uses Markov Chain Monte Carlo algorithm to draw samples from the reconciled
forecast distribution, which is obtained via conditioning.

This is a bare-bones implementation of the Metropolis-Hastings algorithm, we suggest the usage of tools to check the convergence.
The function only works with Poisson or Negative Binomial base forecasts.

The function \code{\link[=reconc_BUIS]{reconc_BUIS()}} is generally faster on most hierarchies.
}
\details{
The parameter \code{base_forecast} is a list containing n elements.
Each element is a vector containing the estimated:
\itemize{
\item mean and sd for the Gaussian base forecast, see \link[stats]{Normal}, if \code{distr}='gaussian';
\item lambda for the Poisson base forecast, see \link[stats]{Poisson}, if \code{distr}='poisson';
\item size and probability of success for the negative binomial base forecast, see \link[stats]{NegBinomial}, if \code{distr}='nbinom'.
}

The order of the \code{base_forecast} list is given by the order of the time series in the summing matrix.
}
\examples{

library(bayesRecon)

# Create a minimal hierarchy with 2 bottom and 1 upper variable
rec_mat <- get_reconc_matrices(agg_levels=c(1,2), h=2)
S <- rec_mat$S

#Set the parameters of the Poisson base forecast distributions
lambda1 <- 2
lambda2 <- 4
lambdaY <- 9
lambdas <- c(lambdaY,lambda1,lambda2)

base_forecasts = list()
for (i in 1:nrow(S)) {
 base_forecasts[[i]] = lambdas[i]
}

#Sample from the reconciled forecast distribution using MCMC
mcmc = reconc_MCMC(S,base_forecasts=lambdas,distr="poisson",
                  num_samples=30000, seed=42)
samples_mcmc <- mcmc$reconciled_samples

#Compare the reconciled means with those obtained via BUIS
buis = reconc_BUIS(S, base_forecasts, in_type="params",
                   distr="poisson", num_samples=100000, seed=42)
samples_buis <- buis$reconciled_samples

print(rowMeans(samples_mcmc))
print(rowMeans(samples_buis))

}
\references{
Corani, G., Rubattu, N., Azzimonti, D., Antonucci, A. (2022). \emph{Probabilistic reconciliation of count time series}. \doi{10.48550/arXiv.2207.09322}.
}
\seealso{
\code{\link[=reconc_BUIS]{reconc_BUIS()}}
}
