% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/auxiliary_filter.R
\name{auxiliary_filter}
\alias{auxiliary_filter}
\title{Auxiliary Particle Filter (APF)}
\usage{
auxiliary_filter(
  y,
  num_particles,
  init_fn,
  transition_fn,
  log_likelihood_fn,
  aux_log_likelihood_fn,
  obs_times = NULL,
  resample_algorithm = c("SISAR", "SISR", "SIS"),
  resample_fn = c("stratified", "systematic", "multinomial"),
  threshold = NULL,
  return_particles = TRUE,
  ...
)
}
\arguments{
\item{y}{A numeric vector or matrix of observations. Each row represents an
observation at a time step. If observations are not equally spaced, use the
\code{obs_times} argument.}

\item{num_particles}{A positive integer specifying the number of particles.}

\item{init_fn}{A function to initialize the particles. Should take
`num_particles` and return a matrix or vector of initial states. Additional
model parameters can be passed via \code{...}.}

\item{transition_fn}{A function for propagating particles. Should take
`particles` and optionally `t`. Additional model parameters via \code{...}.}

\item{log_likelihood_fn}{A function that returns the log-likelihood for each
particle given the current observation, particles, and optionally `t`.
Additional parameters via \code{...}.}

\item{aux_log_likelihood_fn}{A function that computes the log-likelihood of
the next observation given the current particles. It should accept
arguments `y`, `particles`, optionally `t`, and any additional model-specific
parameters via \code{...}. It returns a numeric vector of log-likelihoods.}

\item{obs_times}{A numeric vector specifying observation time points. Must
match the number of rows in \code{y}, or defaults to \code{1:nrow(y)}.}

\item{resample_algorithm}{A character string specifying the filtering
resample algorithm:
\code{"SIS"} for no resampling, \code{"SISR"} for resampling at every time
step, or \code{"SISAR"} for adaptive resampling when ESS
drops below \code{threshold}. Using \code{"SISR"} or \code{"SISAR"} to
avoid weight degeneracy is recommedended. Default is \code{"SISAR"}.}

\item{resample_fn}{A string indicating the resampling method:
\code{"stratified"}, \code{"systematic"}, or \code{"multinomial"}.
Default is \code{"stratified"}.}

\item{threshold}{A numeric value specifying the ESS threshold for
\code{"SISAR"}. Defaults to \code{num_particles / 2}.}

\item{return_particles}{Logical; if \code{TRUE}, returns the full particle
and weight histories.}

\item{...}{Additional arguments passed to \code{init_fn},
\code{transition_fn}, and \code{log_likelihood_fn}.}
}
\value{
A list with components:
\describe{
  \item{state_est}{Estimated states over time (weighted mean of particles).}
  \item{ess}{Effective sample size at each time step.}
  \item{loglike}{Total log-likelihood.}
  \item{loglike_history}{Log-likelihood at each time step.}
  \item{algorithm}{The filtering algorithm used.}
  \item{particles_history}{Matrix of particle states over time
  (if \code{return_particles = TRUE}).}
  \item{weights_history}{Matrix of particle weights over time
  (if \code{return_particles = TRUE}).}
}
}
\description{
The Auxiliary Particle Filter differs from the bootstrap filter by
incorporating a look-ahead step: particles are reweighted using an
approximation of the likelihood of the next observation prior to resampling.
This adjustment can help reduce particle degeneracy and,
improve filtering efficiency compared to the bootstrap approach.
}
\section{The Auxiliary Particle Filter (APF)}{


The Auxiliary Particle Filter (APF) was introduced by Pitt and Shephard
(1999) to improve upon the standard bootstrap filter by incorporating a
look ahead step. Before resampling at time \eqn{t}, particles are weighted by
an auxiliary weight proportional to an estimate of the likelihood of the next
observation, guiding resampling to favor particles likely to contribute to
future predictions.

Specifically, if \eqn{w_{t-1}^i} are the normalized weights and
\eqn{x_{t-1}^i} are the particles at time \eqn{t-1}, then auxiliary weights
are computed as
\deqn{
  \tilde{w}_t^i \propto w_{t-1}^i \, p(y_t | \mu_t^i),
}
where \eqn{\mu_t^i} is a predictive summary (e.g., the expected next state)
of the particle \eqn{x_{t-1}^i}. Resampling is performed using
\eqn{\tilde{w}_t^i} instead of \eqn{w_{t-1}^i}.
This can reduce the variance of the importance weights at time \eqn{t} and
help mitigate particle degeneracy, especially if the auxiliary weights are
chosen well.

Default resampling method is stratified resampling, which has lower variance
than multinomial resampling (Douc et al., 2005).
}

\section{Model Specification}{

Particle filter implementations in this package assume a discrete-time
state-space model defined by:

\itemize{
  \item A sequence of latent states \eqn{x_0, x_1, \ldots, x_T} evolving
  according to a Markov process.
  \item Observations \eqn{y_1, \ldots, y_T} that are conditionally independent
  given the corresponding latent states.
}

The model is specified as:
\deqn{x_0 \sim \mu_\theta}
\deqn{x_t \sim f_\theta(x_t \mid x_{t-1}), \quad t = 1, \ldots, T}
\deqn{y_t \sim g_\theta(y_t \mid x_t), \quad t = 1, \ldots, T}

where \eqn{\theta} denotes model parameters passed via \code{...}.

The user provides the following functions:
\itemize{
  \item \code{init_fn}: draws from the initial distribution
  \eqn{\mu_\theta}.
  \item \code{transition_fn}: generates or evaluates the transition
  density \eqn{f_\theta}.
  \item \code{weight_fn}: evaluates the observation likelihood
  \eqn{g_\theta}.
}
}

\examples{
init_fn <- function(num_particles) rnorm(num_particles, 0, 1)
transition_fn <- function(particles) particles + rnorm(length(particles))
log_likelihood_fn <- function(y, particles) {
  dnorm(y, mean = particles, sd = 1, log = TRUE)
}
aux_log_likelihood_fn <- function(y, particles) {
  # Predict next state (mean stays same) and compute log p(y | x)
  mean_forecast <- particles # since E[x'] = x in this model
  dnorm(y, mean = mean_forecast, sd = 1, log = TRUE)
}

y <- cumsum(rnorm(50)) # dummy data
num_particles <- 100

result <- auxiliary_filter(
  y = y,
  num_particles = num_particles,
  init_fn = init_fn,
  transition_fn = transition_fn,
  log_likelihood_fn = log_likelihood_fn,
  aux_log_likelihood_fn = aux_log_likelihood_fn
)
plot(result$state_est,
  type = "l", col = "blue", main = "APF: State Estimates",
  ylim = range(c(result$state_est, y))
)
points(y, col = "red", pch = 20)

# ---- With parameters ----
init_fn <- function(num_particles) rnorm(num_particles, 0, 1)
transition_fn <- function(particles, mu) {
  particles + rnorm(length(particles), mean = mu)
}
log_likelihood_fn <- function(y, particles, sigma) {
  dnorm(y, mean = particles, sd = sigma, log = TRUE)
}
aux_log_likelihood_fn <- function(y, particles, mu, sigma) {
  # Forecast mean of x' given x, then evaluate p(y | forecast)
  forecast <- particles + mu
  dnorm(y, mean = forecast, sd = sigma, log = TRUE)
}

y <- cumsum(rnorm(50)) # dummy data
num_particles <- 100

result <- auxiliary_filter(
  y = y,
  num_particles = num_particles,
  init_fn = init_fn,
  transition_fn = transition_fn,
  log_likelihood_fn = log_likelihood_fn,
  aux_log_likelihood_fn = aux_log_likelihood_fn,
  mu = 1,
  sigma = 1
)
plot(result$state_est,
  type = "l", col = "blue", main = "APF with Parameters",
  ylim = range(c(result$state_est, y))
)
points(y, col = "red", pch = 20)

# ---- With observation gaps ----
simulate_ssm <- function(num_steps, mu, sigma) {
  x <- numeric(num_steps)
  y <- numeric(num_steps)
  x[1] <- rnorm(1, mean = 0, sd = sigma)
  y[1] <- rnorm(1, mean = x[1], sd = sigma)
  for (t in 2:num_steps) {
    x[t] <- mu * x[t - 1] + sin(x[t - 1]) + rnorm(1, mean = 0, sd = sigma)
    y[t] <- x[t] + rnorm(1, mean = 0, sd = sigma)
  }
  y
}

data <- simulate_ssm(10, mu = 1, sigma = 1)
obs_times <- c(1, 2, 3, 5, 6, 7, 8, 9, 10) # Missing at t = 4
data_obs <- data[obs_times]

init_fn <- function(num_particles) rnorm(num_particles, 0, 1)
transition_fn <- function(particles, mu) {
  particles + rnorm(length(particles), mean = mu)
}
log_likelihood_fn <- function(y, particles, sigma) {
  dnorm(y, mean = particles, sd = sigma, log = TRUE)
}
aux_log_likelihood_fn <- function(y, particles, mu, sigma) {
  forecast <- particles + mu
  dnorm(y, mean = forecast, sd = sigma, log = TRUE)
}

num_particles <- 100
result <- auxiliary_filter(
  y = data_obs,
  num_particles = num_particles,
  init_fn = init_fn,
  transition_fn = transition_fn,
  log_likelihood_fn = log_likelihood_fn,
  aux_log_likelihood_fn = aux_log_likelihood_fn,
  obs_times = obs_times,
  mu = 1,
  sigma = 1
)
plot(result$state_est,
  type = "l", col = "blue", main = "APF with Observation Gaps",
  ylim = range(c(result$state_est, data))
)
points(data_obs, col = "red", pch = 20)
}
\references{
Pitt, M. K., & Shephard, N. (1999). Filtering via simulation: Auxiliary
particle filters. Journal of the American Statistical Association, 94(446),
590–599. \doi{doi:10.1080/01621459.1999.10474153}

Douc, R., Cappé, O., & Moulines, E. (2005). Comparison of
Resampling Schemes for Particle Filtering.
Accessible at: \url{https://arxiv.org/abs/cs/0507025}
}
