% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/equivalence_test.R
\name{equivalence_test}
\alias{equivalence_test}
\alias{equivalence_test.default}
\alias{equivalence_test.data.frame}
\alias{equivalence_test.brmsfit}
\title{Test for Practical Equivalence}
\usage{
equivalence_test(x, ...)

\method{equivalence_test}{default}(x, ...)

\method{equivalence_test}{data.frame}(
  x,
  range = "default",
  ci = 0.95,
  rvar_col = NULL,
  verbose = TRUE,
  ...
)

\method{equivalence_test}{brmsfit}(
  x,
  range = "default",
  ci = 0.95,
  effects = "fixed",
  component = "conditional",
  parameters = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{x}{Vector representing a posterior distribution. Can also be a
\code{stanreg} or \code{brmsfit} model.}

\item{...}{Currently not used.}

\item{range}{ROPE's lower and higher bounds. Should be \code{"default"} or
depending on the number of outcome variables a vector or a list. For models
with one response, \code{range} can be:
\itemize{
\item a vector of length two (e.g., \code{c(-0.1, 0.1)}),
\item a list of numeric vector of the same length as numbers of parameters (see
'Examples').
\item a list of \emph{named} numeric vectors, where names correspond to parameter
names. In this case, all parameters that have no matching name in \code{range}
will be set to \code{"default"}.
}

In multivariate models, \code{range} should be a list with another list (one for
each response variable) of numeric vectors . Vector names should correspond to
the name of the response variables. If \code{"default"} and input is a vector, the
range is set to \code{c(-0.1, 0.1)}. If \code{"default"} and input is a Bayesian model,
\code{\link[=rope_range]{rope_range()}} is used. See 'Examples'.}

\item{ci}{The Credible Interval (CI) probability, corresponding to the
proportion of HDI, to use for the percentage in ROPE.}

\item{rvar_col}{A single character - the name of an \code{rvar} column in the data
frame to be processed. See example in \code{\link[=p_direction]{p_direction()}}.}

\item{verbose}{Toggle off warnings.}

\item{effects}{Should variables for fixed effects (\code{"fixed"}), random effects
(\code{"random"}) or both (\code{"all"}) be returned? Only applies to mixed models. May
be abbreviated.

For models of from packages \strong{brms} or \strong{rstanarm} there are additional
options:
\itemize{
\item \code{"fixed"} returns fixed effects.
\item \code{"random_variance"} return random effects parameters (variance and
correlation components, e.g. those parameters that start with \code{sd_} or
\code{cor_}).
\item \code{"grouplevel"} returns random effects group level estimates, i.e. those
parameters that start with \code{r_}.
\item \code{"random"} returns both \code{"random_variance"} and \code{"grouplevel"}.
\item \code{"all"} returns fixed effects and random effects variances.
\item \code{"full"} returns all parameters.
}}

\item{component}{Which type of parameters to return, such as parameters for
the conditional model, the zero-inflated part of the model, the dispersion
term, etc. See details in section \emph{Model Components}. May be abbreviated.
Note that the \emph{conditional} component also refers to the \emph{count} or \emph{mean}
component - names may differ, depending on the modeling package. There are
three convenient shortcuts (not applicable to \emph{all} model classes):
\itemize{
\item \code{component = "all"} returns all possible parameters.
\item If \code{component = "location"}, location parameters such as \code{conditional},
\code{zero_inflated}, \code{smooth_terms}, or \code{instruments} are returned (everything
that are fixed or random effects - depending on the \code{effects} argument -
but no auxiliary parameters).
\item For \code{component = "distributional"} (or \code{"auxiliary"}), components like
\code{sigma}, \code{dispersion}, \code{beta} or \code{precision} (and other auxiliary
parameters) are returned.
}}

\item{parameters}{Regular expression pattern that describes the parameters
that should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}
}
\value{
A data frame with following columns:
\itemize{
\item \code{Parameter} The model parameter(s), if \code{x} is a model-object. If \code{x} is a vector, this column is missing.
\item \code{CI} The probability of the HDI.
\item \code{ROPE_low}, \code{ROPE_high} The limits of the ROPE. These values are identical for all parameters.
\item \code{ROPE_Percentage} The proportion of the HDI that lies inside the ROPE.
\item \code{ROPE_Equivalence} The "test result", as character. Either "rejected", "accepted" or "undecided".
\item \code{HDI_low} , \code{HDI_high} The lower and upper HDI limits for the parameters.
}
}
\description{
Perform a \strong{Test for Practical Equivalence} for Bayesian and frequentist models.
}
\details{
Documentation is accessible for:
\itemize{
\item \href{https://easystats.github.io/bayestestR/reference/equivalence_test.html}{Bayesian models}
\item \href{https://easystats.github.io/parameters/reference/equivalence_test.lm.html}{Frequentist models}
}

For Bayesian models, the \strong{Test for Practical Equivalence} is based on the
\emph{"HDI+ROPE decision rule"} (\cite{Kruschke, 2014, 2018}) to check whether
parameter values should be accepted or rejected against an explicitly
formulated "null hypothesis" (i.e., a ROPE). In other words, it checks the
percentage of the \verb{89\%} \link[=hdi]{HDI} that is the null region (the ROPE). If
this percentage is sufficiently low, the null hypothesis is rejected. If this
percentage is sufficiently high, the null hypothesis is accepted.

Using the \link[=rope]{ROPE} and the \link[=hdi]{HDI}, \cite{Kruschke (2018)}
suggests using the percentage of the \verb{95\%} (or \verb{89\%}, considered more stable)
HDI that falls within the ROPE as a decision rule. If the HDI
is completely outside the ROPE, the "null hypothesis" for this parameter is
"rejected". If the ROPE completely covers the HDI, i.e., all most credible
values of a parameter are inside the region of practical equivalence, the
null hypothesis is accepted. Else, it is undecided whether to accept or
reject the null hypothesis. If the full ROPE is used (i.e., \verb{100\%} of the
HDI), then the null hypothesis is rejected or accepted if the percentage
of the posterior within the ROPE is smaller than to \verb{2.5\%} or greater than
\verb{97.5\%}. Desirable results are low proportions inside the ROPE  (the closer
to zero the better).

Some attention is required for finding suitable values for the ROPE limits
(argument \code{range}). See 'Details' in \code{\link[=rope_range]{rope_range()}} for further
information.

\strong{Multicollinearity: Non-independent covariates}

When parameters show strong correlations, i.e. when covariates are not
independent, the joint parameter distributions may shift towards or
away from the ROPE. In such cases, the test for practical equivalence may
have inappropriate results. Collinearity invalidates ROPE and hypothesis
testing based on univariate marginals, as the probabilities are conditional
on independence. Most problematic are the results of the "undecided"
parameters, which may either move further towards "rejection" or away
from it (\cite{Kruschke 2014, 340f}).

\code{equivalence_test()} performs a simple check for pairwise correlations
between parameters, but as there can be collinearity between more than two variables,
a first step to check the assumptions of this hypothesis testing is to look
at different pair plots. An even more sophisticated check is the projection
predictive variable selection (\cite{Piironen and Vehtari 2017}).
}
\note{
There is a \code{print()}-method with a \code{digits}-argument to control
the amount of digits in the output, and there is a
\href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method}
to visualize the results from the equivalence-test (for models only).
}
\section{Model components}{


Possible values for the \code{component} argument depend on the model class.
Following are valid options:
\itemize{
\item \code{"all"}: returns all model components, applies to all models, but will only
have an effect for models with more than just the conditional model
component.
\item \code{"conditional"}: only returns the conditional component, i.e. "fixed
effects" terms from the model. Will only have an effect for models with
more than just the conditional model component.
\item \code{"smooth_terms"}: returns smooth terms, only applies to GAMs (or similar
models that may contain smooth terms).
\item \code{"zero_inflated"} (or \code{"zi"}): returns the zero-inflation component.
\item \code{"location"}: returns location parameters such as \code{conditional},
\code{zero_inflated}, or \code{smooth_terms} (everything that are fixed or random
effects - depending on the \code{effects} argument - but no auxiliary
parameters).
\item \code{"distributional"} (or \code{"auxiliary"}): components like \code{sigma},
\code{dispersion}, \code{beta} or \code{precision} (and other auxiliary parameters) are
returned.
}

For models of class \code{brmsfit} (package \strong{brms}), even more options are
possible for the \code{component} argument, which are not all documented in detail
here. See also \href{https://easystats.github.io/insight/reference/find_parameters.BGGM.html}{\code{?insight::find_parameters}}.
}

\examples{
\dontshow{if (all(insight::check_if_installed(c("rstanarm", "brms", "emmeans", "BayesFactor", "see"), quietly = TRUE))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(bayestestR)

equivalence_test(x = rnorm(1000, 0, 0.01), range = c(-0.1, 0.1))
equivalence_test(x = rnorm(1000, 0, 1), range = c(-0.1, 0.1))
equivalence_test(x = rnorm(1000, 1, 0.01), range = c(-0.1, 0.1))
equivalence_test(x = rnorm(1000, 1, 1), ci = c(.50, .99))

# print more digits
test <- equivalence_test(x = rnorm(1000, 1, 1), ci = c(.50, .99))
print(test, digits = 4)
\donttest{
model <- rstanarm::stan_glm(mpg ~ wt + cyl, data = mtcars)
equivalence_test(model)
# multiple ROPE ranges - asymmetric, symmetric, default
equivalence_test(model, range = list(c(10, 40), c(-5, -4), "default"))
# named ROPE ranges
equivalence_test(model, range = list(wt = c(-5, -4), `(Intercept)` = c(10, 40)))

# plot result
test <- equivalence_test(model)
plot(test)

equivalence_test(emmeans::emtrends(model, ~1, "wt", data = mtcars))

model <- brms::brm(mpg ~ wt + cyl, data = mtcars)
equivalence_test(model)

bf <- BayesFactor::ttestBF(x = rnorm(100, 1, 1))
# equivalence_test(bf)
}
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Kruschke, J. K. (2018). Rejecting or accepting parameter values in Bayesian estimation. Advances in Methods and Practices in Psychological Science, 1(2), 270-280. \doi{10.1177/2515245918771304}
\item Kruschke, J. K. (2014). Doing Bayesian data analysis: A tutorial with R, JAGS, and Stan. Academic Press
\item Piironen, J., & Vehtari, A. (2017). Comparison of Bayesian predictive methods for model selection. Statistics and Computing, 27(3), 711–735. \doi{10.1007/s11222-016-9649-y}
}
}
