% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rope.R
\name{rope}
\alias{rope}
\alias{rope.numeric}
\alias{rope.data.frame}
\alias{rope.brmsfit}
\title{Region of Practical Equivalence (ROPE)}
\usage{
rope(x, ...)

\method{rope}{numeric}(x, range = "default", ci = 0.95, ci_method = "ETI", verbose = TRUE, ...)

\method{rope}{data.frame}(
  x,
  range = "default",
  ci = 0.95,
  ci_method = "ETI",
  rvar_col = NULL,
  verbose = TRUE,
  ...
)

\method{rope}{brmsfit}(
  x,
  range = "default",
  ci = 0.95,
  ci_method = "ETI",
  effects = "fixed",
  component = "conditional",
  parameters = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{x}{Vector representing a posterior distribution. Can also be a
\code{stanreg} or \code{brmsfit} model.}

\item{...}{Currently not used.}

\item{range}{ROPE's lower and higher bounds. Should be \code{"default"} or
depending on the number of outcome variables a vector or a list. For models
with one response, \code{range} can be:
\itemize{
\item a vector of length two (e.g., \code{c(-0.1, 0.1)}),
\item a list of numeric vector of the same length as numbers of parameters (see
'Examples').
\item a list of \emph{named} numeric vectors, where names correspond to parameter
names. In this case, all parameters that have no matching name in \code{range}
will be set to \code{"default"}.
}

In multivariate models, \code{range} should be a list with another list (one for
each response variable) of numeric vectors . Vector names should correspond to
the name of the response variables. If \code{"default"} and input is a vector, the
range is set to \code{c(-0.1, 0.1)}. If \code{"default"} and input is a Bayesian model,
\code{\link[=rope_range]{rope_range()}} is used. See 'Examples'.}

\item{ci}{The Credible Interval (CI) probability, corresponding to the
proportion of HDI, to use for the percentage in ROPE.}

\item{ci_method}{The type of interval to use to quantify the percentage in
ROPE. Can be 'HDI' (default) or 'ETI'. See \code{\link[=ci]{ci()}}.}

\item{verbose}{Toggle off warnings.}

\item{rvar_col}{A single character - the name of an \code{rvar} column in the data
frame to be processed. See example in \code{\link[=p_direction]{p_direction()}}.}

\item{effects}{Should variables for fixed effects (\code{"fixed"}), random effects
(\code{"random"}) or both (\code{"all"}) be returned? Only applies to mixed models. May
be abbreviated.

For models of from packages \strong{brms} or \strong{rstanarm} there are additional
options:
\itemize{
\item \code{"fixed"} returns fixed effects.
\item \code{"random_variance"} return random effects parameters (variance and
correlation components, e.g. those parameters that start with \code{sd_} or
\code{cor_}).
\item \code{"grouplevel"} returns random effects group level estimates, i.e. those
parameters that start with \code{r_}.
\item \code{"random"} returns both \code{"random_variance"} and \code{"grouplevel"}.
\item \code{"all"} returns fixed effects and random effects variances.
\item \code{"full"} returns all parameters.
}}

\item{component}{Which type of parameters to return, such as parameters for
the conditional model, the zero-inflated part of the model, the dispersion
term, etc. See details in section \emph{Model Components}. May be abbreviated.
Note that the \emph{conditional} component also refers to the \emph{count} or \emph{mean}
component - names may differ, depending on the modeling package. There are
three convenient shortcuts (not applicable to \emph{all} model classes):
\itemize{
\item \code{component = "all"} returns all possible parameters.
\item If \code{component = "location"}, location parameters such as \code{conditional},
\code{zero_inflated}, \code{smooth_terms}, or \code{instruments} are returned (everything
that are fixed or random effects - depending on the \code{effects} argument -
but no auxiliary parameters).
\item For \code{component = "distributional"} (or \code{"auxiliary"}), components like
\code{sigma}, \code{dispersion}, \code{beta} or \code{precision} (and other auxiliary
parameters) are returned.
}}

\item{parameters}{Regular expression pattern that describes the parameters
that should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}
}
\description{
Compute the proportion of the HDI (default to the \verb{89\%} HDI) of a posterior
distribution that lies within a region of practical equivalence.
}
\note{
There is also a \href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\section{ROPE}{

Statistically, the probability of a posterior distribution of being
different from 0 does not make much sense (the probability of a single value
null hypothesis in a continuous distribution is 0). Therefore, the idea
underlining ROPE is to let the user define an area around the null value
enclosing values that are \emph{equivalent to the null} value for practical
purposes (\emph{Kruschke 2010, 2011, 2014}).

Kruschke (2018) suggests that such null value could be set, by default,
to the -0.1 to 0.1 range of a standardized parameter (negligible effect
size according to Cohen, 1988). This could be generalized: For instance,
for linear models, the ROPE could be set as \verb{0 +/- .1 * sd(y)}.
This ROPE range can be automatically computed for models using the
\code{\link[=rope_range]{rope_range()}} function.

Kruschke (2010, 2011, 2014) suggests using the proportion of  the \verb{95\%}
(or \verb{89\%}, considered more stable) \link[=hdi]{HDI} that falls within the
ROPE as an index for "null-hypothesis" testing (as understood under the
Bayesian framework, see \code{\link[=equivalence_test]{equivalence_test()}}).
}

\section{Sensitivity to parameter's scale}{

It is important to consider the unit (i.e., the scale) of the predictors
when using an index based on the ROPE, as the correct interpretation of the
ROPE as representing a region of practical equivalence to zero is dependent
on the scale of the predictors. Indeed, the percentage in ROPE depend on
the unit of its parameter. In other words, as the ROPE represents a fixed
portion of the response's scale, its proximity with a coefficient depends
on the scale of the coefficient itself.
}

\section{Multicollinearity - Non-independent covariates}{

When parameters show strong correlations, i.e. when covariates are not
independent, the joint parameter distributions may shift towards or
away from the ROPE. Collinearity invalidates ROPE and hypothesis
testing based on univariate marginals, as the probabilities are conditional
on independence. Most problematic are parameters that only have partial
overlap with the ROPE region. In case of collinearity, the (joint) distributions
of these parameters may either get an increased or decreased ROPE, which
means that inferences based on \code{rope()} are inappropriate
(\emph{Kruschke 2014, 340f}).

\code{rope()} performs a simple check for pairwise correlations between
parameters, but as there can be collinearity between more than two variables,
a first step to check the assumptions of this hypothesis testing is to look
at different pair plots. An even more sophisticated check is the projection
predictive variable selection (\emph{Piironen and Vehtari 2017}).
}

\section{Strengths and Limitations}{

\strong{Strengths:} Provides information related to the practical relevance of
the effects.

\strong{Limitations:} A ROPE range needs to be arbitrarily defined. Sensitive to
the scale (the unit) of the predictors. Not sensitive to highly significant
effects.
}

\section{Model components}{


Possible values for the \code{component} argument depend on the model class.
Following are valid options:
\itemize{
\item \code{"all"}: returns all model components, applies to all models, but will only
have an effect for models with more than just the conditional model
component.
\item \code{"conditional"}: only returns the conditional component, i.e. "fixed
effects" terms from the model. Will only have an effect for models with
more than just the conditional model component.
\item \code{"smooth_terms"}: returns smooth terms, only applies to GAMs (or similar
models that may contain smooth terms).
\item \code{"zero_inflated"} (or \code{"zi"}): returns the zero-inflation component.
\item \code{"location"}: returns location parameters such as \code{conditional},
\code{zero_inflated}, or \code{smooth_terms} (everything that are fixed or random
effects - depending on the \code{effects} argument - but no auxiliary
parameters).
\item \code{"distributional"} (or \code{"auxiliary"}): components like \code{sigma},
\code{dispersion}, \code{beta} or \code{precision} (and other auxiliary parameters) are
returned.
}

For models of class \code{brmsfit} (package \strong{brms}), even more options are
possible for the \code{component} argument, which are not all documented in detail
here. See also \href{https://easystats.github.io/insight/reference/find_parameters.BGGM.html}{\code{?insight::find_parameters}}.
}

\examples{
\dontshow{if (all(insight::check_if_installed(c("rstanarm", "emmeans", "brms", "BayesFactor"), quietly = TRUE))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(bayestestR)

rope(x = rnorm(1000, 0, 0.01), range = c(-0.1, 0.1))
rope(x = rnorm(1000, 0, 1), range = c(-0.1, 0.1))
rope(x = rnorm(1000, 1, 0.01), range = c(-0.1, 0.1))
rope(x = rnorm(1000, 1, 1), ci = c(0.90, 0.95))
\donttest{
model <- suppressWarnings(
  rstanarm::stan_glm(mpg ~ wt + gear, data = mtcars, chains = 2, iter = 200, refresh = 0)
)
rope(model)
rope(model, ci = c(0.90, 0.95))

# multiple ROPE ranges
rope(model, range = list(c(-10, 5), c(-0.2, 0.2), "default"))

# named ROPE ranges
rope(model, range = list(gear = c(-3, 2), wt = c(-0.2, 0.2)))

rope(emmeans::emtrends(model, ~1, "wt"), ci = c(0.90, 0.95))

model <- brms::brm(mpg ~ wt + cyl, data = mtcars, refresh = 0)
rope(model)
rope(model, ci = c(0.90, 0.95))

model <- brms::brm(
  brms::bf(brms::mvbind(mpg, disp) ~ wt + cyl) + brms::set_rescor(rescor = TRUE),
  data = mtcars,
  refresh = 0
)
rope(model)
rope(model, ci = c(0.90, 0.95))

# different ROPE ranges for model parameters. For each response, a named
# list (with the name of the response variable) is required as list-element
# for the `range` argument.
rope(
  model,
  range = list(
    mpg = list(b_mpg_wt = c(-1, 1), b_mpg_cyl = c(-2, 2)),
    disp = list(b_disp_wt = c(-5, 5), b_disp_cyl = c(-4, 4))
  )
)

bf <- BayesFactor::ttestBF(x = rnorm(100, 1, 1))
rope(bf)
rope(bf, ci = c(0.90, 0.95))
}
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Cohen, J. (1988). Statistical power analysis for the behavioural sciences.
\item Kruschke, J. K. (2010). What to believe: Bayesian methods for data analysis.
Trends in cognitive sciences, 14(7), 293-300. \doi{10.1016/j.tics.2010.05.001}.
\item Kruschke, J. K. (2011). Bayesian assessment of null values via parameter
estimation and model comparison. Perspectives on Psychological Science,
6(3), 299-312. \doi{10.1177/1745691611406925}.
\item Kruschke, J. K. (2014). Doing Bayesian data analysis: A tutorial with R,
JAGS, and Stan. Academic Press. \doi{10.1177/2515245918771304}.
\item Kruschke, J. K. (2018). Rejecting or accepting parameter values in Bayesian
estimation. Advances in Methods and Practices in Psychological Science,
1(2), 270-280. \doi{10.1177/2515245918771304}.
\item Makowski D, Ben-Shachar MS, Chen SHA, Lüdecke D (2019) Indices of Effect
Existence and Significance in the Bayesian Framework. Frontiers in
Psychology 2019;10:2767. \doi{10.3389/fpsyg.2019.02767}
\item Piironen, J., & Vehtari, A. (2017). Comparison of Bayesian predictive
methods for model selection. Statistics and Computing, 27(3), 711–735.
\doi{10.1007/s11222-016-9649-y}
}
}
