% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tryStack.R
\name{tryStack}
\alias{tryStack}
\title{try an expression, returning the error stack}
\usage{
tryStack(expr, silent = FALSE, warn = TRUE, short = TRUE, file = "",
  removetry = TRUE)
}
\arguments{
\item{expr}{Expresssion to try, potentially wrapped in curly braces if 
spanning several commands.}

\item{silent}{Logical: Should printing of error message + stack be suppressed?
Does not affect warnings and messages. DEFAULT: FALSE}

\item{warn}{Logical: trace \code{\link{warning}s} and \code{\link{message}s} also?
They are still handled like regular warnings / messages unless 
\code{file !=""}, when they are catted into that file. DEFAULT: TRUE}

\item{short}{Logical: should trace be abbreviated to upper -> middle -> lower?
If NA, it is set to TRUE for warnings and messages, FALSE for errors.
DEFAULT: TRUE}

\item{file}{File name passed to \code{\link{cat}}. 
If given, Errors will be appended to the file after two empty lines. 
if \code{warn=T} and file!="", warnings and messages will not be shown, 
but also appended to the file.
This is useful in lapply simulation runs.
DEFAULT: "" (catted to the console)}

\item{removetry}{Logical: should all stack entries matching typical tryCatch
expressions be removed? Unless the call contains customized
\code{\link{tryCatch}} code, this can be left to the DEFAULT: TRUE}
}
\value{
Value of \code{expr} if evaluated successfully. If not, an invisible 
object of class "try-error" as in \code{\link{try}} with the stack in \code{object[2]}.
For nested tryStack calls, \code{object[3], object[4]} etc. will contain "-- empty error stack --"
}
\description{
As in \code{\link{try}}, the result of an expression if it works.
If it fails, execution is not halted, but an invisible try-error class object is 
returned and (unless silent=TRUE) a message \code{\link{cat}ted} to the console.\cr
Unlike \code{\link{try}}, \code{tryStack} also returns the calling stack to 
trace errors and warnings and ease debugging.
}
\examples{

# Functions -----

lower <- function(a) {message("fake message, a = ", a); a+10}
middle <- function(b) {plot(b, main=b) ; warning("fake warning, b = ", b); lower(b) }
upper <- function(c) {cat("printing c:", c, "\\n") ; middle(c)}
d <- upper(42)
d
rm(d)


# Classical error management with try -----

\dontrun{ ## intentional error
d <- upper("42")                # error, no d creation 
traceback()                     # calling stack, but only in interactive mode
}

d <- try(upper("42"), silent=TRUE)      # d created
cat(d)                                  # with error message, but no traceback
inherits(d, "try-error")                # use for coding


# way cooler with tryStack -----

d <- tryStack(upper("42"), silent=TRUE) # like try, but with traceback, even for warnings
cat(d)
d <- tryStack(upper("42"), silent=TRUE, warn=0) # don't touch warnings
 
tryStack(upper(42)) # returns normal output, but warnings are easier to debug
# Note: you can also set options(showWarnCalls=TRUE)

stopifnot(inherits(d, "try-error"))
stopifnot(tryStack(upper(42))==52)

\dontrun{ ## file writing not wanted by CRAN checks
d <- tryStack(upper("42"), silent=TRUE, file="log.txt")
system2("open", "log.txt")
unlink("log.txt")
}


# Nested calls -----

f <- function(k) tryStack(upper(k), silent=TRUE)
d <- f(42)                 ; cat("-----\\n", d, "\\n-----\\n") ; rm(d)
d <- f("42")               ; cat("-----\\n", d, "\\n-----\\n") ; rm(d)
d <- tryStack(f(4) )       ; cat("-----\\n", d, "\\n-----\\n") ; rm(d) 
# warnings in nested calls are printed twice, unless warn=0
d <- tryStack(f(4), warn=0) # could also be set within 'f'

d <- tryStack(f("4"))      ; cat("-----\\n", d, "\\n-----\\n") 
d[1:3] ; rm(d)
# empty stack at begin - because of tryStack in f, no real error happened in f


# Other tests -----

cat( tryStack(upper("42")) )
f <- function(k) tryStack(stop("oh oh"))
d <- f(42) ; cat("-----\\n", d, "\\n-----\\n") ; rm(d) # level 4 not helpful, but OK

# stuff with base::try
f <- function(k) try(upper(k), silent=TRUE)
d <- f(42)     ; cat("-----\\n", d, "\\n-----\\n") ; rm(d)
d <- f("42")   ; cat("-----\\n", d, "\\n-----\\n") ; rm(d) # regular try output

f2 <- function(k) tryStack(f(k), warn=0, silent=TRUE)
d <- f2(42)    ; cat("-----\\n", d, "\\n-----\\n") ; rm(d)
d <- f2("42")  ; cat("-----\\n", d, "\\n-----\\n") ; rm(d) # try -> no error. 
# -> Use tryCatch and you can nest those calls. note that d gets longer.


}
\seealso{
\code{\link{try}}, \code{\link{traceCall}},
\url{http://r.789695.n4.nabble.com/Stack-trace-td4021537.html},
\url{http://stackoverflow.com/questions/15282471/get-stack-trace-on-trycatched-error-in-r},
\url{http://stackoverflow.com/questions/1975110/printing-stack-trace-and-continuing-after-error-occurs-in-r},
\url{http://stackoverflow.com/questions/16879821/save-traceback-on-error-using-trycatch}
}
\author{
Berry Boessenkool, \email{berry-b@gmx.de}, Nov 2016
}
\keyword{error}
\keyword{programming}
