#' Estimate Standardized Regression Coefficients
#' and Generate the Corresponding Sampling Distribution
#' Using Nonparametric Bootstrapping
#'
#' @details The vector of standardized regression coefficients
#'   (\eqn{\boldsymbol{\hat{\beta}}})
#'   is estimated from bootstrap samples.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{\boldsymbol{\hat{\beta}}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @return Returns an object
#'   of class `betanb` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{object}{The function argument `object`.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{\boldsymbol{\hat{\beta}}}.}
#'     \item{jackknife}{Jackknife estimates.}
#'     \item{est}{Vector of estimated
#'       \eqn{\boldsymbol{\hat{\beta}}}.}
#'     \item{fun}{Function used ("BetaNB").}
#'   }
#'
#' @param object Object of class `mc`, that is,
#'   the output of the `MC()` function.
#'
#' @examples
#' # Fit the regression model
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#' # Generate the sampling distribution of sample covariances
#' # (use a large R, for example, R = 5000 for actual research)
#' nb <- NB(object, R = 50)
#' # Generate confidence intervals for standardized regression slopes
#' std <- BetaNB(nb)
#' # Methods --------------------------------------------------------
#' print(std)
#' summary(std)
#' coef(std)
#' vcov(std)
#' confint(std, level = 0.95)
#' @export
#' @family Beta Nonparametric Bootstrap Functions
#' @keywords betaNB std
BetaNB <- function(object) {
  stopifnot(
    methods::is(
      object,
      "nb"
    )
  )
  fun <- "BetaNB"
  est <- object$lm_process$betastar
  foo <- function(x) {
    return(
      .BetaStarofSigma(
        sigmacap = x,
        q = 1 / sqrt(diag(x)),
        k = object$lm_process$k
      )
    )
  }
  thetahatstar <- lapply(
    X = object$thetahatstar,
    FUN = foo
  )
  vcov <- stats::var(
    do.call(
      what = "rbind",
      args = thetahatstar
    )
  )
  colnames(vcov) <- rownames(vcov) <- names(est)
  out <- list(
    call = match.call(),
    object = object,
    thetahatstar = thetahatstar,
    jackknife = lapply(
      X = object$jackknife,
      FUN = foo
    ),
    vcov = vcov,
    est = est,
    fun = fun
  )
  class(out) <- c(
    "betanb",
    class(out)
  )
  return(
    out
  )
}
