% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pruning.R
\name{pruning-clumping}
\alias{pruning-clumping}
\alias{snp_clumping}
\alias{snp_pruning}
\alias{snp_indLRLDR}
\title{LD pruning and clumping}
\usage{
snp_clumping(G, infos.chr, ind.row = rows_along(G), S = NULL, size = 500,
  is.size.in.bp = FALSE, infos.pos = NULL, thr.r2 = 0.2, exclude = NULL,
  ncores = 1)

snp_pruning(G, infos.chr, ind.row = rows_along(G), size = 49,
  is.size.in.bp = FALSE, infos.pos = NULL, thr.r2 = 0.2, exclude = NULL,
  nploidy = getOption("bigsnpr.nploidy"), ncores = 1)

snp_indLRLDR(infos.chr, infos.pos, LD.regions = LD.wiki34)
}
\arguments{
\item{G}{A \link[=FBM.code256-class]{FBM.code256}
(typically \code{<bigSNP>$genotypes}).\cr
\strong{You shouldn't have missing values in your data or SNPs with 0 MAF.}}

\item{infos.chr}{Vector of integers specifying each SNP's chromosome.\cr
Typically \code{<bigSNP>$map$chromosome}.}

\item{ind.row}{An optional vector of the row indices (individuals) that
are used. If not specified, all rows are used.\cr
\strong{Don't use negative indices.}}

\item{S}{A vector of column statistics which express the importance
of each SNP (the more important is the SNP, the greater should be
the corresponding statistic).\cr
For example, if \code{S} follows the standard normal distribution, and "important"
means significantly different from 0, you must use \code{abs(S)} instead.\cr
If not specified, the MAF is computed and used.}

\item{size}{For one SNP, number of SNPs at its left and its right to
be tested for being correlated with this particular SNP.
This parameter should be adjusted with respect to the number of SNPs.
The default are
\itemize{
\item \code{49} for pruning (as in PLINK),
\item \code{500} for clumping (I use this for a chip of 500K SNPs).
}}

\item{is.size.in.bp}{Is the specified size is corresponding to
kilo base-pairs. Default is \code{FALSE} (it is only SNP counts).}

\item{infos.pos}{Vector of integers specifying the physical position
on a chromosome (in base pairs) of each SNP.\cr
Typically \code{<bigSNP>$map$physical.pos}.}

\item{thr.r2}{Threshold over the squared correlation between two SNPs.
Default is \code{0.2}.}

\item{exclude}{Vector of SNP indices to exclude anyway. For example,
can be used to exclude long-range LD regions (see Price2008). Another use
can be for thresholding with respect to p-values associated with \code{S}.}

\item{ncores}{Number of cores used. Default doesn't use parallelism.
You may use \link{nb_cores}.}

\item{nploidy}{Number of trials, parameter of the binomial distribution.
Default is \code{2}, which corresponds to diploidy, such as for the human genome.}

\item{LD.regions}{A \code{data.frame} with columns "Chr", "Start" and "Stop".
Default use the table of 34 long-range LD regions that you can find
\href{https://goo.gl/0Ou7uI}{there}.}
}
\value{
\itemize{
\item \code{snp_pruning} & \code{snp_pruning}: SNP indices which are \strong{kept}.
\item \code{snp_indLRLDR}: SNP indices to be used as (part of) the
\strong{\code{exclude}} parameter of \code{snp_pruning} or \code{snp_clumping}.
}
}
\description{
For a \code{bigSNP}:
\itemize{
\item \code{snp_pruning}: LD pruning. Similar to "\code{--indep-pairwise (size+1) 1 thr.r2}"
in \href{https://www.cog-genomics.org/plink/1.9/ld}{PLINK}.
(\code{step} is fixed to 1).
\item \code{snp_clumping}: LD clumping.
\item \code{snp_indLRLDR}: Get SNP indices of long-range LD regions for the
human genome.
}
}
\details{
I recommend to use clumping rather than pruning. See
\href{https://privefl.github.io/bigsnpr/articles/pruning-vs-clumping.html}{this article}.
}
\examples{
set.seed(1)

test <- snp_attachExtdata()
G <- test$genotypes
n <- nrow(G)
m <- ncol(G)

# pruning / clumping with MAF
ind.keep <- snp_pruning(G, infos.chr = test$map$chromosome, thr.r2 = 0.1)
# keep most of them -> not much LD in this simulated dataset
length(ind.keep) / m

ind.keep2 <- snp_clumping(G, infos.chr = test$map$chromosome, thr.r2 = 0.1)
}
\references{
Price AL, Weale ME, Patterson N, et al.
Long-Range LD Can Confound Genome Scans in Admixed Populations.
Am J Hum Genet. 2008;83(1):132-135.
\url{http://dx.doi.org/10.1016/j.ajhg.2008.06.005}
}
