#' @title Download a NCBI Database to Your Local Hard Drive
#' @description This function allows users to download a database selected by
#' \code{\link{listDatabases}} to their local hard drive.
#' @param db a character string specifying the database that shall be downloaded (selected from \code{\link{listDatabases}}).
#' @param path a character string specifying the location (a folder) in which the corresponding
#' database shall be stored. Default is \code{path = "database"}. In case this folder does not exist yet, it will be created.
#' @details
#' This function downloads large databases to your hard drive. For this purpose a folder
#' named \code{database} (default) is created and the correspondning database then stored in this folder. 
#' 
#' @author Hajk-Georg Drost
#' @examples 
#' \dontrun{
#'   # search for available NCBI nr databases
#'   listDatabases(db = "nr")
#'   
#'   # select NCBI nr version 27 =  "nr.27.tar.gz"
#'   # and download it to your hard drive
#'   # -> please note that large databases take some time for download!
#'   download.database(db = "nr.27.tar.gz")
#' 
#' }
#' @seealso \code{\link{download.database.all}}, \code{\link{listDatabases}}
#' @export

download.database <- function(db, path = "database"){
    
    # test if internet connection is available
    connected.to.internet()
    
    db.name <- names(table(unlist(sapply(db, function(x)
        unlist(stringr::str_split(x, "[.]"))[1]))))
    
    if (!is.element(db.name, listDatabases(db = db.name)) || !is.element(db.name, listDatabases("all")))
        stop(
            "The specified database '",
            db,
            "' could not be found on NCBI. Please use the listDatabases() command to retrieve available databases or check if the name was written correctly.",
            call. = FALSE
        )
    
    
    if (!file.exists(path))
        dir.create(path)
    
    tryCatch({
        downloader::download(paste0("ftp://ftp.ncbi.nlm.nih.gov/blast/db/", db),
                             file.path(path, db) ,
                             mode = "wb")
    }, error = function(e)
        stop(
            "The FTP site 'ftp://ftp.ncbi.nlm.nih.gov/blast/db/' cannot be reached. Are you connected to the internet?",
            call. = FALSE
        ))
    
    # limit NCBI queries
    Sys.sleep(0.33)
}









