% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hclu_hierarclust.R
\name{hclu_hierarclust}
\alias{hclu_hierarclust}
\title{Hierarchical clustering based on dissimilarity or beta-diversity}
\usage{
hclu_hierarclust(
  dissimilarity,
  index = names(dissimilarity)[3],
  method = "average",
  randomize = TRUE,
  n_runs = 30,
  keep_trials = FALSE,
  optimal_tree_method = "best",
  n_clust = NULL,
  cut_height = NULL,
  find_h = TRUE,
  h_max = 1,
  h_min = 0
)
}
\arguments{
\item{dissimilarity}{the output object from \code{\link[=dissimilarity]{dissimilarity()}} or
\code{\link[=similarity_to_dissimilarity]{similarity_to_dissimilarity()}}, or a \code{dist} object.
If a \code{data.frame} is used, the first two
columns represent pairs of sites (or any pair of nodes), and the next
column(s) are the dissimilarity indices.}

\item{index}{name or number of the dissimilarity column to use. By default,
the third column name of \code{dissimilarity} is used.}

\item{method}{name of the hierarchical classification method, as in
\link[fastcluster:hclust]{hclust}. Should be one of "ward.D",
"ward.D2", "single", "complete", "average"
(= UPGMA), "mcquitty" (= WPGMA), "median" (= WPGMC) or
"centroid" (= UPGMC).}

\item{randomize}{a \code{boolean} indicating if the dissimilarity matrix should be
randomized, to account for the order of sites in the dissimilarity matrix.}

\item{n_runs}{number of trials to randomize the dissimilarity matrix.}

\item{keep_trials}{a \code{boolean} indicating if all random trial results.
should be stored in the output object (set to FALSE to save space if your
\code{dissimilarity} object is large).}

\item{optimal_tree_method}{a \code{character} indicating how the final tree
should be obtained from all trials. The only option currently is
"best", which means the tree with the best cophenetic correlation
coefficient will be chosen.}

\item{n_clust}{an \code{integer} or an \code{integer} vector indicating the number of
clusters to be obtained from the hierarchical tree, or the output from
\link{partition_metrics}. Should not be used at the same time as
\code{cut_height}.}

\item{cut_height}{a \code{numeric} vector indicating the height(s) at which the
tree should be cut. Should not be used at the same time as \code{n_clust}.}

\item{find_h}{a \code{boolean} indicating if the height of cut should be found for
the requested \code{n_clust}.}

\item{h_max}{a \code{numeric} indicating the maximum possible tree height for
the chosen \code{index}.}

\item{h_min}{a \code{numeric} indicating the minimum possible height in the tree
for the chosen \code{index}.}
}
\value{
A \code{list} of class \code{bioregion.clusters} with five slots:
\enumerate{
\item{\strong{name}: \code{character} containing the name of the algorithm}
\item{\strong{args}: \code{list} of input arguments as provided by the user}
\item{\strong{inputs}: \code{list} of characteristics of the clustering process}
\item{\strong{algorithm}: \code{list} of all objects associated with the
clustering procedure, such as original cluster objects}
\item{\strong{clusters}: \code{data.frame} containing the clustering results}}

In the \code{algorithm} slot, users can find the following elements:

\itemize{
\item{\code{trials}: a list containing all randomization trials. Each trial
contains the dissimilarity matrix, with site order randomized, the
associated tree and the cophenetic correlation coefficient (Spearman) for
that tree}
\item{\code{final.tree}: a \code{hclust} object containing the final
hierarchical tree to be used}
\item{\code{final.tree.coph.cor}: the cophenetic correlation coefficient
between the initial dissimilarity matrix and \code{final.tree}}
}
}
\description{
This function generates a hierarchical tree from a dissimilarity
(beta-diversity) \code{data.frame}, calculates the cophenetic correlation
coefficient, and can get clusters from the tree if requested by the user.
The function implements randomization of the dissimilarity matrix to
generate the tree, with a selection method based on the optimal cophenetic
correlation coefficient. Typically, the dissimilarity \code{data.frame} is a
\code{bioregion.pairwise.metric} object obtained by running \code{similarity}
or \code{similarity} and then \code{similarity_to_dissimilarity}.
}
\details{
The function is based on \link[fastcluster:hclust]{hclust}.
The default method for the hierarchical tree is \code{average}, i.e.
UPGMA as it has been recommended as the best method to generate a tree
from beta diversity dissimilarity \insertCite{Kreft2010}{bioregion}.

Clusters can be obtained by two methods:
\itemize{
\item{Specifying a desired number of clusters in \code{n_clust}}
\item{Specifying one or several heights of cut in \code{cut_height}}}

To find an optimal number of clusters, see \code{\link[=partition_metrics]{partition_metrics()}}
}
\examples{
comat <- matrix(sample(0:1000, size = 500, replace = TRUE, prob = 1/1:1001),
20, 25)
rownames(comat) <- paste0("Site",1:20)
colnames(comat) <- paste0("Species",1:25)

dissim <- dissimilarity(comat, metric = "all")

# User-defined number of clusters
tree1 <- hclu_hierarclust(dissim, n_clust = 5)
tree1
plot(tree1)
str(tree1)
tree1$clusters

# User-defined height cut
# Only one height
tree2 <- hclu_hierarclust(dissim, cut_height = .05)
tree2
tree2$clusters

# Multiple heights
tree3 <- hclu_hierarclust(dissim, cut_height = c(.05, .15, .25))

tree3$clusters # Mind the order of height cuts: from deep to shallow cuts
# Info on each partition can be found in table cluster_info
tree3$cluster_info
plot(tree3)

# Recut the tree afterwards
tree3.1 <- cut_tree(tree3, n = 5)

tree4 <- hclu_hierarclust(dissim, n_clust = 1:19)

}
\references{
\insertRef{Kreft2010}{bioregion}
}
\seealso{
\link{cut_tree}
}
\author{
Boris Leroy (\email{leroy.boris@gmail.com}),
Pierre Denelle (\email{pierre.denelle@gmail.com}) and
Maxime Lenormand (\email{maxime.lenormand@inrae.fr})
}
