#' Create a file with email access credentials
#'
#' Create a file with access credentials for
#' the purpose of automatically emailing
#' notification messages.
#' @param user the username for the email account.
#' @param password the password associated with the
#' email account.
#' @param provider an optional provider email provider
#' with which an STMP account is available. Options
#' currently include \code{gmail}, \code{outlook},
#' and \code{office365}. If nothing is provided then
#' values for \code{host}, \code{port},
#' \code{sender}, \code{use_ssl}, \code{use_tls},
#' and \code{authenticate} are expected.
#' @param host the \code{host} name.
#' @param port the port number.
#' @param sender the sender name.
#' @param use_ssl an option as to whether to use
#' SSL; supply a \code{TRUE} or \code{FALSE}
#' value (\code{TRUE} is the default value).
#' @param use_tls a logical value to
#' indicate whether to use TLS; supply a \code{TRUE}
#' or \code{FALSE} value (\code{FALSE} is the
#' default value).
#' @param authenticate an option as to whether to
#' authenticate; supply a \code{TRUE} or \code{FALSE}
#' value (\code{TRUE} is the default value).
#' @param creds_file_name an option to specify a
#' name for the credentials file. If no name is
#' provided, one will be automatically generated.
#' The autogenerated file will be invisible and
#' have its name constructed in the following way:
#' \code{.bls_<host_name>}.
#' @examples
#' \dontrun{
#' # Create a credentials file to facilitate
#' # the sending of email messages
#' create_email_creds_file(
#'   user = "user_name@gmail.com",
#'   password = "************",
#'   provider = "gmail")
#' }
#' @importFrom stringr str_replace_all
#' @importFrom dplyr pull filter
#' @export
create_email_creds_file <- function(user,
                                    password,
                                    provider = NULL,
                                    host = NULL,
                                    port = NULL,
                                    sender = NULL,
                                    use_ssl = TRUE,
                                    use_tls = FALSE,
                                    authenticate = TRUE,
                                    creds_file_name = NULL) {

  # Add bindings to global variables
  short_name <- server <- NULL

  # Ensure that `use_ssl` is either TRUE or FALSE
  if (!(use_ssl %in% c(TRUE, FALSE))) {
    stop("The value supplied to `use_ssl` must be TRUE or FALSE.")
  }

  # Ensure that `use_tls` is either TRUE or FALSE
  if (!(use_tls %in% c(TRUE, FALSE))) {
    stop("The value supplied to `use_tls` must be TRUE or FALSE.")
  }

  # Ensure that `authenticate` is either TRUE or FALSE
  if (!(authenticate %in% c(TRUE, FALSE))) {
    stop("The value supplied to `authenticate` must be TRUE or FALSE.")
  }

  # If a `provider` name is given, extract values for `host`,
  # `port`, `use_ssl`, `use_tls`, and `authenticate`
  if (!is.null(provider) &&
      provider %in% (smtp_settings() %>% dplyr::pull(short_name))) {

    # Extract the record for the SMTP provider
    settings_record <-
      smtp_settings() %>%
      dplyr::filter(short_name == provider)

    # Extract settings for the provider
    host <- settings_record %>% dplyr::pull(server)
    port <- settings_record %>% dplyr::pull(port)
    use_ssl <- settings_record %>% dplyr::pull(use_ssl)
    use_tls <- settings_record %>% dplyr::pull(use_tls)
    authenticate <- settings_record %>% dplyr::pull(authenticate)
  }

  # Collect all credential values into a
  # named vector
  credentials <- c(
    sender = as.character(sender),
    host = as.character(host),
    port = as.character(port),
    user = as.character(user),
    password = as.character(password),
    use_ssl = as.character(use_ssl),
    use_tls = as.character(use_tls),
    authenticate = as.character(authenticate))

  if (is.null(creds_file_name)) {

    # Construct a file name
    file <-
      paste0(
        ".bls_",
        stringr::str_replace_all(
          string = host,
          pattern = "\\.",
          replacement = "_"))

  } else {
    file <- as.character(creds_file_name)
  }

  # Save the credential values as an RDS file
  saveRDS(credentials, file = file)
}
