% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/environBlock.R
\name{envBlock}
\alias{envBlock}
\title{Use environmental clustering to separate train and test folds}
\usage{
envBlock(
  rasterLayer,
  speciesData,
  species = NULL,
  k = 5,
  standardization = "normal",
  rasterBlock = TRUE,
  sampleNumber = 10000,
  biomod2Format = TRUE,
  numLimit = 0,
  verbose = TRUE
)
}
\arguments{
\item{rasterLayer}{A raster object of covariates to identify environmental groups.}

\item{speciesData}{A simple features (sf) or SpatialPoints object containing species data (response variable).}

\item{species}{Character. Indicating the name of the field in which species data (binary response i.e. 0 and 1) is stored. If \code{speceis = NULL}
the presence and absence data (response variable) will be treated the same and only training and testing records will be counted. This can be used for multi-class responses
such as land cover classes for remote sensing image classification, but it is not necessary. \emph{Do not use this argument when the response variable is
continuous or count data}.}

\item{k}{Integer value. The number of desired folds for cross-validation. The default is \code{k = 5}.}

\item{standardization}{Standardize input raster layers. Three possible inputs are "normal" (the default), "standard" and "none".
See details for more information.}

\item{rasterBlock}{Logical. If TRUE, the clustering is done in the raster layer rather than species data. See details for
more information.}

\item{sampleNumber}{Integer. The number of samples from raster layers to build the clusters.}

\item{biomod2Format}{Logical. Creates a matrix of folds that can be directly used in the \pkg{biomod2} package as
a \emph{DataSplitTable} for cross-validation.}

\item{numLimit}{Integer value. The minimum number of points in each category of data (\emph{train_0},
\emph{train_1}, \emph{test_0} and \emph{test_1}). Shows a message if the number of points
in any of the folds happens to be less than this number.}

\item{verbose}{Logical. To print the report of the recods per fold.}
}
\value{
An object of class S3. A list of objects including:
   \itemize{
    \item{folds - a list containing the folds. Each fold has two vectors with the training (first) and testing (second) indices}
    \item{foldID - a vector of values indicating the number of the fold for each observation (each number corresponds to the same point in species data)}
    \item{biomodTable - a matrix with the folds to be used in \pkg{biomod2} package}
    \item{k - number of the folds}
    \item{species - the name of the species (column), if provided}
    \item{records - a table with the number of points in each category of training and testing}
    }
}
\description{
Environmental blocking for cross-validation. This function uses clustering methods to specify sets of similar environmental
conditions based on the input covariates. Species data corresponding to any of these groups or clusters are assigned to a
fold. This function does the clustering in raster space and species data. Clustering is done using \code{\link[stats]{kmeans}}
for both approaches. This function works on single or
multiple raster files; multiple rasters need to be in a raster brick or stack format.
}
\details{
As k-means algorithms use Euclidean distance to estimate clusters, the input covariates should be quantitative variables. Since
variables with wider ranges of values might dominate the clusters and bias the environmental clustering (Hastie et al., 2009),
all the input rasters are first standardized within the function. This is done either by normalizing based on subtracting the
mean and dividing by the standard deviation of each raster (the default) or optionally by standardizing using linear scaling
to constrain all raster values between 0 and 1.

By default, the clustering is done in the raster space. In this approach the clusters will be consistent throughout the region
and across species (in the same region). However, this may result in a cluster(s) that covers none of the species records (the spatial location of response samples),
espcially when species data is not dispersed throughout the region or the number of clusters (k or folds) is high. In this
case, the number of folds is less than specified \code{k}. If \code{rasterBlock = FALSE}, the clustering will be done in
species points and the number of the folds will be the same as \code{k}.

Note that the input raster layer should cover all the species points, otherwise an error will rise. The records with no raster
value should be deleted prior to the analysis or another raster layer would be provided.
}
\examples{
\donttest{

# load package data
awt <- raster::brick(system.file("extdata", "awt.grd", package = "blockCV"))
# import presence-absence species data
PA <- read.csv(system.file("extdata", "PA.csv", package = "blockCV"))
# make a sf object from data.frame
pa_data <- sf::st_as_sf(PA, coords = c("x", "y"), crs = raster::crs(awt))

# environmental clustering
eb <- envBlock(rasterLayer = awt,
               speciesData = pa_data,
               species = "Species", # name of the column with response
               k = 5,
               standardization = "standard",
               rasterBlock = TRUE)
}

}
\references{
Hastie, T., Tibshirani, R., & Friedman, J. (2009). The elements of statistical learning: Data Mining, Inference,
and Prediction (2nd ed., Vol. 1). Springer series in statistics New York.

Roberts et al., (2017). Cross-validation strategies for data with temporal, spatial, hierarchical, or phylogenetic structure. Ecography. 40: 913-929.
}
\seealso{
\code{\link{spatialBlock}} and \code{\link{buffering}} for alternative blocking strategies; \code{\link{foldExplorer}}
for visualisation of the generated folds.

For \emph{DataSplitTable} see \code{\link[biomod2]{BIOMOD_cv}} in \pkg{biomod2} package.
for clustering.
}
