% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{cIRT}
\alias{cIRT}
\title{Generic Implementation of Choice IRT MCMC}
\usage{
cIRT(subject_ids, fixed_effects, B_elem_plus1, rv_effects, trial_matrix,
  choices_nk, burnit, chain_length = 10000L)
}
\arguments{
\item{subject_ids}{A \code{vector} that contains subject IDs for each line of data in the choice vector 
(e.g. For 1 subject that made 5 choices, we would have the number 1 appear five times consecutively.)}

\item{fixed_effects}{A \code{matrix} with NK x P1 dimensions that acts as the design matrix for terms WITHOUT theta.}

\item{B_elem_plus1}{A V[[1]] dimensional column \code{vector} indicating which zeta_i relate to theta_i.}

\item{rv_effects}{A \code{matrix} with NK x V dimensions for random effects design matrix.}

\item{trial_matrix}{A \code{matrix} with N x J dimensions, where J denotes the number of items presented.
The matrix MUST contain only 1's and 0's.}

\item{choices_nk}{A \code{vector} with NK length that contains the choice value e.g. 0 or 1.}

\item{burnit}{An \code{int} that describes how many MCMC draws should be discarded.}

\item{chain_length}{An \code{int} that controls how many MCMC draws there are. (>0)}
}
\value{
A \code{list} that contains:
\describe{
  \item{\code{as}}{A \code{matrix} of dimension chain_length x J}
  \item{\code{bs}}{A \code{matrix} of dimension chain_length x J}
  \item{\code{gs}}{A \code{matrix} of dimension chain_length x P_1}
  \item{\code{Sigma_zeta_inv}}{An \code{array} of dimension V x V x chain_length}
  \item{\code{betas}}{A \code{matrix} of dimension chain_length x P_2}
}
}
\description{
Builds a model using MCMC
}
\examples{
\dontrun{
# Variables
# Y = trial matix
# C = KN vector of binary choices
# N = #of subjects
# J = # of items
# K= # of choices
# atrue = true item discriminations
# btrue = true item locations
# thetatrue = true thetas/latent performance
# gamma = fixed effects coefficients
# Sig = random-effects variance-covariance
# subid = id variable for subjects

library(cIRT)
# Simulate 2PNO Data
N = 1000  # Students
J = 20    # Total numbers of possible items per SA

# Randomly pick a's and b's
set.seed(1337)
# Generate as, bs
atrue=runif(J)+1
btrue=2*runif(J)-1
# 2 Level Probit Data
K = 30
gam_notheta = c(.5,1)
gam_theta   = c(3,.25)
gamma = c(gam_notheta,gam_theta)
Sig = matrix(c(.25,0,0,.125),2,2)
    
# True theta and etay
thetatrue = rnorm(N)
etay = outer(rep(1,N),atrue) * thetatrue - outer(rep(1,N),btrue)
  
# Generate Y for 2PNO model
p.correct = pnorm(etay)
Y = matrix(rbinom(N*J, 1, p.correct),N,J)

#################################################
# Simulating 2 level probit data
#################################################
    
subid = expand.grid(cid = 1:K,sid = 1:N)[,2]

pred = rnorm(K*N,0,1) # Pred

center_pred = center_matrix(as.matrix(pred))

Xnotheta = cbind(1,center_pred)
      
Xtheta = rep(thetatrue,each=K)*Xnotheta

X = cbind(Xnotheta,Xtheta)
zetas = mvtnorm::rmvnorm(N,mean=c(0,0),sigma=Sig) # mvtnorm environment accessed
W_veczeta = apply(Xnotheta*zetas[rep(1:N,each=K),],1,sum)
etac = X\%*\%gamma + W_veczeta
Zc = rnorm(N*K,mean=etac,sd=1)
C = 1*(Zc>0)
    
out1 = cIRT(subid, 
            Xnotheta,
            c(1,2), 
            Xnotheta,
            Y, 
            C, 
            5000)
}  
}
\author{
Steven Culpepper, James J Balamuta
}
\seealso{
\code{\link{TwoPLChoicemcmc}}, \code{\link{probitHLM}}, \code{\link{center_matrix}}, \code{\link{rmvnorm}}, \code{\link{rwishart}}, and \code{\link{riwishart}}
}

