% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derivative.R
\name{derivative}
\alias{derivative}
\title{Numerical and Symbolic Derivatives}
\usage{
derivative(f, var = "x", order = 1, accuracy = 2, stepsize = NULL,
  deparse = TRUE)
}
\arguments{
\item{f}{function, expression or character array.}

\item{var}{character vector, giving the variable names with respect to which derivatives will be computed. If a named vector is provided, derivatives will be computed at that point. See examples.}

\item{order}{integer vector, giving the differentiation order for each variable. See details.}

\item{accuracy}{accuracy degree for numerical derivatives.}

\item{stepsize}{finite differences stepsize for numerical derivatives. Auto-optimized by default.}

\item{deparse}{logical. Return character instead of expression or call?}
}
\value{
array of derivatives.
}
\description{
Computes symbolic derivatives based on the \code{\link[stats]{D}} function, or accurate and reliable numerical derivatives based on finite differences.
}
\details{
The function behaves differently depending on the length of the \code{order} argument.

If \code{order} is of length 1, then the n-th order derivative is computed for each function with respect to each 
variable.
\deqn{D = \partial^{(n)} \otimes F \rightarrow D_{i,.,j,k,.,l} = \partial^{(n)}_{k,.,l} F_{i,.,j}}
where \eqn{F} is the tensor of functions and \eqn{\partial} is the tensor of variable names with respect to which 
the \eqn{n}-th order derivatives will be computed.

If \code{order} matches the length of \code{var}, then it is assumed that the differentiation order is provided
for each variable. In this case, each function will be derived \eqn{n_i} times with respect to the \eqn{i}-th variable, 
for each of the \eqn{j} variables.
\deqn{D = \partial^{(n_1)}_1\partial^{(...)}_{...}\partial^{(n_i)}_i\partial^{(...)}_{...}\partial^{(n_j)}_j F}
where \eqn{F} is the tensor of functions to differentiate. 

If \code{var} is a named vector, e.g. \code{c(x = 0, y = 0)}, derivatives will be computed at that point. 
Note that if \code{f} is a function, then \code{var} must be a named vector giving the point at which the numerical derivatives will be computed.
}
\examples{
# derive f with respect to x
derivative(f = "sin(x)", var = "x")

# derive f with respect to x and evaluate in x = 0
derivative(f = "sin(x)", var = c("x" = 0))

# derive f twice with respect to x
derivative(f = "sin(x)", var = "x", order = 2)

# derive f once with respect to x, and twice with respect to y
derivative(f = "y^2*sin(x)", var = c("x","y"), order = c(1,2))

# compute the gradient of f with respect to (x,y)
derivative(f = "y*sin(x)", var = c("x","y"))

# compute the Jacobian of f with respect to (x,y)
f <- c("y*sin(x)", "x*cos(y)")
derivative(f = f, var = c("x","y"))

# compute the Hessian of f with respect to (x,y)
g <- derivative(f = "y^2*sin(x)", var = c("x","y"))
derivative(f = g, var = c("x","y"))

# compute the Jacobian of f with respect to (x,y) and evaluate in (0,0)
f1 <- function(x, y) y*sin(x)
f2 <- function(x, y) x*cos(y)
derivative(f = c(f1, f2), var = c("x"=0,"y"=0))

}
