% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitting.R
\name{fitSmoothHazard}
\alias{fitSmoothHazard}
\title{Fit smooth-in-time parametric hazard functions.}
\usage{
fitSmoothHazard(formula, data, time, censored.indicator, ...)
}
\arguments{
\item{formula}{an object of class "formula" (or one that can be coerced to
that class): a symbolic description of the model to be fitted. The details
of model specification are given under Details.}

\item{data}{a data frame, list or environment containing the variables in the
model. If not found in data, the variables are taken from
\code{environment(formula)}, typically the environment from which
\code{fitSmoothHazard} is called.}

\item{time}{a character string giving the name of the time variable. See
Details.}

\item{censored.indicator}{a character string of length 1 indicating which
value in \code{event} is the censored. This function will use
\code{\link[stats]{relevel}} to set \code{censored.indicator} as the
reference level. This argument is ignored if the \code{event} variable is a
numeric}

\item{...}{Additional parameters passed to \code{\link{sampleCaseBase}}. If
\code{data} inherits from the class \code{cbData}, then these parameters
are ignored.}
}
\value{
An object of \code{glm} and \code{lm} when there is only one event of
  interest, or of class \code{\link{CompRisk}}, which inherits from
  \code{vglm}, for a competing risk analysis. As such, functions like
  \code{summary}, \code{deviance} and \code{coefficients} give familiar
  results.
}
\description{
Miettinen and Hanley (2009) explained how case-base sampling can be used to
estimate smooth-in-time parametric hazard functions. The idea is to sample
person-moments, which may or may not correspond to an event, and then fit the
hazard using logistic regression.
}
\details{
The object \code{data} should either be the output of the function
\code{\link{sampleCaseBase}} or the source dataset on which case-base
sampling will be performed. In the latter case, it is assumed that
\code{data} contains the two columns corresponding to the supplied time and
event variables. If \code{time} is missing, the function looks for a column
named \code{"time"} in the data. Note that the event variable is inferred
from \code{formula}, since it is the left hand side.
}
\examples{
# Simulate censored survival data for two outcome types from exponential distributions
library(data.table)
set.seed(12345)
nobs <- 5000
tlim <- 20

# simulation parameters
b1 <- 200
b2 <- 50

# event type 0-censored, 1-event of interest, 2-competing event
# t observed time/endpoint
# z is a binary covariate
DT <- data.table(z=rbinom(nobs, 1, 0.5))
DT[,`:=` ("t_event" = rweibull(nobs, 1, b1),
          "t_comp" = rweibull(nobs, 1, b2))]
DT[,`:=`("event" = 1 * (t_event < t_comp) + 2 * (t_event >= t_comp),
         "time" = pmin(t_event, t_comp))]
DT[time >= tlim, `:=`("event" = 0, "time" = tlim)]

out_linear <- fitSmoothHazard(event ~ time + z, DT)
out_log <- fitSmoothHazard(event ~ log(time) + z, DT)
}
