% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/track-measures.R
\name{TrackMeasures}
\alias{TrackMeasures}
\alias{trackLength}
\alias{duration}
\alias{speed}
\alias{displacement}
\alias{squareDisplacement}
\alias{displacementVector}
\alias{maxDisplacement}
\alias{displacementRatio}
\alias{outreachRatio}
\alias{straightness}
\alias{overallAngle}
\alias{meanTurningAngle}
\alias{overallDot}
\alias{overallNormDot}
\alias{asphericity}
\alias{hurstExponent}
\alias{fractalDimension}
\title{Track Measures}
\usage{
trackLength(x)

duration(x)

speed(x)

displacement(x, from = 1, to = nrow(x))

squareDisplacement(x, from = 1, to = nrow(x))

displacementVector(x)

maxDisplacement(x)

displacementRatio(x)

outreachRatio(x)

straightness(x)

overallAngle(x, from = 1, to = nrow(x), xdiff = diff(x), degrees = FALSE)

meanTurningAngle(x, degrees = FALSE)

overallDot(x, from = 1, to = nrow(x), xdiff = diff(x))

overallNormDot(x, from = 1, to = nrow(x), xdiff = diff(x))

asphericity(x)

hurstExponent(x)

fractalDimension(x)
}
\arguments{
\item{x}{a single input track; a matrix whose first column is time and whose
remaining columns are a spatial coordinate.}

\item{from}{index, or vector of indices, of the first row of the track.}

\item{to}{index, or vector of indices, of last row of the track.}

\item{xdiff}{row differences of x.}

\item{degrees}{logical; should angles be returned in degrees rather than radians?}
}
\value{
\code{trackLength} sums up the distances between subsequent positsion; in other words,
it estimates the length of the underlying track by linear interpolation (usually
an underestimation). The estimation could be improved in some circumstances by using
\code{\link{interpolateTrack}}. The function returns a single, non-negative number.

\code{duration} returns the time elapsed between \code{x}'s first and last
positions (a single, non-negative number).

\code{speed} simply divides \code{\link{trackLength}} by \code{\link{duration}}

\code{displacement} returns the Euclidean distance between the track endpoints
and \code{squareDisplacement} returns the squared Euclidean distance.

\code{displacementVector} returns the vector between the track endpoints. This
vector has an element (can be negative) for each (x,y,z) dimension of the coordinates
in the track.

\code{maxDisplacement} computes the maximal Euclidean distance of any position
on the track from the first position.

\code{displacementRatio} divides the \code{displacement} by the \code{maxDisplacement};
\code{outreachRatio} divides the \code{maxDisplacement} by the \code{trackLength}
(Mokhtari et al, 2013). Both measures return
values between 0 and 1, where 1 means a perfectly straight track.
If the track has \code{trackLength} 0, then \code{NaN} is returned.

\code{straightness} divides the \code{displacement} by the \code{trackLength}.
This gives a number between 0 and 1, with 1 meaning a perfectly straight track.
If the track has \code{trackLength} 0, then \code{NaN} is returned.

\code{asphericity} is a different appraoch to measure straightness
(Mokhtari et al, 2013): it computes the asphericity of the set of positions on the
track _via_ the length of its principal components. Again this gives a number between 0
and 1, with higher values indicating straighter tracks.
Unlike \code{\link{straightness}}, however, asphericity ignores
back-and-forth motion of the object, so something that bounces between two positions
will have low \code{straightness} but high \code{asphericity}. We define the
asphericity of every track with two or fewer positions to be 1. For one-dimensional
tracks with one or more positions, \code{NA} is returned.

\code{overallAngle} Computes the angle (in radians) between the first and the last
segment of the given track. Angles are measured symmetrically, thus the return values
range from 0 to pi; for instance, both a 90 degrees left and right turns yield the
value pi/2. This function is useful to generate autocorrelation plots
(together with \code{\link{aggregate.tracks}}). Angles can also be returned in degrees,
in that case: set \code{degrees = TRUE}.

\code{meanTurningAngle} averages the \code{overallAngle} over all
adjacent segments of a given track; a low \code{meanTurningAngle} indicates high
persistence of orientation, whereas for an uncorrelated random walk we expect
90 degrees. Note that angle measurements will yield \code{NA} values for tracks
in which two subsequent positions are identical. By default returns angles in
radians; use \code{degrees = TRUE} to return angles in degrees instead.

\code{overallDot} computes the dot product between the first and the last
segment of the given track. This function is useful to generate autocovariance plots
(together with \code{\link{aggregate.tracks}}).

\code{overallNormDot} computes the dot product between the unit vectors along
the first and the last segment of the given track. This function is useful to
generate autocorrelation plots (together with
\code{\link{aggregate.tracks}}).

\code{hurstExponent} computes the corrected empirical Hurst exponent of the track.
This uses the function \code{\link[pracma]{hurstexp}} from the `pracma` package.
If the track has less than two positions, NA is returned.
\code{fractalDimension} estimates the fractal dimension of a track using the function
\code{\link[fractaldim]{fd.estim.boxcount}} from the
`fractaldim` package. For self-affine processes in \eqn{n} dimensions,
fractal dimension and Hurst exponent
are related by the formula \eqn{H=n+1-D}.
For non-Brownian motion, however, this relationship
need not hold. Intuitively, while the Hurst exponent takes a global approach to the
track's properties, fractal dimension is a local approach to the track's properties
(Gneiting and Schlather, 2004).
}
\description{
Statistics that can be used to quantify tracks. All of these functions take a single
track as input and give a single number as output.
}
\details{
Some track measures consider only the first and last position (or steps) of a track,
and are most useful in conjunction with \code{\link{aggregate.tracks}}; for instance,
\code{squareDisplacement} combined with \code{\link{aggregate.tracks}} gives a mean
square displacement plot, and \code{overallAngle} combined with
\code{\link{aggregate.tracks}} gives a turning angle plot (see the examples for
\code{\link{aggregate.tracks}}). To speed up computation of these measures on
subtracks of the same track, the arguments \code{from}, \code{to} and
possibly \code{xdiff} are exploited by \code{\link{aggregate.tracks}}.
}
\examples{
## show a turning angle plot with error bars for the T cell data.
with( (aggregate(BCells,overallDot,FUN="mean.se",na.rm=TRUE)),{
  plot( mean ~ i, xlab="time step",
  	ylab="turning angle (rad)", type="l" )
  segments( i, lower, y1=upper )
} )

}
\references{
Zeinab Mokhtari, Franziska Mech, Carolin Zitzmann, Mike Hasenberg, Matthias Gunzer
and Marc Thilo Figge (2013), Automated Characterization and
Parameter--Free Classification of Cell Tracks Based on Local Migration
Behavior. \emph{PLoS ONE} \bold{8}(12), e80808. doi:10.1371/journal.pone.0080808

Tillmann Gneiting and Martin Schlather (2004), Stochastic Models That Separate Fractal
Dimension and the Hurst Effect. \emph{SIAM Review} \bold{46}(2), 269--282.
doi:10.1137/S0036144501394387
}
\seealso{
\code{\link{AngleAnalysis}} for methods to compute angles and distances
 between pairs of tracks, or of tracks to a reference point, direction, or plane.
}
