% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tracts_to_polygon.R
\name{tracts_to_polygon}
\alias{tracts_to_polygon}
\title{Convert census tract aggregates to user-provided polygons using CNEFE points}
\usage{
tracts_to_polygon(
  code_muni,
  polygon,
  year = 2022,
  vars = c("pop_ph", "pop_ch"),
  crs_output = NULL,
  cache = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{code_muni}{Integer. Seven-digit IBGE municipality code.}

\item{polygon}{An \code{\link[sf:sf]{sf::sf}} object with polygon geometries (POLYGON or
MULTIPOLYGON). The function will automatically align CRS and issue a warning
reporting the percentage of the polygon area that falls outside the municipality.}

\item{year}{Integer. The CNEFE data year. Currently only 2022 is supported.
Defaults to 2022.}

\item{vars}{Character vector. Names of tract-level variables to interpolate.
Supported variables:
\itemize{
\item \code{pop_ph}: population in private households (\emph{Domicílios particulares}).
\item \code{pop_ch}: population in collective households (\emph{Domicílios coletivos}).
\item \code{male}: total male population.
\item \code{female}: total female population.
\item \code{age_0_4}, \code{age_5_9}, \code{age_10_14}, \code{age_15_19}, \code{age_20_24}, \code{age_25_29},
\code{age_30_39}, \code{age_40_49}, \code{age_50_59}, \code{age_60_69}, \code{age_70m}: population by age group.
\item \code{race_branca}, \code{race_preta}, \code{race_amarela}, \code{race_parda}, \code{race_indigena}:
population by race/color (\emph{cor ou raça}).
\item \code{n_resp}: number of household heads (\emph{Pessoas responsáveis por domicílios}).
\item \code{avg_inc_resp}: average income of the household heads.
}

For a reference table mapping these variable names to the official IBGE
census tract codes and descriptions, see \link{tracts_variables_ref}.

Allocation rules:
\itemize{
\item \code{pop_ph} is allocated only to private dwellings.
\item \code{pop_ch} is allocated only to collective dwellings.
\item \code{n_resp} is allocated only to private dwellings (same rule as \code{pop_ph}).
\item Demographic variables (\code{male}, \code{female}, \verb{age_*}, \verb{race_*}) are allocated
to private dwellings when the tract has any; if the tract has zero private
dwellings but has collective dwellings, they are allocated to collective.
\item \code{avg_inc_resp} is assigned (not split) to each private dwelling point;
tracts with no private dwellings receive no allocation.
}}

\item{crs_output}{The CRS for the output object. Default is \code{NULL}, which uses
the original CRS of the \code{polygon} argument. Can be an EPSG code (e.g., 4326,
31983) or any CRS object accepted by \code{\link[sf:st_transform]{sf::st_transform()}}.}

\item{cache}{Logical. Whether to use the existing package cache for assets and CNEFE zips.}

\item{verbose}{Logical. Whether to print step messages and timing.}
}
\value{
An \code{sf} object with the user-provided polygons and the requested
interpolated variables. The output CRS matches the original \code{polygon} CRS
(or \code{crs_output} if specified).
}
\description{
\code{tracts_to_polygon()} performs a dasymetric interpolation with the following steps:
\enumerate{
\item census tract totals are allocated to CNEFE dwelling points inside each tract;
\item allocated values are aggregated to user-provided polygons (neighborhoods,
administrative divisions, custom areas, etc.).
}

The function uses DuckDB with spatial extensions for the heavy work.
}
\examples{
\donttest{
# Interpolate population to user-provided polygons (neighborhoods of Lauro de Freitas-BA)
# Using geobr to download neighborhood boundaries
library(geobr)
nei_ldf <- subset(
  read_neighborhood(year = 2022),
  code_muni == 2919207
)
poly_pop <- tracts_to_polygon(
  code_muni = 2919207,
  polygon = nei_ldf,
  vars = c("pop_ph", "pop_ch")
)
}

}
