#' Compute isochrones/isodistances from nodes.
#' 
#' @param Graph  An object generated by cppRouting::makegraph() function.
#' @param from numeric or character. A vector of one or more vertices from which isochrones/isodistances are calculated.
#' @param lim numeric. A vector of one or multiple breaks.
#' @param setdif logical. If TRUE and length(lim)>1, nodes that are reachable in a given break will not appear in a greater one.
#' @return List containing reachable nodes below cost limit(s).
#' @note get_isochrone() recursively perform Dijkstra algorithm for each 'from' nodes and stop when cost limit is reached. 
#' @details If length(lim)>1, value is a list of length(from), containing lists of length(lim)
#' @examples 
#' #Data describing edges of the graph
#' edges<-data.frame(from_vertex=c(0,0,1,1,2,2,3,4,4), 
#'                   to_vertex=c(1,3,2,4,4,5,1,3,5), 
#'                   cost=c(9,2,11,3,5,12,4,1,6))
#'                   
#' #Construct directed graph 
#' directed_graph<-makegraph(edges,directed=TRUE)
#' 
#' #Get nodes reachable around node 4 with maximum distances of 1 and 2
#' iso<-get_isochrone(Graph=directed_graph,from = "4",lim=c(1,2))
#' 
#' #With setdif set to TRUE
#' iso2<-get_isochrone(Graph=directed_graph,from = "4",lim=c(1,2),setdif=TRUE)
#' print(iso)
#' print(iso2)

get_isochrone<-function(Graph,from,lim,setdif=FALSE){
  
  
  if (any(is.na(from))) stop("NAs are not allowed in origin nodes")
  from<-as.character(from)
  if (sum(from %in% Graph$dict$ref)<length(from)) stop("Some nodes are not in the graph")
  
  from_id<-Graph$dict$id[match(from,Graph$dict$ref)]
  lim<-as.numeric(lim)
  if (any(is.na(lim))) stop("NAs are not allowed in cost value(s)")
  
  if (length(lim)>1){
    lim<-sort(lim)
    maxlim<-max(lim)
    res<-Isochrone_multi(Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,from_id,lim,maxlim,setdif,Graph$dict$ref)
    for (i in 1:length(res)) names(res[[i]])<-as.character(lim)
  }
  else {
    res<-Isochrone(Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,from_id,lim,Graph$dict$ref)
    
  }
  
  names(res)<-from
  
  return(res)
}
