#' Compute shortest path between origin and destination nodes.
#' 
#' @param Graph  An object generated by cppRouting::makegraph() function.
#' @param from A vector of one or more vertices from which shortest paths are calculated (origin).
#' @param to A vector of one or more vertices (destination).
#' @param algorithm character. "Dijkstra" for uni-directional Dijkstra, "bi" for bi-directional Dijkstra, "A*" for A star unidirectional search or "NBA" for New bi-directional A star .Default to "Dijkstra"
#' @param constant numeric. Constant to maintain the heuristic function admissible in A* algorithm. 
#'Default to 1, when cost is expressed in the same unit than coordinates. See details
#' @return List containing shortest path nodes between from and to.
#' @note 'from' and 'to' must be the same length.
#' @details To perform A* and New Bidirectional A star, projected coordinates should be provided in the Graph object.  
#' In A* and New Bidirectional A star algorithms, euclidean distance is used as heuristic function.
#' To understand how A star algorithm work, see https://en.wikipedia.org/wiki/A*_search_algorithm .
#' To understand the importance of constant parameter, see the package description : https://github.com/vlarmet/cppRouting
#' @examples
#' #Data describing edges of the graph
#' edges<-data.frame(from_vertex=c(0,0,1,1,2,2,3,4,4), 
#'                   to_vertex=c(1,3,2,4,4,5,1,3,5), 
#'                   cost=c(9,2,11,3,5,12,4,1,6))
#'                   
#' #Get all nodes
#' nodes<-unique(c(edges$from_vertex,edges$to_vertex))
#'  
#' #Construct directed and undirected graph 
#' directed_graph<-makegraph(edges,directed=TRUE)
#' non_directed<-makegraph(edges,directed=FALSE)
#' 
#' #Sampling origin and destination nodes
#' origin<-sample(nodes,10,replace=TRUE)
#' destination<-sample(nodes,10,replace=TRUE)
#' 
#' #Get distance between origin and destination in the two graphs
#' dir_paths<-get_path_pair(Graph=directed_graph, from=origin, to=destination)
#' non_dir_paths<-get_path_pair(Graph=non_directed, from=origin, to=destination)
#' print(dir_paths)
#' print(non_dir_paths)

get_path_pair<-function(Graph,from,to,algorithm="Dijkstra",constant=1){
  
  if (length(from)!=length(to)) stop("From and to have not the same length")
  
  if (any(is.na(cbind(from,to)))) stop("NAs are not allowed in origin/destination nodes")
  from<-as.character(from)
  
  to<-as.character(to)
  allnodes<-c(from,to)
  if (sum(allnodes %in% Graph$dict$ref)<length(allnodes)) stop("Some nodes are not in the graph")
  
  from_id<-Graph$dict$id[match(from,Graph$dict$ref)]
  to_id<-Graph$dict$id[match(to,Graph$dict$ref)]
  
  
  if (!is.null(Graph$coords)){
    if (algorithm %in% c("NBA","A*","bi")){
    if (algorithm=="A*"){
      message("Running A* ...")
      res<-Astar_paths(Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,from_id,to_id,Graph$coords[,2],Graph$coords[,3],constant,Graph$dict$ref)
    }
    
    if (algorithm=="NBA"){
      message("Running NBA* ...")
      res<-NBA_path(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,Graph$coords[,2],Graph$coords[,3],constant,Graph$dict$ref)
    }
    
    
     if (algorithm=="bi"){
      message("Running bidirectional Dijkstra...")
      res<-Bidir_path(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
      
     }
    }
    
    else {
      message("Running Dijkstra ...")
      res<-Dijkstra_early_stop_path(Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,from_id,to_id,Graph$dict$ref)
      
    }
    
  }  else {
    if (algorithm=="bi"){
      message("Running bidirectional Dijkstra...")
    res<-Bidir_path(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
  }
  else {
    message("Running Dijkstra ...")
    res<-Dijkstra_early_stop_path(Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,from_id,to_id,Graph$dict$ref)
    
  }
}
  
  
  names(res)<-paste0(from,"_",to)
  return(res)
  
  
}
