#' Compute all shortest paths between origin and destination nodes.
#' 
#' @param Graph  An object generated by makegraph() or cpp_simplify() function.
#' @param from A vector of one or more vertices from which shortest paths are calculated (origin).
#' @param to A vector of one or more vertices (destination).
#' @param keep numeric or character. Vertices of interest that will be returned. 
#' @param long logical. If TRUE, a long data.frame is returned instead of a list.
#' @return List or a data.frame containing shortest paths.
#' @details get_multi_paths() recursively perform Dijkstra algorithm for each 'from' nodes.
#' @note Be aware that if 'from' and 'to' have consequent size, output will require much memory space. 
#' @examples 
#' #Data describing edges of the graph
#' edges<-data.frame(from_vertex=c(0,0,1,1,2,2,3,4,4), 
#'                   to_vertex=c(1,3,2,4,4,5,1,3,5), 
#'                   cost=c(9,2,11,3,5,12,4,1,6))
#'                   
#' #Get all nodes
#' nodes<-unique(c(edges$from_vertex,edges$to_vertex))
#'                   
#' #Construct directed and undirected graph 
#' directed_graph<-makegraph(edges,directed=TRUE)
#' non_directed<-makegraph(edges,directed=FALSE)
#' 
#' #Get all shortest paths between all nodes in the two graphs
#' dir_paths<-get_multi_paths(Graph=directed_graph, from=nodes, to=nodes)
#' non_dir_paths<-get_multi_paths(Graph=non_directed, from=nodes, to=nodes)
#' print(dir_paths)
#' print(non_dir_paths)


get_multi_paths<-function(Graph,from,to,keep=NULL,long=FALSE){
  
  if (length(Graph)!=4) stop("Input should be generated by makegraph() or cpp_simplify() function")
  if (any(is.na(cbind(from,to)))) stop("NAs are not allowed in origin/destination nodes")
  
  
  from<-as.character(from)
  to<-as.character(to)
  allnodes<-c(from,to)
  if (sum(allnodes %in% Graph$dict$ref)<length(allnodes)) stop("Some nodes are not in the graph")
  
  #Nodes to keep
  if (!is.null(keep)) {
    to_keep<-rep(0,Graph$nbnode)
    keep<-as.character(keep)
    to_keep[Graph$dict$ref %in% keep]<-1
  }else{
    to_keep<-rep(1,Graph$nbnode)
  }
  
  from_id<-Graph$dict$id[match(from,Graph$dict$ref)]
  to_id<-Graph$dict$id[match(to,Graph$dict$ref)]
  
  
  res<-Dijkstra_multi_path(Graph$data[,1],Graph$data[,2],Graph$data[,3],Graph$nbnode,from_id,to_id,Graph$dict$ref,to_keep)
  
  names(res)<-from
  
  for (i in 1:length(res)) names(res[[i]])<-to
  
  if (long){
    res<-lapply(res,function(x){
      return(stack(setNames(x,names(x))))
    })
    res<-data.table::rbindlist(res,idcol=TRUE)
    res<-res[,c(1,3,2)]
    res$ind<-as.character(res$ind)
    colnames(res)<-c("from","to","node")
    res<-data.frame(res)
  }
  
  return(res)
}
