# Extract drug manufacturers df
get_manufacturer_rec <- function(r, drug_key) {
  tibble(
    name = xmlValue(r),
    url = xmlGetAttr(r, name = "url"),
    generic = xmlGetAttr(r, name = "generic"),
    parent_key = drug_key
  )
}

get_manufactures_df <- function(rec) {
  return(map_df(xmlChildren(rec[["manufacturers"]]), ~ get_manufacturer_rec(., xmlValue(rec["drugbank-id"][[1]]))))
}

#' Extracts the drug manufacturers element and return data as data frame.
#'
#' \code{parse_drug_manufacturers} returns data frame of drug manufacturers elements.
#'
#' This functions extracts the manufacturers element of drug node in drug bank
#' xml database with the option to save it in a predefined database via
#' \code{\link{open_db}} method. It takes one single optional argument to
#' save the returned dataframe in the database.
#' It must be called after \code{\link{get_xml_db_rows}} function like
#' any other parser function.
#' If \code{\link{get_xml_db_rows}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @return drug manufacturers node attributes date frame
#'
#' @examples
#' \donttest{
#' parse_drug_manufacturers()
#' parse_drug_manufacturers(TRUE)
#' parse_drug_manufacturers(save_table = FALSE)
#' }
#' @export
parse_drug_manufacturers <- function(save_table = FALSE) {
  drug_manufacturers <-
    map_df(pkg.env$children, ~ drug_sub_df(.x, "manufacturers")) %>%
    unique()

  if (nrow(drug_manufacturers) > 0) {
    colnames(drug_manufacturers) <- c("manufacturer", "drugbank_id")
  }

  if (save_table) {
    save_drug_sub(con = pkg.env$con,
                  df = drug_manufacturers,
                  table_name = "drug_manufacturers")
  }
  return(tibble::as_tibble(drug_manufacturers))
}
