
##################################
#                                #
#   MARKOVDATA OBJECT FUNCTION   #
#                                #
##################################

markovdata <- function(dat,itemtypes,nitems=length(itemtypes),ntimes=length(as.matrix(dat))/nitems,inames=NULL,dname=NULL,xm=-9999) {
	TYPES=c("continuous","categorical","count","covariate")
	itt = sapply(itemtypes,FUN=function(x){pmatch(tolower(as.character(x)),TYPES)})
	itemtypes[which(!is.na(itt))]=TYPES[itt[which(!is.na(itt))]]
	if(any(is.na(itemtypes))) stop("Itemtypes incorrectly specified.\n")
	if(is.data.frame(dat)) dat <- data.matrix(dat)
	if(!is.matrix(dat)) dat <- matrix(dat)
	if(!(dim(dat)[1]==sum(ntimes) && dim(dat)[2]==nitems)) dat <- matrix(unlist(dat),sum(ntimes),nitems,byrow=TRUE)
	if(!(length(dat)==(sum(ntimes)*nitems))) stop("Data length incompatible with ntimes and nitems.\n")
	dat=replace(dat,which(dat==xm),NA)
	#rearrange data such that items occur first, and then the covariates, is this neccessary?
	x=c(which(itemtypes!="covariate"),which(itemtypes=="covariate"))
	dat=dat[,x,drop=FALSE]
	itemtypes=itemtypes[x]
 	if(is.null(colnames(dat))) colnames(dat)=itemtypes
	if(!is.null(inames)) colnames(dat)=inames[x]
	if(is.null(dname)) dname=paste(nitems,"-item data")
# 	rownames(dat)=unlist(sapply(ntimes,seq)) make rownames correspond with number in tseries???
	rownames(dat)=NULL
	attr(dat,"dname")=dname
	attr(dat,"itemtypes")=itemtypes
	attr(dat,"ntimes")=ntimes
  	class(dat) = "md"
	return(dat)
}

# functions that return attributes of md objects
ntimes <- function(object) {return(attributes(object)$ntimes)}
itemtypes <- function(object) {return(attributes(object)$itemtypes)}
dname <- function(object) {return(attributes(object)$dname)}

# ... and functions of those same attributes
ncov <- function(object) {return(sum(as.logical(which(itemtypes(object)=="covariate"))))}
inames <- function(object) {return(colnames(object))}
nitems <- function(object) {return(dim(object)[2])}
ind <- function(object) {return(length(attributes(object)$ntimes))}

summary.md <- function(object, ...) {
	cat("Data set:                  ", dname(object), "\n")
	cat(" nr of items:              ", nitems(object), "\n")
	cat(" item type(s):             ", itemtypes(object), "\n")
	if(ncov(object)>0) 
	cat(" nr of covariates:         ", ncov(object), "\n")
	cat(" item name(s):             ", inames(object), "\n")
	cat(" length(s) of series:      ", ntimes(object)[1:min(5,ind(object))])
	if(ind(object)>5) cat(" ... \n") else cat("\n")
	if(ind(object)>1) 
	cat(" nr of independent series: ", ind(object), "\n")
	cat(" data:                     ", object[1,]," ... \n")
}

print.md <- function(x, ...) {
	cat("Data set:                  ", dname(x), "\n")
	cat(" nr of items:              ", nitems(x), "\n")
	cat(" item type(s):             ", itemtypes(x), "\n")
	if(ncov(x)>0) 
	cat(" nr of covariates:         ", ncov(x), "\n")
	cat(" item name(s):             ", inames(x), "\n")
	cat(" length(s) of series:      ", ntimes(x)[1:min(5,ind(x))])
	if(ind(x)>5) cat(" ... \n") else cat("\n")
	if(ind(x)>1) 
	cat(" nr of independent series: ", ind(x), "\n")
	cat(" data: \n")
	print(x[,], ...)
}

plot.md <- function(x, nitems=1:(min(5,dim(x)[2])),nind=1:(min(5,length(attributes(x)$ntimes))), ...) {
	dat=x[,nitems,drop=FALSE]
	ind<-length(attributes(x)$ntimes)
	if(ind==1) plot.ts2(dat,plot.type="multiple",type="l",main=attributes(x)$dname, ...)
 	else {
		layout.show()
		# make dat into (multiple) timeseries objects
		ntimes=attributes(x)$ntimes
		dat2 <- list()
		for (i in nind) {
			if(i==1) bg=1 else bg=sum(ntimes[1:(i-1)])+1
			en=(sum(ntimes[1:i]))
			dat2[[i]] <- ts(dat[bg:en,,drop=FALSE])
		}
		y=dat2[[nind[1]]]
 		if(length(nind)>1) {
 			for(i in 2:length(nind)) y=cbind(y,dat2[[nind[i]]])
 		}
		colnames(y)=rep(colnames(dat),length(nind))
		plot.ts2(y,plot.type="multiple",type="l",main=attributes(x)$dname, ...)
 	}
}


# the only difference with plot.ts is that this is not limited to 10 series to
# plot, in fact it is unlimited now, which may not be a good idea ... 

plot.ts2 <-
	function (x, y = NULL, plot.type = c("multiple", "single"),
		  xy.labels, xy.lines, panel = lines, nc, yax.flip = FALSE,
		  mar.multi = c(0, 5.1, 0, if(yax.flip) 5.1 else 2.1),
		  oma.multi = c(6, 0, 5, 0), axes = TRUE, ...)
{
	plotts <-
	function (x, y = NULL, plot.type = c("multiple", "single"),
		  xy.labels, xy.lines, panel = lines, nc,
		  xlabel, ylabel, type = "l", xlim = NULL, ylim = NULL,
		  xlab = "Time", ylab, log = "", col = par("col"), bg = NA,
		  pch = par("pch"), cex = par("cex"),
		  lty = par("lty"), lwd = par("lwd"),
		  axes = TRUE, frame.plot = axes, ann = par("ann"),
		  main = NULL, ...)
	{
	plot.type <- match.arg(plot.type)
	nser <- NCOL(x)

	if(plot.type == "multiple" && nser > 1) {
		addmain <- function(main, cex.main=par("cex.main"),
				font.main=par("font.main"),
				col.main=par("col.main"), ...)
		## pass 'cex.main' etc	via "..." from main function
		mtext(main, side=3, line=3,
			  cex=cex.main, font=font.main, col=col.main, ...)
		panel <- match.fun(panel)
		nser <- NCOL(x)
#  		if(nser > 30) stop("Can't plot more than 10 series as \"multiple\"")
		if(is.null(main)) main <- xlabel
		nm <- colnames(x)
		if(is.null(nm)) nm <- paste("Series", 1:nser)
		if(missing(nc)) nc <- if(nser > 4) 2 else 1
		nr <- ceiling(nser/nc)

		oldpar <- par(mar = mar.multi, oma = oma.multi, mfcol = c(nr, nc))
		on.exit(par(oldpar))
		for(i in 1:nser) {
		plot.default(x[, i], axes = FALSE, xlab="", ylab="",
			 log = log, col = col, bg = bg, pch = pch, ann = ann,
			 type = "n", ...)
		panel(x[, i], col = col, bg = bg, pch = pch, type=type, ...)
		if(frame.plot) box(...)
		y.side <- if (i %% 2 || !yax.flip) 2 else 4
		do.xax <- i %% nr == 0 || i == nser
		if(axes) {
			axis(y.side, xpd = NA)
			if(do.xax)
			axis(1, xpd = NA)
		}
		if(ann) {
			mtext(nm[i], y.side, line=3, ...)
			if(do.xax)
			mtext(xlab, side=1, line=3, ...)
		}
		}
		if(ann && !is.null(main)) {
		par(mfcol=c(1,1))
		addmain(main, ...)
		}
		return(invisible())
	}
	## end of multiple plot section

	x <- as.ts(x)
	if(!is.null(y)) {
		## want ("scatter") plot of y ~ x
		y <- hasTsp(y)
		if(NCOL(x) > 1 || NCOL(y) > 1)
		stop("scatter plots only for univariate time series")
		if (is.ts(x) && is.ts(y)) {
		xy <- ts.intersect(x, y)
		xy <- xy.coords(xy[,1], xy[,2], xlabel, ylabel, log)
		} else
		xy <- xy.coords(x, y, xlabel, ylabel, log)
		xlab <- if (missing(xlab)) xy$xlab else xlab
		ylab <- if (missing(ylab)) xy$ylab else ylab
		xlim <- if (is.null(xlim)) range(xy$x[is.finite(xy$x)]) else xlim
		ylim <- if (is.null(ylim)) range(xy$y[is.finite(xy$y)]) else ylim
		n <- length(xy $ x)		  #-> default for xy.l(ines|abels)
		if(missing(xy.labels)) xy.labels <- (n <= 150)
		if(!is.logical(xy.labels)) {
		if(!is.character(xy.labels))
			stop("`xy.labels' must be logical or character")
		do.lab <- TRUE
		} else do.lab <- xy.labels

		ptype <-
		if(do.lab) "n" else if(missing(type)) "p" else type
		plot.default(xy, type = ptype,
			 xlab = xlab, ylab = ylab,
			 xlim = xlim, ylim = ylim, log = log, col = col, bg = bg,
			 pch = pch, axes = axes, frame.plot = frame.plot,
			 ann = ann, main = main, ...)
		if(missing(xy.lines)) xy.lines <- do.lab
		if(do.lab)
		text(xy, labels =
			 if(is.character(xy.labels)) xy.labels
			 else if(all(tsp(x) == tsp(y))) formatC(time(x), wid = 1)
			 else seq(along = x),
			 col = col, cex = cex)
		if(xy.lines)
		lines(xy, col = col, lty = lty, lwd = lwd,
			  type = if(do.lab) "c" else "l")
		return(invisible())
	}
	## Else : no y, only x

	if(missing(ylab)) {
		ylab <- colnames(x)
		if(length(ylab) != 1)
		ylab <- xlabel
	}
	## using xy.coords() mainly for the log treatment
	if(is.matrix(x)) {
		k <- ncol(x)
		tx <- time(x)
		xy <- xy.coords(x = matrix(rep.int(tx, k), ncol = k),
				y = x, log=log)
		xy$x <- tx
	}
	else xy <- xy.coords(x, NULL, log=log)
	if(is.null(xlim)) xlim <- range(xy$x)
	if(is.null(ylim)) ylim <- range(xy$y[is.finite(xy$y)])
	plot.new()
	plot.window(xlim, ylim, log, ...)
	if(is.matrix(x)) {
		for(i in seq(length=k))
		lines.default(xy$x, x[,i],
				  col = col[(i-1) %% length(col) + 1],
				  lty = lty[(i-1) %% length(lty) + 1],
				  lwd = lwd[(i-1) %% length(lwd) + 1],
				  bg  = bg [(i-1) %% length(bg) + 1],
				  pch = pch[(i-1) %% length(pch) + 1],
				  type = type)
	}
	else {
		lines.default(xy$x, x, col = col[1], bg = bg, lty = lty[1],
			  lwd = lwd[1], pch = pch[1], type = type)
	}
	if (ann)
		title(main = main, xlab = xlab, ylab = ylab, ...)
	if (axes) {
		axis(1, ...)
		axis(2, ...)
	}
	if (frame.plot) box(...)
	}
	xlabel <- if (!missing(x)) deparse(substitute(x))# else NULL
	ylabel <- if (!missing(y)) deparse(substitute(y))
	plotts(x = x, y = y, plot.type = plot.type,
	   xy.labels = xy.labels, xy.lines = xy.lines,
	   panel = panel, nc = nc, xlabel = xlabel, ylabel = ylabel,
		   axes = axes, ...)
}

