% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/drr.R
\docType{class}
\name{DRR-class}
\alias{DRR}
\alias{DRR-class}
\title{Dimensionality Reduction via Regression}
\description{
An S4 Class implementing Dimensionality Reduction via Regression (DRR).
}
\details{
DRR is a non-linear extension of PCA that uses Kernel Ridge regression.
}
\section{Slots}{

\describe{
\item{\code{fun}}{A function that does the embedding and returns a
dimRedResult object.}

\item{\code{stdpars}}{The standard parameters for the funciton.}
}}
\section{General usage}{

Dimensionality reduction methods are S4 Classes that either be used
directly, in which case they have to be initialized and a full
list with parameters has to be handed to the \code{@fun()}
slot, or the method name be passed to the embed function and
parameters can be given to the \code{...}, in which case
missing parameters will be replaced by the ones in the
\code{@stdpars}.
}

\section{Parameters}{

DRR can take the following parameters:
\describe{
  \item{ndim}{The number of dimensions}
  \item{lambda}{The regularization parameter for the ridge
  regression.}
  \item{kernel}{The kernel to use for KRR, defaults to
  \code{"rbfdot"}.}
  \item{kernel.pars}{A list with kernel parameters, elements depend
  on the kernel used, \code{"rbfdot"} uses \code{"sigma"}.}
  \item{pca}{logical, should an initial pca step be performed,
  defaults to \code{TRUE}.}
  \item{pca.center}{logical, should the data be centered before the
  pca step. Defaults to \code{TRUE}.}
  \item{pca.scale}{logical, should the data be scaled before the
  pca ste. Defaults to \code{FALSE}.}
  \item{fastcv}{logical, should \code{\link[CVST]{fastCV}} from the
  CVST package be used instead of normal cross-validation.}
  \item{fastcv.test}{If \code{fastcv = TRUE}, separate test data set for fastcv.}
  \item{cv.folds}{if \code{fastcv = FALSE}, specifies the number of
  folds for crossvalidation.}
  \item{fastkrr.nblocks}{integer, higher values sacrifice numerical
  accuracy for speed and less memory, see below for details.}
  \item{verbose}{logical, should the cross-validation results be
  printed out.}
}
}

\section{Implementation}{

Wraps around \code{\link[DRR]{drr}}, see there for details. DRR is
a non-linear extension of principal components analysis using Kernel
Ridge Regression (KRR, details see \code{\link[CVST]{constructKRRLearner}}
and \code{\link[DRR]{constructFastKRRLearner}}). Non-linear
regression is used to explain more variance than PCA. DRR provides
an out-of-sample extension and a backward projection.

The most expensive computations are matrix inversions therefore the
implementation profits a lot from a multithreaded BLAS library.
The best parameters for each KRR are determined by cross-validaton
over all parameter combinations of \code{lambda} and
\code{kernel.pars}, using less parameter values will speed up
computation time. Calculation of KRR can be accelerated by
increasing \code{fastkrr.nblocks}, it should be smaller than
n^{1/3} up to sacrificing some accuracy, for details see
\code{\link[DRR]{constructFastKRRLearner}}. Another way to speed up
is to use \code{pars$fastcv = TRUE} which might provide a more
efficient way to search the parameter space but may also miss the
global maximum, I have not ran tests on the accuracy of this method.
}
\examples{
\dontrun{
dat <- loadDataSet("variable Noise Helix", n = 200)[sample(200)]

## use the S4 Class directly:
drr <- DRR()
pars <- drr@stdpars
pars$ndim <- 3
emb <- drr@fun(dat, pars)

## simpler, use embed():
emb2 <- embed(dat, "DRR", ndim = 3)


plot(dat, type = "3vars")
plot(emb, type = "3vars")
plot(emb@inverse(emb@data@data[, 1, drop = FALSE]), type = "3vars")
}


}
\references{
Laparra, V., Malo, J., Camps-Valls, G.,
    2015. Dimensionality Reduction via Regression in Hyperspectral
    Imagery. IEEE Journal of Selected Topics in Signal Processing
    9, 1026-1036. doi:10.1109/JSTSP.2015.2417833
}
\seealso{
Other dimensionality reduction methods: \code{\link{DiffusionMaps-class}},
  \code{\link{DrL-class}}, \code{\link{FastICA-class}},
  \code{\link{FruchtermanReingold-class}},
  \code{\link{HLLE-class}}, \code{\link{Isomap-class}},
  \code{\link{KamadaKawai-class}}, \code{\link{LLE-class}},
  \code{\link{MDS-class}}, \code{\link{PCA-class}},
  \code{\link{dimRedMethod-class}},
  \code{\link{kPCA-class}}, \code{\link{nMDS-class}},
  \code{\link{tSNE-class}}
}

